package com.instabug.library.util

import android.annotation.SuppressLint
import android.graphics.PorterDuff
import android.graphics.PorterDuffColorFilter
import android.graphics.Typeface
import android.graphics.drawable.Drawable
import android.os.Build
import android.view.View
import android.widget.Button
import android.widget.EditText
import android.widget.TextView
import com.instabug.library.BuildFieldsProvider
import com.instabug.library.model.IBGTheme
import com.instabug.library.settings.SettingsManager

object ThemeApplier {
    const val DEFAULT_COLOR = 0

    @JvmStatic
    fun TextView?.applyTitleStyle(theme: IBGTheme?) {
        theme?.let {
            applyStyle(it.primaryTextFont, it.titleTextColor, it.primaryTextStyle)
        }
    }


    @JvmStatic
    fun TextView?.applyPrimaryTextStyle(theme: IBGTheme?) {
        theme?.let {
            applyStyle(it.primaryTextFont, it.primaryTextColor, it.primaryTextStyle)
        }
    }

    @JvmStatic
    fun TextView?.applySecondaryTextStyle(theme: IBGTheme?) {
        theme?.let {
            applyStyle(it.secondaryTextFont, it.secondaryTextColor, it.secondaryTextStyle)
        }
    }

    @JvmStatic
    fun Button?.applyCtaStyle(theme: IBGTheme?) {
        this?.let { button ->
            val ctaColor = SettingsManager.getInstance().primaryColor
            if (ctaColor != -1) {
                button.setTextColor(ctaColor)
            }

            theme?.let { theme ->
                val ctaFont = theme.ctaTextFont

                if (ctaFont != null) {
                    button.setTypeface(ctaFont, theme.ctaTextStyle)
                } else {
                    button.setTypeface(button.typeface, theme.ctaTextStyle)
                }
            }
        }
    }

    @JvmStatic
    fun View?.setScreenBackground(theme: IBGTheme?) {
        this ?: return
        theme?.takeIf { it.backgroundColor != DEFAULT_COLOR }
            ?.backgroundColor
            ?.let(::setBackgroundColor)
    }

    @JvmStatic
    fun EditText?.applyEditTextStyle(theme: IBGTheme?) {
        this ?: return
        theme?.let {
            setTypeface(it.primaryTextFont ?: typeface, it.primaryTextStyle)
            if (it.secondaryTextColor != DEFAULT_COLOR) {
                setHintTextColor(it.secondaryTextColor)
                background.colorFilter =
                    PorterDuffColorFilter(it.secondaryTextColor, PorterDuff.Mode.SRC_IN)
            }
            if (it.primaryTextColor != DEFAULT_COLOR) setTextColor(it.primaryTextColor)
        }
    }

    @SuppressLint("NewApi")
    @JvmStatic
    fun setViewBackground(view: View?, drawable: Drawable?) {
        view?.takeIf { drawable != null }?.run {
            if (BuildFieldsProvider.provideBuildVersion() >= Build.VERSION_CODES.JELLY_BEAN) {
                background = drawable
            } else {
                setBackgroundDrawable(drawable)
            }
        }
    }

    private fun TextView?.applyStyle(font: Typeface?, color: Int, style: @IBGTheme.TextStyle Int) {
        this?.let { textView ->
            font?.let {
                textView.setTypeface(font, style)
            } ?: textView.setTypeface(typeface, style)

            if (color != DEFAULT_COLOR) {
                textView.setTextColor(color)
            }
        }
    }
}