package com.instabug.library.util;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.res.Resources;
import android.os.Build;
import android.util.DisplayMetrics;
import android.util.Pair;
import android.view.DisplayCutout;
import android.view.KeyCharacterMap;
import android.view.KeyEvent;
import android.view.ViewConfiguration;
import android.view.Window;
import android.view.WindowInsets;
import android.view.WindowManager;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.core.InstabugCore;

/**
 * Created by tarek on 2/25/18.
 */

public class ScreenUtility {

    public static int getScreenHeight(@Nullable Context context) {
        if (context != null) {
            DisplayMetrics metrics = context.getResources().getDisplayMetrics();
            return metrics.heightPixels;
        }
        return 0;
    }

    public static int getWindowHeight(Activity activity) {
        if (activity != null) {
            Window currentWindow = activity.getWindow();
            if (currentWindow != null)
                return currentWindow.getDecorView().getHeight();
        }
        return 0;
    }

    public static int getWindowWidth(Activity activity) {
        if (activity != null) {
            Window currentWindow = activity.getWindow();
            if (currentWindow != null)
                return currentWindow.getDecorView().getWidth();
        }
        return 0;
    }

    public static boolean isLandscape(Activity activity) {
        DisplayMetrics displayMetrics = new DisplayMetrics();
        activity.getWindowManager()
                .getDefaultDisplay()
                .getMetrics(displayMetrics);
        return displayMetrics.widthPixels > displayMetrics.heightPixels;
    }

    public static boolean hasNavBar(Context context) {
        if (context == null)
            return false;

        boolean hasMenuKey = ViewConfiguration.get(context).hasPermanentMenuKey();
        boolean hasBackKey = KeyCharacterMap.deviceHasKey(KeyEvent.KEYCODE_BACK);
        return (!(hasMenuKey && hasBackKey));
    }

    public static int getNavigationBarWidth(Resources resources) {
        @SuppressLint({"InternalInsetResource", "DiscouragedApi"})  int resourceId = resources.getIdentifier("navigation_bar_width", "dimen", "android");
        if (resourceId > 0) {
            return resources.getDimensionPixelSize(resourceId);
        }
        return 0;
    }




    public static int getNavigationBarHeight(Resources resources) {


       @SuppressLint({"InternalInsetResource", "DiscouragedApi"}) int resourceId = resources.getIdentifier("navigation_bar_height", "dimen", "android");
        if (resourceId > 0) {
            return resources.getDimensionPixelSize(resourceId);
        }
        return 0;
    }

    public static int getBottomInsets(Activity activity) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            Window currentWindow = activity.getWindow();
            if (currentWindow != null) {
                WindowInsets windowInsets = currentWindow.getDecorView().getRootWindowInsets();
                if (windowInsets != null && Build.VERSION.SDK_INT >= Build.VERSION_CODES.S_V2) {
                    return windowInsets.getInsets(WindowInsets.Type.systemBars()).bottom;
                } else if (windowInsets != null) {
                    return windowInsets.getStableInsetBottom();
                }
            }
        } else if (hasNavBar(activity) && InstabugCore.getTargetActivity()!= null) {
            return getNavigationBarHeight(activity.getResources());
        }
        return 0;
    }

    @Nullable
    public static Pair<Integer, Integer> getEdgesCutoutIfApplicable(@NonNull Activity activity) {
        if (android.os.Build.VERSION.SDK_INT < android.os.Build.VERSION_CODES.P) return null;
        int cutoutMode = activity.getWindow().getAttributes().layoutInDisplayCutoutMode;
        if (shouldSkipCutoutReduction(cutoutMode)) return null;
        return getEdgesCutoutCompat(activity);

    }

    private static boolean shouldSkipCutoutReduction(int cutoutMode) {
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.R)
            return cutoutMode == WindowManager.LayoutParams.LAYOUT_IN_DISPLAY_CUTOUT_MODE_ALWAYS ||
                    cutoutMode == WindowManager.LayoutParams.LAYOUT_IN_DISPLAY_CUTOUT_MODE_SHORT_EDGES;
        else
            return false;
    }

    @Nullable
    private static Pair<Integer, Integer> getEdgesCutoutCompat(@NonNull Activity activity) {
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.Q) {
            DisplayCutout cutout = activity.getWindowManager().getDefaultDisplay().getCutout();
            if (cutout == null) return null;
            return new Pair<>(cutout.getSafeInsetLeft(), cutout.getSafeInsetRight());
        }
        return null;
    }
}
