package com.instabug.library.sessionV3.cache

import androidx.annotation.WorkerThread
import com.instabug.library.core.plugin.PluginsManager
import com.instabug.library.internal.storage.cache.dbv2.IBGContentValues
import com.instabug.library.internal.storage.cache.dbv2.IBGCursor
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_DURATION
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_ID
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_RATING_DIALOG_DETECTION
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_SERIAL
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_SR_ENABLED
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_SR_EVALUATED
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_SYNC_STATUS
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_UUID
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.TABLE_NAME
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.TRIM_WHERE_CLAUSE
import com.instabug.library.internal.storage.cache.dbv2.IBGWhereArg
import com.instabug.library.internal.storage.cache.dbv2.WhereClause
import com.instabug.library.internal.storage.cache.dbv2.and
import com.instabug.library.internal.storage.cache.dbv2.args
import com.instabug.library.internal.storage.cache.dbv2.asArgs
import com.instabug.library.internal.storage.cache.dbv2.joinToArgs
import com.instabug.library.internal.storage.cache.dbv2.kDelete
import com.instabug.library.internal.storage.cache.dbv2.kQuery
import com.instabug.library.internal.storage.cache.dbv2.selection
import com.instabug.library.model.v3Session.IBGSession
import com.instabug.library.model.v3Session.IBGSessionMapper.toContentValues
import com.instabug.library.model.v3Session.IBGSessionMapper.toSession
import com.instabug.library.model.v3Session.IBGSessionMapper.toSessionList
import com.instabug.library.model.v3Session.IBGSessions
import com.instabug.library.model.v3Session.RatingDialogDetection.RATING_DIALOG_CACHING_ERROR_MESSAGE
import com.instabug.library.model.v3Session.RatingDialogDetection.RATING_DIALOG_QUERY_ERROR_MESSAGE
import com.instabug.library.model.v3Session.RatingDialogDetection.SOMETHING_WENT_WRONG_WHILE_QUERY_SR_EVALUATED
import com.instabug.library.model.v3Session.RatingDialogDetection.SR_DISABLE_SESSIONS_FAILED
import com.instabug.library.model.v3Session.SyncStatus
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator
import com.instabug.library.sessionV3.providers.SessionID
import com.instabug.library.sessionV3.ratingDialogDetection.RatingDialogData
import com.instabug.library.util.extenstions.asInt
import com.instabug.library.util.extenstions.getNullableString
import com.instabug.library.util.extenstions.getString
import com.instabug.library.util.extenstions.runOrLogError
import com.instabug.library.util.threading.PoolProvider

typealias SimpleSession = Pair<String, SyncStatus>

val SimpleSession.id
    get() = first

val SimpleSession.syncStatus
    get() = second

interface SessionCacheManager {
    fun insertOrUpdate(session: IBGSession): Long
    fun querySessions(status: SyncStatus? = null, limit: Int? = null): IBGSessions
    fun queryLastSession(): IBGSession?
    fun trimToLimit(limit: Int)
    fun migrateUUID(oldUUID: String, newUUID: String)
    fun changeSyncStatus(from: SyncStatus, to: SyncStatus, sessionsIds: List<String>? = null)
    fun deleteSessionByID(ids: List<String>)
    fun deleteAllSessions()
    fun querySessionsIdsBySyncStatus(vararg statuses: SyncStatus): List<SimpleSession>
    fun updateSessionDuration(sessionId: String, duration: Long)
    fun updateRatingDialogDetection(ratingDialogData: RatingDialogData?, sessionId: String?)
    fun disableSessionsSR(sessionId: String? = null)
    fun querySessionsStoreRatingData(): List<String>
    fun querySessionsBySrEvaluated(isSrEvaluated: Boolean): IBGSessions
    fun updateSrEvaluated(sessionId: String, isSrEvaluated: Boolean)
}

object SessionCacheManagerImpl : SessionCacheManager {
    private val database get() = IBGSessionServiceLocator.database
    private val configurations by lazy { IBGSessionServiceLocator.sessionConfigurations }
    private val featureSessionDataControllers
        get() = PluginsManager.getFeaturesSessionDataControllers()

    override fun insertOrUpdate(session: IBGSession): Long = session
        .takeIf { it.serial == -1L }
        ?.also { changeSyncStatus(SyncStatus.RUNNING, SyncStatus.OFFLINE) }
        ?.let { insert(session) }
        ?.also { trimToLimit(configurations.sessionsStoreLimit) }
        ?: update(session)


    private fun insert(session: IBGSession): Long = database
        .runOrLogError(errorMessage = "Something went wrong while inserting the new session ") {
            insert(TABLE_NAME, null, session.toContentValues())
        }.getOrNull() ?: -1

    private fun update(session: IBGSession): Long = database
        .runOrLogError(errorMessage = "Something went wrong while updating the new session ") {
            val whereClause = " $COLUMN_ID = ? AND $COLUMN_SERIAL = ? "
            val whereArgs = listOf(
                IBGWhereArg(session.id, true),
                IBGWhereArg(session.serial.toString(), true)
            )
            update(TABLE_NAME, session.toContentValues(), whereClause, whereArgs)
        }.getOrNull()
        .let { session.serial }

    override fun queryLastSession(): IBGSession? = database
        .runOrLogError(errorMessage = "Something went wrong while getting the Last session") {
            kQuery(table = TABLE_NAME, limit = "1", orderBy = "$COLUMN_SERIAL DESC")
        }.getOrNull()
        ?.toSession()

    override fun querySessions(status: SyncStatus?, limit: Int?): IBGSessions {
        return database.runOrLogError(errorMessage = "Something went wrong while query sessions") {
            val whereSyncStatusIs = status?.let { whereSyncStatusIs(status) }
            kQuery(
                table = TABLE_NAME,
                limit = limit?.toString(),
                whereClause = whereSyncStatusIs
            )

        }.getOrNull()
            ?.toSessionList()
            .orEmpty()
    }

    override fun trimToLimit(limit: Int) {
        database
            .runOrLogError("Something went wrong while trimming sessions ") {
                kQuery(
                    table = TABLE_NAME,
                    columns = arrayOf(COLUMN_ID),
                    whereClause = TRIM_WHERE_CLAUSE to
                            listOf(
                                IBGWhereArg("-1", true),
                                IBGWhereArg(limit.toString(), true)
                            )
                )?.use { cursor ->
                    mutableListOf<String>().apply {
                        while (cursor.moveToNext()) {
                            add(cursor.getString(COLUMN_ID))
                        }
                    }
                }
            }
            .getOrNull()
            .takeUnless { it.isNullOrEmpty() }
            ?.also(::deleteFeaturesSessionsData)
            ?.also(::deleteSessionByID)
            ?.size
            ?.let(configurations::incrementDroppedSessionCount)
    }

    private fun deleteFeaturesSessionsData(sessionsIds: MutableList<String>) {
        featureSessionDataControllers
            .map { controller -> PoolProvider.submitIOTask { controller.dropSessionData(sessionsIds) } }
            .forEach { future ->
                runOrLogError(errorMessage = "Something went wrong while deleting Features Sessions Data") {
                    future.get()
                }
            }
    }

    override fun migrateUUID(oldUUID: String, newUUID: String) {
        database.runOrLogError(errorMessage = "Something went wrong while migrate old uuid to the new uuid") {
            val values = IBGContentValues()
                .apply { put(COLUMN_UUID, newUUID, true) }
            val whereClause = "$COLUMN_UUID = ?"
            val whereArgs = listOf(IBGWhereArg(oldUUID, true))
            update(TABLE_NAME, values, whereClause, whereArgs)
        }
    }

    override fun changeSyncStatus(from: SyncStatus, to: SyncStatus, sessionsIds: List<String>?) {
        runOrLogError(errorMessage = "Something wen wrong while changing sync status from ${from.name} to ${to.name}") {
            val contentValues = IBGContentValues().apply {
                put(COLUMN_SYNC_STATUS, to.name, true)
            }
            val idsWhereClause = sessionsIds?.let { whereIdIn(sessionsIds) }
            database.update(
                TABLE_NAME, contentValues,
                "$COLUMN_SYNC_STATUS = ?" and idsWhereClause?.selection,
                listOf(IBGWhereArg(from.name, true)) + idsWhereClause?.args.orEmpty()
            )

        }
    }


    override fun deleteSessionByID(ids: List<String>) {
        database
            .runOrLogError(errorMessage = "Something went wrong while deleting session by id") {
                val whereColumnIdInIds = whereIdIn(ids)
                kDelete(
                    table = TABLE_NAME,
                    selection = whereColumnIdInIds.selection,
                    selectionArgs = whereColumnIdInIds.args
                )
            }
    }


    override fun deleteAllSessions() {
        database
            .runOrLogError(errorMessage = "Something went wrong while deleting all sessions") {
                kDelete(table = TABLE_NAME)
            }

    }

    override fun querySessionsIdsBySyncStatus(vararg statuses: SyncStatus): List<SimpleSession> =
        database.runOrLogError(errorMessage = "Something went wrong while getting simple sessions by status") {
            val whereSyncStatusIs = whereSyncStatusIs(*statuses)
            kQuery(
                table = TABLE_NAME,
                columns = arrayOf(COLUMN_ID, COLUMN_SYNC_STATUS),
                whereClause = whereSyncStatusIs
            )?.toSimpleSessions()
        }.getOrNull().orEmpty()

    override fun updateSessionDuration(sessionId: String, duration: Long) {
        database.runOrLogError(errorMessage = "Something went wrong while updating session $sessionId duration") {
            val values = IBGContentValues()
                .apply {
                    put(COLUMN_DURATION, duration, false)
                }
            update(
                TABLE_NAME,
                values,
                "$COLUMN_ID = ?",
                listOf(IBGWhereArg(sessionId, true))
            )
        }
    }

    private fun IBGCursor.toSimpleSessions() = use { cursor ->
        buildList {
            while (moveToNext())
                SimpleSession(
                    getString(COLUMN_ID),
                    SyncStatus.valueOf(getString(COLUMN_SYNC_STATUS))
                ).let { add(it) }
        }
    }

    private fun whereSyncStatusIs(vararg statuses: SyncStatus) = statuses.toList()
        .map { status -> status.name }
        .let { statusesList -> "$COLUMN_SYNC_STATUS IN ${statusesList.joinToArgs()}" to statusesList.asArgs() }

    private fun whereIdIn(ids: List<String>): WhereClause =
        "$COLUMN_ID IN ${ids.joinToArgs()}" to ids.asArgs()

    override fun updateRatingDialogDetection(
        ratingDialogData: RatingDialogData?,
        sessionId: String?
    ) {
        if (sessionId.isNullOrBlank() || ratingDialogData == null) return
        database.runOrLogError(errorMessage = RATING_DIALOG_CACHING_ERROR_MESSAGE) {
            val values = IBGContentValues().apply {
                put(
                    COLUMN_RATING_DIALOG_DETECTION,
                    IBGSessionServiceLocator.ratingDetectionMapper.map(ratingDialogData),
                    false
                )
            }
            update(
                TABLE_NAME,
                values,
                "$COLUMN_ID = ?",
                listOf(IBGWhereArg(sessionId, true))
            )
        }
    }

    override fun querySessionsStoreRatingData(): List<String> =
        database.runOrLogError(errorMessage = RATING_DIALOG_QUERY_ERROR_MESSAGE) {
            database.kQuery(
                TABLE_NAME,
                arrayOf(COLUMN_RATING_DIALOG_DETECTION),
                whereClause = "$COLUMN_RATING_DIALOG_DETECTION IS NOT NULL" to emptyList()
            )?.use(::toRatingData)
        }.getOrNull().orEmpty()

    override fun querySessionsBySrEvaluated(isSrEvaluated: Boolean): IBGSessions =
        database.runOrLogError(errorMessage = SOMETHING_WENT_WRONG_WHILE_QUERY_SR_EVALUATED) {
            kQuery(
                table = TABLE_NAME,
                whereClause = whereSessionReplayEvaluationIs(isSrEvaluated)
            )
        }.getOrNull()?.toSessionList().orEmpty()

    override fun updateSrEvaluated(sessionID: SessionID, isSrEvaluated: Boolean) {
        database.runOrLogError(errorMessage = "Something Went Wrong while updating sr_evaluated") {
            IBGContentValues()
                .apply { put(COLUMN_SR_EVALUATED, isSrEvaluated.asInt, true) }
                .let { values ->
                    update(
                        TABLE_NAME,
                        values,
                        "$COLUMN_ID = ?",
                        listOf(IBGWhereArg(sessionID, true))
                    )
                }
        }
    }

    private fun whereSessionReplayEvaluationIs(isSrEvaluated: Boolean) =
        "$COLUMN_SR_EVALUATED = ?" to listOf(
            IBGWhereArg(
                isSrEvaluated.asInt.toString(),
                true
            )
        )

    private fun toRatingData(cursor: IBGCursor) = mutableListOf<String>().apply {
        while (cursor.moveToNext()) {
            cursor
                .getNullableString(COLUMN_RATING_DIALOG_DETECTION)
                ?.let(::add)
        }
    }


    @WorkerThread
    override fun disableSessionsSR(sessionId: String?) {
        database.runOrLogError(errorMessage = SR_DISABLE_SESSIONS_FAILED) {
            val values = IBGContentValues().apply {
                put(COLUMN_SR_ENABLED, false.asInt, true)
            }
            update(
                TABLE_NAME,
                values,
                sessionId?.let { "$COLUMN_ID = ?" },
                sessionId?.let { listOf(IBGWhereArg(sessionId, true)) }
            )
        }
    }
}