package com.instabug.library.screenshot;

import static android.view.ViewGroup.LayoutParams.WRAP_CONTENT;
import static com.instabug.library.screenshot.instacapture.ScreenShotType.EXTRA;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.graphics.Bitmap;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.widget.FrameLayout;
import android.widget.ImageButton;
import android.widget.ImageView;

import androidx.annotation.MainThread;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.appcompat.content.res.AppCompatResources;
import androidx.core.content.ContextCompat;
import androidx.core.view.ViewCompat;
import androidx.fragment.app.DialogFragment;

import com.google.android.material.bottomsheet.BottomSheetDialogFragment;
import com.instabug.library.Constants;
import com.instabug.library.PresentationManager;
import com.instabug.library.R;
import com.instabug.library.core.CurrentFragmentLifeCycleEventBus;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.ActivityLifecycleSubscriber;
import com.instabug.library.core.eventbus.DefaultActivityLifeCycleEventHandler;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.invocation.InvocationManagerContract;
import com.instabug.library.screenshot.instacapture.ScreenshotRequest;
import com.instabug.library.screenshot.instacapture.ScreenshotRequestArgs;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.tracking.FragmentLifeCycleEvent;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.util.BitmapUtils;
import com.instabug.library.util.Colorizer;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.LocaleUtils;
import com.instabug.library.util.ScreenUtility;
import com.instabug.library.view.ViewUtils;

import java.lang.ref.WeakReference;

import io.reactivexport.disposables.Disposable;

public class ExtraScreenshotHelper implements DefaultActivityLifeCycleEventHandler {

    @Nullable
    @VisibleForTesting
    public WeakReference<ImageButton> captureBtnWeakReference;
    @VisibleForTesting
    public boolean isCaptureBtnShown = false;
    // RxBus Disposables
    @Nullable
    @VisibleForTesting
    public ActivityLifecycleSubscriber currentActivityLifeCycleSubscriber;

    @Nullable
    @VisibleForTesting
    public Disposable fragmentsLifeCycleSubscriber;

    // runnable to be run on captureButton click
    @Nullable
    @VisibleForTesting
    public OnCaptureListener onCaptureListener;

    @SuppressLint("NULL_DEREFERENCE")
    public void init(OnCaptureListener onCaptureListener) {
        this.onCaptureListener = onCaptureListener;
        subscribeToCurranActivityLifeCycle();
        subscribeToFragmentsLifecycle();
        InvocationManagerContract contract = CoreServiceLocator.getInvocationManagerContract();
        if (contract != null) {
            contract.switchOffInvocation();
        }
        PresentationManager.getInstance().setInInstabugContext(true);
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public void release() {
        if (captureBtnWeakReference != null) {
            View captureButton = captureBtnWeakReference.get();
            if (captureButton != null)
                captureButton.setOnClickListener(null);
        }
        onCaptureListener = null;
        captureBtnWeakReference = null;

        if (currentActivityLifeCycleSubscriber != null) {
            currentActivityLifeCycleSubscriber.unsubscribe();
            currentActivityLifeCycleSubscriber = null;
        }
        if (fragmentsLifeCycleSubscriber != null) {
            fragmentsLifeCycleSubscriber.dispose();
            fragmentsLifeCycleSubscriber = null;
        }
        PresentationManager.getInstance().setInInstabugContext(false);
        InvocationManagerContract contract = CoreServiceLocator.getInvocationManagerContract();
        if (contract != null) {
            contract.switchOnInvocation();
        }
    }

    @MainThread
    private void show(@NonNull final Activity hostActivity) {
        show(hostActivity, null);
    }

    @MainThread
    @VisibleForTesting
    public void show(@NonNull final Activity hostActivity, @Nullable final Window window) {
        if (isCaptureBtnShown || SettingsManager.getInstance().isProcessingForeground()) {
            return;
        }
        ImageButton captureButton = createCaptureButton(hostActivity);
        ViewCompat.setElevation(captureButton, ViewUtils.convertDpToPx(hostActivity.getApplicationContext(), 5));
        Window currentWindow = window;
        //Respect the passed window object first and add the button to it's decor view.
        if (currentWindow == null)
            currentWindow = hostActivity.getWindow();

        ViewGroup windowDecorView = (ViewGroup) currentWindow.getDecorView();
        windowDecorView.addView(captureButton, windowDecorView.getChildCount(), createCaptureButtonContainer(hostActivity));

        isCaptureBtnShown = true;
        captureButton.setOnClickListener(v -> {
            hide();
            captureExtraScreenshot(hostActivity, onCaptureListener);
        });
        captureBtnWeakReference = new WeakReference<>(captureButton);
    }

    @VisibleForTesting
    public ImageButton createCaptureButton(Activity hostActivity) {
        ImageButton captureButton = new ImageButton(hostActivity);
        captureButton.setId(R.id.instabug_extra_screenshot_button);
        captureButton.setScaleType(ImageView.ScaleType.CENTER);
        captureButton.setContentDescription(
                LocaleUtils.getLocaleStringResource(InstabugCore.getLocale(captureButton.getContext()),
                        R.string.ibg_extra_screenshot_button_content_description, captureButton.getContext()));
        Drawable backgroundDrawable = ContextCompat.getDrawable(hostActivity, R.drawable.ibg_core_bg_white_oval);
        Drawable imageDrawable = AppCompatResources.getDrawable(hostActivity, R.drawable.ibg_core_ic_screenshot);
        if (backgroundDrawable != null) {
            captureButton.setBackgroundDrawable(Colorizer.getPrimaryColorTintedDrawable(backgroundDrawable));
        }
        if (imageDrawable != null) {
            captureButton.setImageDrawable(imageDrawable);
        }
        return captureButton;
    }

    private FrameLayout.LayoutParams createCaptureButtonContainer(Activity hostActivity) {
        FrameLayout.LayoutParams frameLayoutParams = new FrameLayout.LayoutParams(
                WRAP_CONTENT, WRAP_CONTENT,
                Gravity.BOTTOM | Gravity.CENTER);
        frameLayoutParams.setMargins(0, 0, 0, 20);

        if (ScreenUtility.hasNavBar(hostActivity) &&
                android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP) {
            frameLayoutParams.bottomMargin += ScreenUtility.getBottomInsets(hostActivity);
        }
        return frameLayoutParams;
    }

    private void captureExtraScreenshot(
            final Activity hostActivity, @Nullable final OnCaptureListener onCaptureListener
    ) {
        ScreenshotProvider.INSTANCE.capture(ScreenshotRequest.createScreenshotRequest(
                new ScreenshotRequestArgs(EXTRA, hostActivity, createCapturingListener(onCaptureListener))
        ));
    }

    @NonNull
    private ScreenshotCaptor.CapturingCallback createCapturingListener(@Nullable OnCaptureListener onCaptureListener) {
        return new ScreenshotCaptor.CapturingCallback() {
            @Override
            public void onCapturingSuccess(@NonNull final Bitmap bitmap) {
                saveBitmap(bitmap, onCaptureListener);
            }

            @Override
            public void onCapturingFailure(@NonNull Throwable throwable) {
                if (onCaptureListener != null)
                    onCaptureListener.onExtraScreenshotError(throwable);
                runOnUiThread(ExtraScreenshotHelper.this::hide);

                InstabugSDKLogger.e(Constants.LOG_TAG,
                        "Error while capturing screenshot: " + throwable.getMessage());
            }
        };
    }

    private void saveBitmap(Bitmap bitmap, @Nullable OnCaptureListener onCaptureListener) {
        Activity currentActivity = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
        if (currentActivity != null)
            BitmapUtils.saveBitmap(bitmap, currentActivity, new
                    BitmapUtils.OnSaveBitmapCallback() {
                        @Override
                        public void onSuccess(Uri screenshotUri) {
                            InstabugSDKLogger.v(Constants.LOG_TAG,
                                    "Saving screenshot file to: " + screenshotUri.toString());
                            if (onCaptureListener != null)
                                onCaptureListener.onExtraScreenshotCaptured(screenshotUri);
                        }

                        @Override
                        public void onError(Throwable throwable) {
                            InstabugSDKLogger.e(Constants.LOG_TAG,
                                    "Error while saving screenshot: " + throwable.getMessage());
                        }
                    });
    }

    @VisibleForTesting
    public void reset(@Nullable Window currentWindow) {
        runOnUiThread(() -> {
                    Activity activity = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
                    if (activity == null) return;
                    hide();
                    show(activity, currentWindow);
                }
        );
    }

    @MainThread
    @VisibleForTesting
    public void hide() {
        if (captureBtnWeakReference == null || !isCaptureBtnShown) return;
        ImageButton captureBtn = captureBtnWeakReference.get();
        if (captureBtn == null) return;

        if (captureBtn.getParent() instanceof ViewGroup) {
            ViewGroup viewGroup = (ViewGroup) (captureBtn.getParent());
            captureBtn.setOnClickListener(null);
            viewGroup.removeView(captureBtn);
        } else {
            return;
        }

        isCaptureBtnShown = false;
        captureBtnWeakReference = null;
    }

    private void subscribeToCurranActivityLifeCycle() {
        if (currentActivityLifeCycleSubscriber == null) {
            currentActivityLifeCycleSubscriber = CoreServiceLocator.createActivityLifecycleSubscriber(this);
        }
        currentActivityLifeCycleSubscriber.subscribe();
    }

    private void subscribeToFragmentsLifecycle() {
        if (fragmentsLifeCycleSubscriber != null) return;

        fragmentsLifeCycleSubscriber = CurrentFragmentLifeCycleEventBus.getInstance().subscribe(event -> {
            if (event == FragmentLifeCycleEvent.RESUMED) handleFragmentResumed();
            else if (event == FragmentLifeCycleEvent.DETACHED) reset(null);
        });
    }

    @Override
    public void handleActivityResumed() {
        Activity hostActivity = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
        if (hostActivity == null) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Couldn't handle resume event current activity equal null");
            return;
        }
        runOnUiThread(() -> show(hostActivity));
    }

    @Override
    public void handleActivityPaused() {
        runOnUiThread(this::hide);
    }

    private void handleFragmentResumed() {
        Object lastSeenView = InstabugCore.getLastSeenView();
        //Limit the solution to bottom sheets only and ignore other dialog fragments right now.
        if (!(lastSeenView instanceof BottomSheetDialogFragment)) return;

        DialogFragment androidxDialogFragment = (DialogFragment) lastSeenView;
        if (androidxDialogFragment.getDialog() == null) return;
        if (!androidxDialogFragment.isResumed()) return;
        if (androidxDialogFragment.isRemoving()) return;
        if (androidxDialogFragment.isDetached()) return;

        final Window dialogWindow = androidxDialogFragment.getDialog().getWindow();
        runOnUiThread(() -> reset(dialogWindow));
    }

    public void cancel() {
        runOnUiThread(() -> {
            hide();
            release();
        });
    }

    private void runOnUiThread(Runnable runnable) {
        Activity currenActivity = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
        if (currenActivity == null) return;
        currenActivity.runOnUiThread(runnable);
    }

    public interface OnCaptureListener {
        void onExtraScreenshotCaptured(Uri extraScreenshotUri);

        void onExtraScreenshotError(Throwable throwable);
    }

}
