package com.instabug.library.screenshot;

import android.graphics.Bitmap;
import android.graphics.Rect;
import android.media.Image;
import android.media.ImageReader;

import androidx.annotation.IntRange;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.instabug.library.Constants;
import com.instabug.library.core.eventbus.ScreenCaptureEventBus;
import com.instabug.library.internal.utils.memory.IBGLowMemroyWarning;
import com.instabug.library.model.ScreenshotCaptureStatus;
import com.instabug.library.model.ScreenshotResponse;
import com.instabug.library.util.InstabugSDKLogger;

import java.nio.ByteBuffer;

@RequiresApi(21)
public class BitmapProviderRunnable implements Runnable {

    private final int width;
    private final int height;
    @Nullable
    private ImageReader reader;


    BitmapProviderRunnable(@IntRange(from = 1) int width, @IntRange(from = 1) int height, @Nullable ImageReader reader) {
        this.width = width;
        this.height = height;
        this.reader = reader;
    }

    @Override
    public void run() {
        if (width == 0 || height == 0 || reader == null) return;
        Bitmap bitmap = null;
        try {
            Image image = reader.acquireLatestImage();
            Image.Plane[] planes = image.getPlanes();
            ByteBuffer buffer = planes[0].getBuffer();
            int pixelStride = planes[0].getPixelStride(), rowStride = planes[0].getRowStride(), rowPadding = rowStride - pixelStride * width;
            bitmap = Bitmap.createBitmap(width + rowPadding / pixelStride, height, Bitmap.Config.ARGB_8888);
            bitmap.copyPixelsFromBuffer(buffer);
            Rect rect = image.getCropRect();
            bitmap = Bitmap.createBitmap(bitmap, rect.left, rect.top, rect.width(), rect.height());
            if (bitmap != null) {
                ScreenCaptureEventBus.getInstance().post(new ScreenshotResponse(ScreenshotCaptureStatus.SUCCESS, bitmap));
            } else {
                ScreenCaptureEventBus.getInstance().post(new ScreenshotResponse(ScreenshotCaptureStatus.FAILURE,
                        new Exception("Failed to capture screenshot using media projection ")));
            }
            //handle in case any out Of memory happened
        } catch (OutOfMemoryError outOfMemoryError) {
            ScreenCaptureEventBus.getInstance().post(new ScreenshotResponse(ScreenshotCaptureStatus.FAILURE,
                    new IBGLowMemroyWarning("Failed to capture screenshot using media projection due to low memory")));
        } catch (Exception e) {
            if (bitmap != null)
                bitmap.recycle();
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error " +
                    "occurred while processing the taken screenshot ", e);

        } finally {
            reader.close();
            reader = null;
        }
    }
}