package com.instabug.library.logging.disklogs;

import static com.instabug.library.logging.disklogs.LogFilesHelper.createTodayFile;
import static com.instabug.library.logging.disklogs.LogFilesHelper.deleteOldestFileFromDirectory;
import static com.instabug.library.logging.disklogs.LogFilesHelper.directoryReachedSizeLimit;
import static com.instabug.library.logging.disklogs.LogFilesHelper.fileReachedSingleFileSizeLimit;
import static com.instabug.library.logging.disklogs.LogFilesHelper.getTodayFileFromDirectory;
import static com.instabug.library.logging.disklogs.LogFilesHelper.isTodayFile;

import android.annotation.SuppressLint;
import android.content.Context;
import android.text.format.DateUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.Constants;
import com.instabug.library.internal.dataretention.core.Contract;
import com.instabug.library.internal.dataretention.files.FileDisposalPolicy;
import com.instabug.library.internal.dataretention.files.logs.LogFileDisposalPolicy;
import com.instabug.library.internal.dataretention.files.logs.TodayLogFileRule;
import com.instabug.library.internal.resolver.LoggingSettingResolver;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.model.LoggingSettings;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.TimeUtils;
import com.instabug.library.util.memory.MemoryUtils;

import java.io.File;
import java.io.IOException;
import java.lang.ref.WeakReference;

/**
 * A provider to provide today's log file
 * ***************
 * DEPENDENCIES
 * ***************
 * 1. Context
 * 2. {@link DiskUtils}
 * 3. {@link DateUtils}
 * 4. {@link TimeUtils}
 */
public class LoggingFileProvider {
    public static final String LOGS_DIRECTORY_NAME = "logs/";

    public static final String LOGS_EXTENSION = ".txt";
    private static final Object lock = new Object();
    @VisibleForTesting
    @Nullable
    File todayFile;
    @VisibleForTesting
    @Nullable
    File logsDirectory;
    private WeakReference<Context> context;

    LoggingFileProvider(Context context) {
        this.context = new WeakReference<>(context);
    }

    @Nullable
    public static FileDisposalPolicy getDisposalPolicy(Context context) {
        File logDirectory = DiskUtils.getInstabugLogDirectory(LOGS_DIRECTORY_NAME, context, false);
        if (logDirectory == null) return null;
        return new LogFileDisposalPolicy.Factory()
                .create(logDirectory.getAbsolutePath(), Contract.LOGS, new TodayLogFileRule());
    }

    public void init() {
        synchronized (lock) {
            prepareLogs();
        }
    }


    /**
     * A method to prepare logs,
     * it checks if the logs directory exists then {@link LoggingFileProvider#prepareTodayFile(File)}
     * if the directory doesn't exist then create using {@link DiskUtils#getInstabugLogDirectory(String, Context, Boolean)}
     * then {@link LoggingFileProvider#prepareTodayFile(File)}
     */
    @VisibleForTesting
    @SuppressLint("THREAD_SAFETY_VIOLATION")
    void prepareLogs() {
        try {
            if (this.context != null) {
                Context context = this.context.get();
                if (context != null) {
                    LoggingSettings settings = LoggingSettingResolver.getInstance().getLoggingSettings();
                    if (!MemoryUtils.isLowMemory(context)
                            && settings != null
                            && settings.getLevel() != LoggingSettings.LogLevels.NO_LOGS) {
                        logsDirectory = DiskUtils.getInstabugLogDirectory(LOGS_DIRECTORY_NAME, context, true);
                        if (logsDirectory == null) return;
                        todayFile = prepareTodayFile(logsDirectory);
                    }
                }
            }
        } catch (IOException ex) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while preparing disk logs", ex);
        }
    }

    /**
     * A method to prepare today file if file exists then return it otherwise,
     * create new one and return it
     *
     * @param logsDir
     * @return file if exist/created or null if directory not provided
     */
    @Nullable
    @VisibleForTesting
    private File prepareTodayFile(@NonNull File logsDir) throws IOException {

        if (directoryReachedSizeLimit(logsDir)) {
            deleteOldestFileFromDirectory(logsDir);
        }

        return getTodayFileFromDirectory(logsDir);
    }


    /**
     * A method to get today's file if the file, if the last fetched file is today's file then return it,
     * otherwise, create new one for today
     *
     * @return
     * @throws IOException
     */
    @Nullable
    synchronized File getTodayFile() throws IOException {
        if (todayFile != null) {

            File logsDir = getLogsDirectory();

            if (isTodayFile(todayFile)) {
                if (!fileReachedSingleFileSizeLimit(todayFile)) {
                    return todayFile;
                } else {
                    todayFile = createTodayFile(logsDir);
                }
            } else {
                if (logsDir != null) {
                    todayFile = prepareTodayFile(logsDir);
                }
            }
        } else {
            prepareLogs();
        }
        return todayFile;

    }


    synchronized void trimDirectoryIfNeeded() {
        File logsDir = getLogsDirectory();
        if (directoryReachedSizeLimit(logsDir)) {
            deleteOldestFileFromDirectory(logsDirectory);
        }
    }


    /**
     * A method to get direcotry folder if the last saved directory is null or isn't today's director then {@link LoggingFileProvider#prepareLogs()}
     * otherwise, return the saved directory.
     */
    @Nullable
    private synchronized File getLogsDirectory() {
        if (logsDirectory == null)
            prepareLogs();
        return logsDirectory;
    }
}
