package com.instabug.library.logging;

import static com.instabug.library.sessionreplay.model.SRLogKt.getMaxSize;

import android.annotation.SuppressLint;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.Constants;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.InstabugFeaturesManager;
import com.instabug.library.apichecker.APIChecker;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.datahub.DataHubLog;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.filestore.ActiveCurrentSpanSelector;
import com.instabug.library.internal.filestore.JSONArrayDataAggregator;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.sessionreplay.SRConstantsKt;
import com.instabug.library.sessionreplay.model.SRLog;
import com.instabug.library.sessionreplay.model.SRLogType;
import com.instabug.library.util.InstabugDateFormatter;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.concurrent.ExecutionException;

/**
 * @author mesbah
 */
public class InstabugLog {
    public static final int INSTABUG_LOG_LIMIT = 1000;

    /**
     * Appends a verbose log message to Instabug internal log
     * <p>
     * These logs are then sent along the next uploaded report. All log messages are timestamped <br/>
     * Logs aren't cleared per single application run. If you wish to reset the logs, use {@link #clearLogs()} ()}
     * </p>
     * Note: logs passed to this method are <b>NOT</b> printed to Logcat
     *
     * @param logMessage The message you would like logged
     * @since 3.1
     */
    public static void v(final String logMessage) {
        APIChecker.checkAndRunInExecutor("InstabugLog.v", () -> {
                    if (isInstabugLogsDisabled()) {
                        return;
                    }
                    addLog(new LogMessage().setLogMessage(logMessage)
                            .setLogMessageLevel(LogLevel.V)
                            .setLogMessageDate(getDate()));
                }
        );
    }

    /**
     * Appends a debug log message to Instabug internal log
     * <p>
     * These logs are then sent along the next uploaded report. All log messages are timestamped <br/>
     * Logs aren't cleared per single application run. If you wish to reset the logs, use {@link #clearLogs()} ()}
     * </p>
     * Note: logs passed to this method are <b>NOT</b> printed to Logcat
     *
     * @param logMessage The message you would like logged
     * @since 3.1
     */
    public static void d(final String logMessage) {
        APIChecker.checkAndRunInExecutor("InstabugLog.d", () -> {
            if (isInstabugLogsDisabled()) {
                return;
            }
            addLog(new LogMessage().setLogMessage(logMessage)
                    .setLogMessageLevel(LogLevel.D)
                    .setLogMessageDate(getDate()));
        });

    }

    /**
     * Appends a info log message to Instabug internal log
     * <p>
     * These logs are then sent along the next uploaded report. All log messages are timestamped <br/>
     * Logs aren't cleared per single application run. If you wish to reset the logs, use {@link #clearLogs()} ()}
     * </p>
     * Note: logs passed to this method are <b>NOT</b> printed to Logcat
     *
     * @param logMessage The message you would like logged
     * @since 3.1
     */
    public static void i(final String logMessage) {

        APIChecker.checkAndRunInExecutor("InstabugLog.i", () -> {
            if (isInstabugLogsDisabled()) {
                return;
            }
            addLog(new LogMessage().setLogMessage(logMessage)
                    .setLogMessageLevel(LogLevel.I)
                    .setLogMessageDate(getDate()));

        });
    }

    /**
     * Appends a error log message to Instabug internal log
     * <p>
     * These logs are then sent along the next uploaded report. All log messages are timestamped <br/>
     * Logs aren't cleared per single application run. If you wish to reset the logs, use {@link #clearLogs()} ()}
     * </p>
     * Note: logs passed to this method are <b>NOT</b> printed to Logcat
     *
     * @param logMessage The message you would like logged
     * @since 3.1
     */
    public static void e(final String logMessage) {
        APIChecker.checkAndRunInExecutor("InstabugLog.e", () -> {
            if (isInstabugLogsDisabled()) {
                return;
            }
            addLog(new LogMessage().setLogMessage(logMessage)
                    .setLogMessageLevel(LogLevel.E)
                    .setLogMessageDate(getDate()));

        });

    }

    /**
     * Appends a warning log message to Instabug internal log
     * <p>
     * These logs are then sent along the next uploaded report. All log messages are timestamped <br/>
     * Logs aren't cleared per single application run. If you wish to reset the logs, use {@link #clearLogs()} ()}
     * </p>
     * Note: logs passed to this method are <b>NOT</b> printed to Logcat
     *
     * @param logMessage The message you would like logged
     * @since 3.1
     */
    public static void w(final String logMessage) {
        APIChecker.checkAndRunInExecutor("InstabugLog.w", () -> {
            if (isInstabugLogsDisabled()) {
                return;
            }
            addLog(new LogMessage().setLogMessage(logMessage)
                    .setLogMessageLevel(LogLevel.W)
                    .setLogMessageDate(getDate()));
        });
    }

    /**
     * Appends a terrible failure log message to Instabug internal log
     * <p>
     * These logs are then sent along the next uploaded report. All log messages are timestamped <br/>
     * Logs aren't cleared per single application run. If you wish to reset the logs, use {@link #clearLogs()} ()}
     * </p>
     * Note: logs passed to this method are <b>NOT</b> printed to Logcat
     *
     * @param logMessage The message you would like logged
     * @since 3.1
     */
    public static void wtf(final String logMessage) {
        APIChecker.checkAndRunInExecutor("InstabugLog.wtf", () -> {
            if (isInstabugLogsDisabled()) {
                return;
            }
            addLog(new LogMessage().setLogMessage(logMessage)
                    .setLogMessageLevel(LogLevel.WTF)
                    .setLogMessageDate(getDate()));
        });
    }

    /**
     * Clears Instabug internal log
     *
     * @since 3.1
     */
    public static void clearLogs() {
        clearLogMessages();
    }

    /**
     * get Instabug internal log
     *
     * @return Instabug internal log
     * @since 4.0.3
     */
    public static String getLogs() {
        return getLogMessages();
    }

    private static synchronized void addLog(LogMessage logMessage) {
        CoreServiceLocator.getIbgLogsDistributor().invoke(logMessage);
    }

    private static void clearLogMessages() {
        CoreServiceLocator.getIbgLogStore().clear();
    }

    private static String getLogMessages() {
        String instabugLogs = "[]";
        try {
            JSONArray logMessages = CoreServiceLocator.getIbgLogStore()
                    .retrieve(new JSONArrayDataAggregator(), new ActiveCurrentSpanSelector<>()).get();
            if (logMessages != null)
                instabugLogs = logMessages.toString();
        } catch (OutOfMemoryError error) {
            InstabugCore.reportError(error, "Couldn't parse Instabug logs due to an OOM");
            InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't parse Instabug logs due to an OOM", error);
        } catch (InterruptedException exception) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while getting log messages", exception);
            Thread.currentThread().interrupt();
        } catch (ExecutionException exception) {
            InstabugCore.reportError(exception, "Error retrieving log messages from store");
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error retrieving log messages from store", exception);
        }
        return instabugLogs;
    }

    private static long getDate() {
        return InstabugDateFormatter.getCurrentUTCTimeStampInMiliSeconds();
    }

    private static boolean isInstabugLogsDisabled() {
        return InstabugFeaturesManager.getInstance().getFeatureState(IBGFeature.INSTABUG_LOGS)
                == Feature.State.DISABLED;
    }

    public enum LogLevel {

        V("v"), D("d"), I("i"), E("e"), W("w"), WTF("wtf");

        private final String level;

        LogLevel(String level) {
            this.level = level;
        }

        @NonNull
        @Override
        public String toString() {
            return level;
        }
    }

    public static class LogMessage implements SRLog, DataHubLog {
        // keys used by toJson() method
        static final String KEY_LOG_MESSAGE = "log_message";
        static final String KEY_LOG_MESSAGE_LEVEL = "log_message_level";
        static final String KEY_LOG_MESSAGE_DATE = "log_message_date";
        public static final String FAILED_TO_PARSE_INSTABUG_LOG_TO_JSON = "Failed to parse Instabug Log to JSON:";
        public static final String TRIMMING_SUSFIX = "...";
        public static final String NULL_LOG = "null";

        @Nullable
        private String logMessage;
        @Nullable
        private LogLevel logMessageLevel;
        private long logMessageDate;

        @Nullable
        String getLogMessage() {
            return logMessage;
        }

        LogMessage setLogMessage(String logMessage) {
            this.logMessage = validateSize(logMessage);
            return this;
        }

        @Nullable
        LogLevel getLogMessageLevel() {
            return logMessageLevel;
        }

        LogMessage setLogMessageLevel(LogLevel logMessageLevel) {
            this.logMessageLevel = logMessageLevel;
            return this;
        }

        long getLogMessageDate() {
            return logMessageDate;
        }

        @VisibleForTesting
        public LogMessage setLogMessageDate(long logMessageDate) {
            this.logMessageDate = logMessageDate;
            return this;
        }

        @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
        public JSONObject toJson() {
            JSONObject logMessageJsonObject = new JSONObject();
            try {
                logMessageJsonObject.put(KEY_LOG_MESSAGE, getLogMessage());
                if (getLogMessageLevel() != null) {
                    logMessageJsonObject.put(KEY_LOG_MESSAGE_LEVEL, getLogMessageLevel().toString());
                }
                logMessageJsonObject.put(KEY_LOG_MESSAGE_DATE, getLogMessageDate());
            } catch (JSONException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Error while parsing instabug logs", e);
            }
            return logMessageJsonObject;
        }

        @Override
        public long getTimestamp() {
            return getLogMessageDate();
        }

        @NonNull
        @Override
        public String getLogType() {
            return SRLogType.IBG_LOG;
        }

        @Nullable
        @Override
        public JSONObject getSrJsonRep() {
            try {
                JSONObject jsonObject = toJson();
                //fill session replay data
                jsonObject
                        .put(SRConstantsKt.LOG_TYPE_KEY, getLogType())
                        .put(SRConstantsKt.TIMESTAMP_KEY, getTimestamp());
                return jsonObject;
            } catch (JSONException e) {
                IBGDiagnostics.reportNonFatalAndLog(e, FAILED_TO_PARSE_INSTABUG_LOG_TO_JSON, Constants.LOG_TAG);
                return null;
            }
        }

        @Nullable
        @Override
        public JSONObject getDataHubRep() {
            return toJson();
        }

        @NonNull
        private String validateSize(@Nullable String logMessage) {
            if (logMessage == null) return NULL_LOG;

            int limit = getMaxSize(this);

            return logMessage.length() > limit
                    ? logMessage.substring(0, limit) + TRIMMING_SUSFIX
                    : logMessage;
        }

        public static LogMessage generateLogMessage(String message, LogLevel level) {
            return new LogMessage().setLogMessage(message)
                    .setLogMessageLevel(level)
                    .setLogMessageDate(getDate());
        }
    }
}
