package com.instabug.library.internal.utils

import android.annotation.SuppressLint
import android.content.Context
import android.content.SharedPreferences
import com.instabug.library.internal.servicelocator.CoreServiceLocator
import com.instabug.library.util.nullRetryLazy

class PreferencesUtils(private var context: Context, private var name: String) {
    private val preferences by nullRetryLazy {
        CoreServiceLocator.getInstabugSharedPreferences(context, name)
    }

    fun getSharedPreferences(): SharedPreferences? = preferences

    /**
     * this method used to save new boolean or update an existing one
     *
     * @param key   that will identifies the passed element
     * @param value that needs to be persisted
     */
    fun saveOrUpdateBoolean(key: String?, value: Boolean) {
        preferences?.edit()?.also { editor ->
            editor.putBoolean(key, value)
            editor.apply()
        }
    }

    /**
     * this method used to retrieve a boolean value
     *
     * @param key that identifies the element
     * @return the value or false if not found
     */
    fun getBoolean(key: String?): Boolean = getBoolean(key, false)

    /**
     * this method used to retrieve a boolean value
     *
     * @param key that identifies the element
     * @return the value or `defaultValue` if not found
     */
    fun getBoolean(key: String?, defaultValue: Boolean): Boolean =
        preferences?.getBoolean(key, defaultValue) ?: defaultValue

    /**
     * this method used to save new long or update an existing one
     *
     * @param key   that will identifies the passed element
     * @param value that needs to be persisted
     */
    fun saveOrUpdateLong(key: String?, value: Long) {
        preferences?.edit()?.also { editor ->
            editor.putLong(key, value)
            editor.apply()
        }
    }

    /**
     * this method used to retrieve a long value
     *
     * @param key that identifies the element
     * @return the value or 0 if not found
     */
    fun getLong(key: String?): Long = getLong(key, 0)

    /**
     * this method used to retrieve a long value
     *
     * @param key that identifies the element
     * @return the value or `defaultValue` if not found
     */
    fun getLong(key: String?, defaultValue: Long): Long =
        preferences?.getLong(key, defaultValue) ?: defaultValue

    /**
     * this method used to save new int or update an existing one
     *
     * @param key   that will identifies the passed element
     * @param value that needs to be persisted
     */
    fun saveOrUpdateInt(key: String?, value: Int) {
        preferences?.edit()?.also { editor ->
            editor.putInt(key, value)
            editor.apply()
        }
    }

    /**
     * this method used to retrieve aa int value
     *
     * @param key that identifies the element
     * @return the value or 0 if not found
     */
    fun getInt(key: String?): Int = getInt(key, 0)

    /**
     * this method used to retrieve aa int value
     *
     * @param key that identifies the element
     * @return the value or `defaultValue` if not found
     */
    fun getInt(key: String?, defaultValue: Int): Int =
        preferences?.getInt(key, defaultValue) ?: defaultValue

    /**
     * this method used to save new string or update an existing one
     *
     * @param key   that will identifies the passed element
     * @param value that needs to be persisted
     */
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    fun saveOrUpdateString(key: String?, value: String?) {
        preferences?.edit()?.also { editor ->
            editor.putString(key, value)
            editor.apply()
        }
    }

    /**
     * this method used to retrieve a string value
     *
     * @param key that identifies the element
     * @return the value or null if not found
     */
    fun getString(key: String?): String? = getString(key, null)

    /**
     * this method used to retrieve a string value
     *
     * @param key that identifies the element
     * @return the value or `defaultValue` if not found
     */
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    fun getString(key: String?, defaultValue: String?): String? =
        preferences?.getString(key, defaultValue) ?: defaultValue

    /**
     * this method used to delete specific element from the sharedPreferences
     *
     * @param key that identifies the element
     */
    fun delete(key: String?) {
        preferences?.edit()?.also { editor ->
            editor.remove(key)
            editor.apply()
        }
    }

    /**
     * this method used to delete all the elements from the sharedPreferences
     */
    fun deleteAll() {
        preferences?.edit()?.also { editor ->
            editor.clear()
            editor.apply()
        }
    }
}