package com.instabug.library.internal.storage.cache.db.userAttribute;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserAttributesEntry.COLUMN_IS_ANONYMOUS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserAttributesEntry.COLUMN_KEY;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserAttributesEntry.COLUMN_TYPE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserAttributesEntry.COLUMN_UUID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserAttributesEntry.COLUMN_VALUE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserAttributesEntry.TABLE_NAME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserAttributesEntry.TRIM_WHERE_CLAUSE;
import static com.instabug.library.util.filters.AttributeFiltersFunctions.isValidAttribute;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.model.UserAttribute;
import com.instabug.library.model.UserAttributes;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.filters.AttributeFiltersFunctions;
import com.instabug.library.util.filters.Filters;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author hossam.
 */
public class UserAttributesDbHelper {


    /**
     * Insert user attribute
     *
     * @param userAttribute .
     * @return the row ID of the newly inserted row, or -1 if an error occurred
     */
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static synchronized long insert(UserAttribute userAttribute) {
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        db.beginTransaction();
        try {
            // Create a new map of values, where column names are the keys
            ContentValues values = getContentValue(userAttribute);
            // Insert the new row, returning the primary key value of the new row
            long rowId = db.insertWithOnConflict(TABLE_NAME, null, values);
            if (rowId == -1) {
                update(userAttribute);
            }
            trimToLimit(CoreServiceLocator.getUserAttributesStoreLimit());
            db.setTransactionSuccessful();
            return rowId;
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    @UserAttribute.Type
    public static int getType(String key, String uuid) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String[] columns = {COLUMN_TYPE};
        String selection = COLUMN_KEY + " LIKE ? AND " + COLUMN_UUID + " =? ";
        String[] selectionArgs = new String[]{"%" + key + "%", uuid};
        Cursor cursor = null;
        int type = UserAttribute.Type.UNSPECIFIED;
        try {
            cursor = db.query(TABLE_NAME, columns, selection, selectionArgs, null, null, null);
            if (cursor != null && cursor.getCount() > 0) {
                cursor.moveToFirst();
                int countColumnIndex = cursor.getColumnIndex(COLUMN_TYPE);
                type = cursor.getInt(countColumnIndex);
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatalAndLog(ex, "Failed to get UserAttribute type due to: " + ex.getMessage(), Constants.LOG_TAG);
        } finally {
            if (cursor != null) {
                cursor.close();
            }
            db.close();
        }
        return type;
    }

    /**
     * Retrieve a userAttribute by key and uuid
     *
     * @param key  the userAttributeKey
     * @param uuid
     * @return a list of {@link com.instabug.library.model.UserAttributes}
     */
    @Nullable
    public static String retrieve(String key, String uuid) {

        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();

        final Cursor cursor = database.query(
                // Name of the table to read from
                TABLE_NAME,
                // String array of the columns which are supposed to be read
                new String[]{COLUMN_VALUE},
                // The selection argument which specifies which row is read. // ? symbols are
                // parameters.
                COLUMN_KEY + " LIKE ? AND " + COLUMN_UUID + " =? ",
                // The actual parameters values for the selection as a String array. // ? above
                // take the value
                // from here
                new String[]{"%" + key + "%", uuid},
                // GroupBy clause. Specify a column name to group similar values // in that
                // column together.
                null,
                // Having clause. When using the GroupBy clause this allows you to // specify
                // which groups to
                // include.
                null,
                // OrderBy clause. Specify a column name here to order the results
                // according to that column. Optionally append ASC or DESC to specify // an
                // ascending or
                // descending order.
                null);
        // To increase performance first get the index of each column in the cursor
        if (cursor != null) {
            final int valueIndex = cursor.getColumnIndex(COLUMN_VALUE);
            try {
                // If moveToFirst() returns false then cursor is empty
                if (!cursor.moveToFirst()) {
                    return null;
                }
                // Read the values of a row in the table using the indexes acquired above
                return cursor.getString(valueIndex);
            } finally {
                // Don't forget to close the Cursor once you are done to avoid memory leaks.
                // Using a try/finally like in this example is usually the best way to handle this
                // close the database
                cursor.close();
                database.close();
            }
        }
        return null;
    }

    /**
     * Retrieve all userAttributes related to uuid
     *
     * @return a HashMap of {@link com.instabug.library.model.UserAttributes}
     */
    @NonNull
    public static HashMap<String, String> retrieveAll(String uuid) {
        Cursor cursor = null;
        SQLiteDatabaseWrapper database = null;
        try {
            // When reading data one should always just get a readable database.
            database = DatabaseManager.getInstance().openDatabase();
            cursor = database.query(
                    // Name of the table to read from
                    TABLE_NAME,
                    // String array of the columns which are supposed to be read
                    new String[]{COLUMN_KEY, COLUMN_VALUE},
                    // The selection argument which specifies which row is read. // ? symbols are
                    // parameters.
                    COLUMN_UUID + " =? ",
                    // The actual parameters values for the selection as a String array. // ? above
                    // take the value
                    // from here
                    new String[]{uuid},
                    // GroupBy clause. Specify a column name to group similar values // in that
                    // column together.
                    null,
                    // Having clause. When using the GroupBy clause this allows you to // specify
                    // which groups to
                    // include.
                    null,
                    // OrderBy clause. Specify a column name here to order the results
                    // according to that column. Optionally append ASC or DESC to specify // an
                    // ascending or
                    // descending order.
                    null);
            // To increase performance first get the index of each column in the cursor
            if (cursor != null) {
                HashMap<String, String> userAttributes = new HashMap<>();
                final int keyIndex = cursor.getColumnIndex(COLUMN_KEY);
                final int valueIndex = cursor.getColumnIndex(COLUMN_VALUE);
                // If moveToFirst() returns false then cursor is empty
                if (!cursor.moveToFirst()) {
                    return userAttributes;
                }
                do {
                    // Read the values of a row in the table using the indexes acquired above
                    final String key = cursor.getString(keyIndex);
                    final String value = cursor.getString(valueIndex);
                    userAttributes.put(key, value);
                } while (cursor.moveToNext());
                // Read the values of a row in the table using the indexes acquired above
                return userAttributes;
            }
        } catch (Exception e) { // To catch CursorWindowAllocationException
            InstabugSDKLogger.e(Constants.LOG_TAG, "an exception has occurred while retrieving user attributes", e);
        } finally {
            // Don't forget to close the Cursor once you are done to avoid memory leaks.
            // Using a try/finally like in this example is usually the best way to handle this
            // close the database
            if (cursor != null) {
                cursor.close();
            }
            if (database != null) {
                database.close();
            }
        }
        return new HashMap<>();
    }

    public static HashMap<String, String> retrieveAllSDKAttributes(String uuid) {


        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();

        final Cursor cursor = database.query(
                // Name of the table to read from
                TABLE_NAME,
                // String array of the columns which are supposed to be read
                new String[]{COLUMN_KEY, COLUMN_VALUE},
                // The selection argument which specifies which row is read. // ? symbols are
                // parameters.
                COLUMN_UUID + " =? " + " AND " + COLUMN_TYPE + " = " + UserAttribute.Type.SDK,
                // The actual parameters values for the selection as a String array. // ? above
                // take the value
                // from here
                new String[]{uuid},
                // GroupBy clause. Specify a column name to group similar values // in that
                // column together.
                null,
                // Having clause. When using the GroupBy clause this allows you to // specify
                // which groups to
                // include.
                null,
                // OrderBy clause. Specify a column name here to order the results
                // according to that column. Optionally append ASC or DESC to specify // an
                // ascending or
                // descending order.
                null);
        // To increase performance first get the index of each column in the cursor
        if (cursor != null) {
            final int keyIndex = cursor.getColumnIndex(COLUMN_KEY);
            final int valueIndex = cursor.getColumnIndex(COLUMN_VALUE);
            try {
                // If moveToFirst() returns false then cursor is empty
                HashMap<String, String> userAttributes = new HashMap<>();
                if (!cursor.moveToFirst()) {
                    return userAttributes;
                }
                do {
                    // Read the values of a row in the table using the indexes acquired above
                    final String key = cursor.getString(keyIndex);
                    final String value = cursor.getString(valueIndex);
                    if (isValidAttribute(key, value))
                        userAttributes.put(key, value);
                } while (cursor.moveToNext());
                // Read the values of a row in the table using the indexes acquired above
                return userAttributes;
            } catch (Exception e) { // To catch CursorWindowAllocationException
                InstabugSDKLogger.e(Constants.LOG_TAG, "an exception has occurred while retrieving user attributes", e);
            } finally {
                // Don't forget to close the Cursor once you are done to avoid memory leaks.
                // Using a try/finally like in this example is usually the best way to handle this
                // close the database
                cursor.close();
                database.close();
            }
        }
        return new HashMap<>();
    }

    /**
     * Delete a specific userAttribute record
     *
     * @param key the key of the userAttribute
     */
    public static synchronized void delete(String key, String uuid) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_KEY + " = ? AND " + COLUMN_UUID + " =?";
        String[] whereArgs = new String[]{key, uuid};
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    public static synchronized void deleteType(String uuid, @UserAttribute.Type int type) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_TYPE + " = ? AND " + COLUMN_UUID + " =?";
        String[] whereArgs = new String[]{String.valueOf(type), uuid};
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * Delete a user attributes by user's uuid
     *
     * @param uuid
     */
    public static synchronized void deleteUserAttributes(String uuid) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_UUID + " = ?";
        String[] whereArgs = new String[]{uuid};
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }


    /**
     * Delete all userAttributes stored in the database
     * i.e. deleting the whole table
     */
    public static synchronized void deleteAll() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, null, null);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private static synchronized long update(UserAttribute userAttribute) {
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_KEY + " = ? AND " + COLUMN_UUID + "=?";
        String[] whereArgs = new String[]{userAttribute.getKey(), userAttribute.getUuid()};
        db.beginTransaction();
        try {
            // Create a new map of values, where column names are the keys
            ContentValues values = getContentValue(userAttribute);
            // Insert the new row, returning the primary key value of the new row
            long rowId = db.update(TABLE_NAME, values, whereClause, whereArgs);
            db.setTransactionSuccessful();
            return rowId;
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * @return All user attributes in cache.
     */
    @NonNull
    public static HashMap<String, String> getAll() {
        return UserAttributeCacheManager.retrieveAll();
    }

    public static String getSDKUserAttributes() {
        String userAttributesString = "{}";
        HashMap<String, String> allAttributes = getAll();
        HashMap<String, String> userAttributesStringStringHashMap = Filters.applyOn(allAttributes)
                .apply(AttributeFiltersFunctions.nonBackEndMapFilter())
                .thenGet();
        if (userAttributesStringStringHashMap != null && userAttributesStringStringHashMap
                .size() != 0) {
            UserAttributes userAttributes = new UserAttributes();
            userAttributes.putMap(userAttributesStringStringHashMap);
            userAttributesString = userAttributes.toString();
        }
        return userAttributesString;
    }

    @NonNull
    public static String getSDKUserAttributesAndAppendToIt(@NonNull Map<String, String> attributes) {
        String userAttributesString = "{}";
        HashMap<String, String> userAttributesStringStringHashMap = Filters.applyOn(getAll())
                .apply(AttributeFiltersFunctions.nonBackEndMapFilter())
                .thenGet();
        if (userAttributesStringStringHashMap != null) {
            if (attributes != null) {
                putExternalAttributes(attributes, userAttributesStringStringHashMap);
            }
            if (userAttributesStringStringHashMap.size() != 0) {
                UserAttributes userAttributes = new UserAttributes();
                userAttributes.putMap(userAttributesStringStringHashMap);
                userAttributesString = userAttributes.toString();
            }
        }
        return userAttributesString;
    }

    private static void putExternalAttributes(
            @NonNull Map<String, String> attributes,
            @NonNull HashMap<String, String> userAttributesStringStringHashMap
    ) {
        for (Map.Entry<String ,String> entry : attributes.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            if (key != null && !key.trim().isEmpty() && value != null && !value.trim().isEmpty())
                userAttributesStringStringHashMap.put(key.trim(), value.trim());
        }
    }


    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private static ContentValues getContentValue(UserAttribute userAttribute) {
        ContentValues values = new ContentValues();
        values.put(COLUMN_KEY, userAttribute.getKey());
        values.put(COLUMN_VALUE, userAttribute.getValue());
        values.put(COLUMN_UUID, userAttribute.getUuid());
        values.put(COLUMN_TYPE, userAttribute.getType());
        values.put(COLUMN_IS_ANONYMOUS, userAttribute.isAnonymous());
        return values;
    }

    @NonNull
    static List<UserAttribute> retrieveAnonymousUserAttribute() {
        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();

        final Cursor cursor = database.query(
                // Name of the table to read from
                TABLE_NAME,
                // String array of the columns which are supposed to be read
                null,                // The selection argument which specifies which row is read. // ? symbols are
                // parameters.
                COLUMN_IS_ANONYMOUS + "=? ",
                // The actual parameters values for the selection as a String array. // ? above
                // take the value
                // from here
                new String[]{"1"},
                // GroupBy clause. Specify a column name to group similar values // in that
                // column together.
                null,
                // Having clause. When using the GroupBy clause this allows you to // specify
                // which groups to
                // include.
                null,
                // OrderBy clause. Specify a column name here to order the results
                // according to that column. Optionally append ASC or DESC to specify // an
                // ascending or
                // descending order.
                null);
        // To increase performance first get the index of each column in the cursor
        if (cursor != null) {
            final int valueIndex = cursor.getColumnIndex(COLUMN_VALUE);
            final int keyIndex = cursor.getColumnIndex(COLUMN_KEY);
            final int uuidIndex = cursor.getColumnIndex(COLUMN_UUID);
            final int typeIndex = cursor.getColumnIndex(COLUMN_TYPE);

            try {
                // If moveToFirst() returns false then cursor is empty
                List<UserAttribute> userAttributeList = new ArrayList<>();
                if (!cursor.moveToFirst()) {
                    return userAttributeList;
                }
                // Read the values of a row in the table using the indexes acquired above

                do {
                    String value = cursor.getString(valueIndex);
                    String key = cursor.getString(keyIndex);
                    String uuid = cursor.getString(uuidIndex);
                    int type = cursor.getInt(typeIndex);
                    if (type != UserAttribute.Type.BACKEND && !isValidAttribute(key, value))
                        continue;

                    UserAttribute userAttribute = new UserAttribute.Builder(key, value)
                            .uuid(uuid)
                            .anonymous(true)
                            .type(type)
                            .build();
                    userAttributeList.add(userAttribute);

                } while (cursor.moveToNext());
                return userAttributeList;
            } catch (Exception e) { // To catch CursorWindowAllocationException
                IBGDiagnostics.reportNonFatalAndLog(e, "an exception has occurred while retrieving user attributes: " + e.getMessage(), Constants.LOG_TAG);
            } finally {
                // Don't forget to close the Cursor once you are done to avoid memory leaks.
                // Using a try/finally like in this example is usually the best way to handle this
                // close the database
                cursor.close();
                database.close();
            }
        }
        return Collections.emptyList();
    }

    public static void deleteAnonymousData() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_IS_ANONYMOUS + " = ?";
        String[] whereArgs = new String[]{"1"};
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static void insertBulk(List<UserAttribute> userAttributeList) {
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        db.beginTransaction();
        try {
            for (UserAttribute userAttribute : userAttributeList) {
                if (userAttribute.getType() != UserAttribute.Type.BACKEND && !isValidAttribute(userAttribute))
                    continue;
                // Create a new map of values, where column names are the keys
                ContentValues values = getContentValue(userAttribute);
                // Insert the new row, returning the primary key value of the new row
                long rowId = db.insertWithOnConflict(TABLE_NAME, null, values);
                if (rowId == -1) {
                    update(userAttribute);
                }
            }
            trimToLimit(CoreServiceLocator.getUserAttributesStoreLimit());
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }

    }

    private static void trimToLimit(long limit) {
        String[] whereArgs = {"-1", String.valueOf(limit)};
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            int deleted = db.delete(TABLE_NAME, TRIM_WHERE_CLAUSE, whereArgs);
            if (deleted > 0) {
                InstabugSDKLogger.w(Constants.LOG_TAG, "Some old user attributes were removed. " +
                        "Max allowed user attributes reached. " +
                        "Please note that you can add up to " + limit + " user attributes.");
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "Error while trimming user attributes: " + ex.getMessage());
        }
    }
}
