package com.instabug.library.internal.storage;

import android.annotation.SuppressLint;
import android.content.Context;
import android.os.Environment;

import androidx.annotation.Nullable;

import com.instabug.library.Instabug;
import com.instabug.library.util.extenstions.CtxKt;

import java.io.File;
import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;


public class AttachmentManager {

    /**
     * Retrieves the Instabug external attachment directory if available otherwise returns Instabug internal attachment directory.<br>
     * This method will create the directory if it does not exist, and create
     * a .nomedia file to prevent media scanning of the directory.
     *
     * @param context          The application context used to get access to storage directories.
     * @param createIfNotExist A boolean flag indicating whether to create the directory
     *                         and its contents if they do not exist. If false, returns null
     *                         without creating the directory.
     * @return The Instabug attachment directory as a {@link File} object, or null if the directory is not created
     * (when createIfNotExist is false).
     */
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    @Nullable
    public static File getAttachmentDirectory(@Nullable Context context, boolean createIfNotExist) {
        if (context == null) {
            if (Instabug.isBuilt())
                // get context from instabug
                context = Instabug.getApplicationContext();
        }
        if (context == null) {
            // return null if context still null
            return null;
        }

        String directoryPath;
        File externalFilesDir = CtxKt.getExternalFilesDir(context);
        if (externalFilesDir != null && Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED)) {
            try {
                directoryPath = externalFilesDir.getAbsolutePath();
            } catch (NullPointerException npe) {
                directoryPath = DiskUtils.getInternalStoragePath(context);
            }
        } else {
            directoryPath = DiskUtils.getInternalStoragePath(context);
        }
        File directory = new File(directoryPath + "/instabug/");

        if (directory.exists()) return directory;
        if (!createIfNotExist) return null;

        directory.mkdirs();
        File noMediaFile = new File(directory, ".nomedia");
        try {
            noMediaFile.createNewFile();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return directory;
    }

    @Nullable
    public static File getAttachmentInternalDirectory(@Nullable Context context) {
        if (context == null) {
            if (Instabug.isBuilt())
                // get context from instabug
                context = Instabug.getApplicationContext();
        }
        if (context == null) {
            // return null if context still null
            return null;
        }
        String directoryPath = DiskUtils.getInternalStoragePath(context);
        File directory = new File(directoryPath + "/instabug/");
        if (!directory.exists()) {
            directory.mkdirs();
            File noMediaFile = new File(directory, ".nomedia");
            try {
                noMediaFile.createNewFile();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return directory;
    }

    /**
     * Retrieves the directory with given name from Instabug attachment directory.<br>
     * This method will create the directory if it does not exist, and create
     * a .nomedia file to prevent media scanning of the directory.
     *
     * @param context          The application context used to get access to storage directories.
     * @param directoryName    The directory name to be retrieved.
     * @param createIfNotExist A boolean flag indicating whether to create the directory
     *                         and its contents if they do not exist. If false, returns null
     *                         without creating the directory.
     * @return The Instabug attachment directory as a {@link File} object, or null if the directory is not created
     * (when createIfNotExist is false).
     * @see AttachmentManager#getAttachmentDirectory(Context context, boolean createIfNotExist)
     */
    @Nullable
    public static File getNewDirectory(@Nullable Context context, String directoryName, boolean createIfNotExist) {
        File attachmentDir = getAttachmentDirectory(context, createIfNotExist);
        if (attachmentDir == null) return null;

        File directory = new File(attachmentDir + "/" + directoryName + "/");

        if (directory.exists()) return directory;
        if (!createIfNotExist) return null;

        directory.mkdirs();
        File noMediaFile = new File(directory, ".nomedia");
        try {
            noMediaFile.createNewFile();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return directory;
    }

    public static File getNewInternalDirectory(@Nullable Context context, String directoryName) {
        File directory = new File(getAttachmentInternalDirectory(context) + "/" + directoryName + "/");
        if (!directory.exists()) {
            directory.mkdirs();
            File noMediaFile = new File(directory, ".nomedia");
            try {
                noMediaFile.createNewFile();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return directory;
    }

    public static File getAutoScreenRecordingVideosDirectory(Context context) {
        File directory = new File(getAttachmentInternalDirectory(context) + "/auto_recording/");
        if (!directory.exists()) {
            directory.mkdirs();
            File noMediaFile = new File(directory, ".nomedia");
            try {
                noMediaFile.createNewFile();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return directory;
    }

    public static File getAutoScreenRecordingFile(Context context) {
        File videoDirectory;
        videoDirectory = getAutoScreenRecordingVideosDirectory(context);
        DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd_HH-mm-ss.SSS", Locale.ENGLISH);

        String videoName = "auto-recording-" + dateFormat.format(new Date()) + ".mp4";
        return new File(videoDirectory, videoName);
    }

    public static void deleteRecursive(File fileOrDirectory) {
        if (fileOrDirectory.isDirectory()) {
            File[] fileList = fileOrDirectory.listFiles();
            if (fileList != null && fileList.length > 0)
                for (File child : fileList)
                    deleteRecursive(child);
            fileOrDirectory.delete();
        } else {
            fileOrDirectory.delete();
        }
    }

}
