package com.instabug.library.internal.sharedpreferences

import android.content.Context
import android.content.SharedPreferences
import androidx.annotation.VisibleForTesting
import com.instabug.library.Feature
import com.instabug.library.InstabugFeaturesManager
import com.instabug.library.encryption.EncryptionManager
import com.instabug.library.internal.servicelocator.CoreServiceLocator


class InstabugSharedPreferences(private val sharedPreferences: SharedPreferences) :
    SharedPreferences {
    companion object {
        fun create(context: Context, name: String): InstabugSharedPreferences? {
            return context.getSharedPreferences(name, Context.MODE_PRIVATE)
                ?.let { InstabugSharedPreferences(it) }
        }
    }

    override fun getAll(): MutableMap<String, *>? {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet { sharedPreferences.all }
    }

    override fun getString(key: String?, defValue: String?): String? {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet {
                val plainValue = sharedPreferences.getString(key, defValue)
                if (InstabugFeaturesManager.getInstance().encryptionState == Feature.State.ENABLED) {
                    val decryptedValue = EncryptionManager.decrypt(plainValue)
                    if (decryptedValue != null) {
                        return@executeAndGet decryptedValue
                    } else {
                        return@executeAndGet plainValue
                    }
                }
                return@executeAndGet plainValue
            }
    }

    override fun getStringSet(key: String?, defValues: MutableSet<String>?): MutableSet<String>? {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet {
                val stringSet = sharedPreferences.getStringSet(key, defValues)
                val decryptedSet = mutableSetOf<String>()
                if (InstabugFeaturesManager.getInstance().encryptionState == Feature.State.ENABLED) {
                    stringSet?.forEach {
                        val decryptedValue = EncryptionManager.decrypt(it)
                        if (decryptedValue != null) {
                            decryptedSet.add(decryptedValue)
                        } else {
                            decryptedSet.add(it)
                        }
                    }
                    return@executeAndGet decryptedSet
                }
                stringSet
            }
    }

    override fun getInt(key: String?, defValue: Int): Int {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet { sharedPreferences.getInt(key, defValue) } ?: defValue
    }

    override fun getLong(key: String?, defValue: Long): Long {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet { sharedPreferences.getLong(key, defValue) } ?: defValue
    }

    override fun getFloat(key: String?, defValue: Float): Float {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet { sharedPreferences.getFloat(key, defValue) } ?: defValue
    }

    override fun getBoolean(key: String?, defValue: Boolean): Boolean {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet { sharedPreferences.getBoolean(key, defValue) } ?: defValue
    }

    override fun contains(key: String?): Boolean {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet { sharedPreferences.contains(key) } ?: false
    }

    override fun edit(): SharedPreferences.Editor {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet { Editor(sharedPreferences.edit()) } ?: Editor(sharedPreferences.edit())
    }

    override fun registerOnSharedPreferenceChangeListener(listener: SharedPreferences.OnSharedPreferenceChangeListener?) {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .execute {
                sharedPreferences.registerOnSharedPreferenceChangeListener(listener)
            }
    }

    override fun unregisterOnSharedPreferenceChangeListener(listener: SharedPreferences.OnSharedPreferenceChangeListener?) {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .execute {
                sharedPreferences.unregisterOnSharedPreferenceChangeListener(listener)
            }
    }
}


class Editor constructor(private val editor: SharedPreferences.Editor) :
    SharedPreferences.Editor {

    override fun putString(key: String?, value: String?): SharedPreferences.Editor {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet {
                if (InstabugFeaturesManager.getInstance().encryptionState == Feature.State.ENABLED) {
                    val encryptedValue = EncryptionManager.encrypt(value)
                    if (encryptedValue != null) {
                        editor.putString(key, encryptedValue)
                    } else {
                        editor.putString(key, value)
                    }
                } else {
                    editor.putString(key, value)
                }
                this
            } ?: this

    }

    override fun putStringSet(
        key: String?,
        values: MutableSet<String>?
    ): SharedPreferences.Editor {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet {
                if (InstabugFeaturesManager.getInstance().encryptionState == Feature.State.ENABLED) {
                    val encryptedSet = mutableSetOf<String>()
                    values?.forEach {
                        val encryptedValue = EncryptionManager.encrypt(it)
                        if (encryptedValue != null) {
                            encryptedSet.add(encryptedValue)
                        } else {
                            encryptedSet.add(it)
                        }
                    }
                    editor.putStringSet(key, encryptedSet)
                } else {
                    editor.putStringSet(key, values)
                }
                this
            }
            ?: if (InstabugFeaturesManager.getInstance().encryptionState == Feature.State.ENABLED) {
                val encryptedSet = mutableSetOf<String>()
                values?.forEach {
                    val encryptedValue = EncryptionManager.encrypt(it)
                    if (encryptedValue != null) {
                        encryptedSet.add(encryptedValue)
                    } else {
                        encryptedSet.add(it)
                    }
                }
                editor.putStringSet(key, encryptedSet)
            } else {
                editor.putStringSet(key, values)
            }
    }

    override fun putInt(key: String?, value: Int): SharedPreferences.Editor {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet {
                editor.putInt(key, value)
                this
            } ?: editor.putInt(key, value)

    }

    override fun putLong(key: String?, value: Long): SharedPreferences.Editor {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet {
                editor.putLong(key, value)
                this
            } ?: editor.putLong(key, value)
    }

    override fun putFloat(key: String?, value: Float): SharedPreferences.Editor {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet {
                editor.putFloat(key, value)
                this
            } ?: editor.putFloat(key, value)
    }

    override fun putBoolean(key: String?, value: Boolean): SharedPreferences.Editor {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet {
                editor.putBoolean(key, value)
                this
            } ?: editor.putBoolean(key, value)
    }

    override fun remove(key: String?): SharedPreferences.Editor {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet {
                editor.remove(key)
                this
            } ?: editor.remove(key)
    }

    override fun clear(): SharedPreferences.Editor {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet {
                editor.clear()
                this
            } ?: editor.clear()
    }

    override fun commit(): Boolean {
        return CoreServiceLocator.getSharedPreferencesExecutor()
            .executeAndGet {
                editor.commit()
            } ?: editor.commit()
    }

    override fun apply() {
        CoreServiceLocator.getSharedPreferencesExecutor()
            .execute {
                editor.commit()
            }
    }
}