package com.instabug.library;

import static com.instabug.library.networkinterception.config.ConstantsKt.AUTO_MASKING_ENABLED_BE_FLAG_DISABLED;
import static com.instabug.library.sessionV3.sync.SessionBatchingFilterKt.getAllFilter;
import static com.instabug.library.sessionV3.sync.SessionBatchingFilterKt.getDataReadinessFilter;
import static com.instabug.library.sessionV3.sync.SessionBatchingFilterKt.getNoneFilter;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.content.IntentFilter;
import android.graphics.Bitmap;
import android.os.Build;
import android.util.Pair;
import android.view.View;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;

import com.instabug.library.broadcast.SDKInvokedBroadcast;
import com.instabug.library.core.IBGStateEventBusSubscriber;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.InstabugStateEventBus;
import com.instabug.library.core.eventbus.NDKSessionCrashedEvent;
import com.instabug.library.core.eventbus.OnSessionCrashedEventBus;
import com.instabug.library.core.eventbus.SessionStateEventBus;
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventPublisher;
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventSubscriber;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent;
import com.instabug.library.core.eventbus.eventpublisher.IBGDisposable;
import com.instabug.library.core.plugin.PluginsManager;
import com.instabug.library.coreSDKChecks.CoreSDKChecks;
import com.instabug.library.crash.InstabugUncaughtExceptionHandler;
import com.instabug.library.customizations.Customizations;
import com.instabug.library.diagnostics.DiagnosticsCoreEventHandler;
import com.instabug.library.diagnostics.DiagnosticsSessionStateEventHandler;
import com.instabug.library.encryption.FilesMigrationEngine;
import com.instabug.library.experiments.di.ServiceLocator;
import com.instabug.library.featuresflags.di.FeaturesFlagServiceLocator;
import com.instabug.library.featuresflags.model.IBGFeatureFlag;
import com.instabug.library.firstseen.FirstSeenRequestFetcher;
import com.instabug.library.internal.InstabugMediaProjectionIntent;
import com.instabug.library.internal.dataretention.DataGarbageCollector;
import com.instabug.library.internal.dataretention.files.FileInspector;
import com.instabug.library.internal.orchestrator.ActionsOrchestrator;
import com.instabug.library.internal.orchestrator.DisposeDataAction;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.sharedpreferences.SharedPreferencesMigrationEngine;
import com.instabug.library.internal.storage.AttachmentsUtility;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.internal.storage.cache.AssetsCacheManager;
import com.instabug.library.internal.storage.cache.UserAttributesCacheManager;
import com.instabug.library.internal.storage.cache.db.userAttribute.UserAttributeCacheManager;
import com.instabug.library.internal.video.InternalAutoScreenRecorderHelper;
import com.instabug.library.migration.MigrationManager;
import com.instabug.library.model.FeaturesCache;
import com.instabug.library.model.StepType;
import com.instabug.library.model.session.SessionState;
import com.instabug.library.model.session.config.SessionsConfig;
import com.instabug.library.model.v3Session.SessionEvent;
import com.instabug.library.networkinterception.NetworkInterceptionServiceLocator;
import com.instabug.library.networkinterception.config.IBGNetworkInterceptionConfigurationProvider;
import com.instabug.library.networkv2.detectors.IBGNetworkAvailabilityManager;
import com.instabug.library.networkv2.service.synclogs.SyncLogFacade;
import com.instabug.library.networkv2.service.userattributes.AttributesSyncManager;
import com.instabug.library.screenshot.analytics.AnalyticsEvent;
import com.instabug.library.screenshot.subscribers.IBGScreenshotsAnalyticsSubscriber;
import com.instabug.library.screenshot.subscribers.ScreenshotsAnalyticsEventBus;
import com.instabug.library.session.SessionsSyncManager;
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator;
import com.instabug.library.sessionV3.manager.IBGSessionManager;
import com.instabug.library.sessionprofiler.SessionProfiler;
import com.instabug.library.sessionreplay.di.SessionReplayServiceLocator;
import com.instabug.library.settings.InstabugMinimalPersistableSettings;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.tokenmapping.TokenMappingServiceLocator;
import com.instabug.library.tracking.CurrentViewProvider;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.user.UserManager;
import com.instabug.library.util.FileUtils;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.OrientationUtils;
import com.instabug.library.util.TaskDebouncer;
import com.instabug.library.util.filters.AttributeFiltersFunctions;
import com.instabug.library.util.filters.Filters;
import com.instabug.library.util.threading.PoolProvider;

import java.io.File;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;

import io.reactivexport.disposables.Disposable;
import io.reactivexport.functions.Consumer;

class InstabugDelegate implements SDKInvokedBroadcast.SDIInvokedListener {

    private final SDKInvokedBroadcast sdkInvokedBroadcast = new SDKInvokedBroadcast(this);
    private final AttributesSyncManager attributesSyncManager;
    private final SessionsSyncManager sessionsSyncManager;
    private final Application application;
    private WeakReference<Context> applicationContextWeakReference;
    @Nullable
    private Disposable sessionStateDisposable;
    @Nullable
    private Disposable onSessionCrashedDisposable;

    @Nullable
    IBGDisposable sdkCoreEventsDisposable;
    private final TaskDebouncer disposeActionDebouncer = new TaskDebouncer(30_000);
    private final TaskDebouncer syncServicesDebouncer = new TaskDebouncer(3_000);
    private boolean isStarted;
    private final FirstSeenRequestFetcher firstSeenRequestFetcher;
    private boolean isFirstSeenFetched = false;
    private final DiagnosticsCoreEventHandler diagnosticsCoreEventHandler;
    private final CoreSDKChecks coreSDKChecks;

    private static InstabugDelegate instance;

    private InstabugDelegate(final Application application) {
        Context context = application.getApplicationContext();
        this.applicationContextWeakReference = new WeakReference<>(context);
        this.firstSeenRequestFetcher = FirstSeenRequestFetcher.getInstance();
        this.attributesSyncManager = AttributesSyncManager.create(context);
        this.sessionsSyncManager = SessionsSyncManager.create(context);
        this.application = application;
        this.isStarted = false;
        this.diagnosticsCoreEventHandler = new DiagnosticsCoreEventHandler();
        this.coreSDKChecks = new CoreSDKChecks();
        InstabugInternalTrackingDelegate.init(application);
    }

    public static synchronized InstabugDelegate getInstance(final Application application) {
        if (instance == null) {
            instance = new InstabugDelegate(application);
        }
        return instance;
    }

    public void prepare(Context context) {
        PluginsManager.initPluginsPromptOptionAvailability();
        prepareCaches();
    }

    public void restoreFeaturesFromSharedPreferences(Context context) {
        InstabugFeaturesManager.getInstance().restoreFeaturesFromSharedPreferences(context);
    }

    @Override
    public void onSDKInvoked(boolean isInvoked) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "SDK Invoked: " + isInvoked);
        InstabugState instabugState = getInstabugState();
        if (instabugState != InstabugState.TAKING_SCREENSHOT && instabugState !=
                InstabugState.RECORDING_VIDEO
                && instabugState != InstabugState.TAKING_SCREENSHOT_FOR_CHAT
                && instabugState != InstabugState.RECORDING_VIDEO_FOR_CHAT
                && instabugState != InstabugState.IMPORTING_IMAGE_FROM_GALLERY_FOR_CHAT) {

            if (isInvoked) {
                setInstabugState(InstabugState.INVOKED);
            } else {

                //unlock required because the previous state might be related to screen recording
                Activity activity = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
                if (activity != null) {
                    OrientationUtils.unlockOrientation(activity);
                }
                if (InstabugFeaturesManager.getInstance().isFeatureAvailable(IBGFeature.INSTABUG)) {
                    setInstabugState(InstabugState.ENABLED);
                } else {
                    setInstabugState(InstabugState.DISABLED);
                }
            }
        }
    }

    private void subscribeToSdkEvents() {
        if (sdkCoreEventsDisposable == null)
            sdkCoreEventsDisposable = subscribeToSDKCoreEvents();
    }

    private void handleFeaturesFetched() {
        if (InstabugCore.getRunningSession() == null) return;
        applyCustomizationsIfAny();
        if (!isFirstSeenFetched) {
            firstSeenRequestFetcher.fetchFirstSeenRequest(false);
            isFirstSeenFetched = true;
        }
    }

    @NonNull
    private IBGDisposable subscribeToSDKCoreEvents() {
        return IBGCoreEventSubscriber.subscribe((this::handleSDKCoreEvents));
    }

    private void handleSDKCoreEvents(IBGSdkCoreEvent coreEvent) {
        if (coreEvent instanceof IBGSdkCoreEvent.NetworkActivated)
            handleNetworkActivated();
        else if (coreEvent instanceof IBGSdkCoreEvent.User.LoggedOut)
            handleLogoutEvent();
        else if (coreEvent instanceof IBGSdkCoreEvent.Features.Fetched)
            handleFeaturesFetched();
        else if (coreEvent instanceof IBGSdkCoreEvent.CrossPlatformCrashed) {
            handleCrashingSession();
        }
        diagnosticsCoreEventHandler.handleCoreEvents(coreEvent);
        SessionReplayServiceLocator.getSessionReplayDelegate().onNewEvent(coreEvent);
        CoreServiceLocator.getDataHubController().onNewEvent(coreEvent);
    }

    private static void handleCrashingSession() {
        IBGSessionManager.INSTANCE.emit(new SessionEvent.End(), true);
        IBGSessionServiceLocator.getBatchEvaluator().evaluate(getNoneFilter());
    }

    private void handleNetworkActivated() {
        startFetchPlanFeatures();

        FeaturesCache lastFeaturesSettings = InstabugFeaturesManager.getInstance().getLastFeaturesSettings();
        if (lastFeaturesSettings != null && !lastFeaturesSettings.isActive()) {
            return;
        }

        runSyncServices();

        if (TokenMappingServiceLocator.getTokenMappingConfigs().isTokenMappingEnabled()) {
            TokenMappingServiceLocator.getTokenMappingSync().start();
        }

        if (InstabugCore.getRunningSession() != null) {
            PoolProvider.postIOTask(() -> firstSeenRequestFetcher.fetchFirstSeenRequest(false));
        }
    }

    private void handleLogoutEvent() {
        PoolProvider.postIOTask(() -> {
            sessionsSyncManager.pushSyncInterval().evaluateSessions().sync();
            IBGSessionServiceLocator.getSyncManger().sync(getDataReadinessFilter());
        });
    }

    private void unsubscribeFromSdkEvents() {
        if (sdkCoreEventsDisposable != null) {
            sdkCoreEventsDisposable.dispose();
            sdkCoreEventsDisposable = null;
        }
    }

    private InstabugState getInstabugState() {
        return InstabugStateProvider.getInstance().getState();
    }

    protected void setInstabugState(@NonNull InstabugState state) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Setting Instabug State to " + state);
        if (state != getInstabugState()) {
            InstabugStateProvider.getInstance().setState(state);
            InstabugStateEventBus.getInstance().post(state);
        }
    }

    protected void updateInstabugFeatureState(Feature.State featureState) {
        InstabugFeaturesManager.getInstance().setFeatureState(IBGFeature.INSTABUG, featureState);
        if (getApplicationContext() != null) {
            InstabugFeaturesManager.getInstance().saveFeaturesToSharedPreferences(getApplicationContext());
            new InstabugMinimalPersistableSettings(getApplicationContext())
                    .setLastSDKStateEnabled(featureState == Feature.State.ENABLED);
        }
    }

    private void subscribeToSessionStateEvents() {
        if (sessionStateDisposable != null) return;

        sessionStateDisposable = SessionStateEventBus.getInstance().subscribe(sessionState -> {
            DiagnosticsSessionStateEventHandler.handleSessionState(sessionState);
            if (sessionState.equals(SessionState.FINISH)) {
                InstabugSDKLogger.logEndSession(System.currentTimeMillis());
                if (!PluginsManager.isForegroundBusy()) {
                    dumpCaches();
                }
                handleSyncServices();
                PluginsManager.sleep();
            } else if (sessionState.equals(SessionState.START)) {
                SessionsConfig config = SettingsManager.getInstance().getSessionsSyncConfigurations();
                sessionsSyncManager.onConfigurationsChanged(config);
                InstabugSDKLogger.logSessionDetails(new SessionDetailsProvider(getApplicationContext()).getSessionDetails());
                disposeActionDebouncer.debounce(this::runDataGarbageCollector);
                subscribeToSdkEvents();
                runSyncServices();
                PluginsManager.wake();
                startFetchPlanFeatures();
            }
        });
    }

    public void handleSyncServices() {
        if (getInstabugState() == InstabugState.DISABLED) {
            runSyncServices();
        }
    }

    private void prepareCaches() {
        Context context = applicationContextWeakReference.get();
        if (context != null) UserAttributesCacheManager.prepareCaches(context);
        else
            InstabugSDKLogger.e(Constants.LOG_TAG, "can't execute prepareCaches() due to null context");
    }

    private void dumpCaches() {
        PoolProvider.postIOTask(new Runnable() {
            @Override
            public void run() {
                InstabugSDKLogger.v(Constants.LOG_TAG, "Dumping caches");
                if (applicationContextWeakReference != null) {
                    Context ctx = applicationContextWeakReference.get();
                    if (ctx != null) {
                        AssetsCacheManager.cleanUpCache(ctx);
                        IBGSdkCoreEvent cacheDumpedEvent = IBGSdkCoreEvent.CacheDumped.INSTANCE;
                        IBGCoreEventPublisher.post(cacheDumpedEvent);
                    }
                }
            }
        });
    }

    private void startFetchPlanFeatures() {
        if (applicationContextWeakReference != null) {
            final Context context = applicationContextWeakReference.get();
            if (context != null) {
                PoolProvider.postIOTask(() -> InstabugFeaturesManager.getInstance().fetchPlanFeatures(context));
            } else {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't fetch plan features because Context was null.");
            }
        }
    }

    private void runSyncServices() {
        syncServicesDebouncer.debounce(() -> {
            FeaturesCache featuresCache = InstabugFeaturesManager.getInstance().getLastFeaturesSettings();
            final boolean isSdkDisabled = getInstabugState() == InstabugState.DISABLED;
            boolean isSdkActive = true;

            if (featuresCache != null) {
                isSdkActive = featuresCache.isActive();
            }

            if (!isSdkDisabled && isSdkActive) {
                attributesSyncManager.sync();
            }

            if (isSdkDisabled || !isSdkActive) {
                sessionsSyncManager.pushSyncInterval();
            }

            PoolProvider.postIOTask(() -> {
                sessionsSyncManager.evaluateSessions().sync();
                IBGSessionServiceLocator.getSyncManger()
                        .sync(isSdkDisabled ? getDataReadinessFilter() : getAllFilter());
            });
        });

        ActionsOrchestrator.obtainOrchestrator()
                .addWorkerThreadAction(this::runLogSyncService).orchestrate();
    }

    private void runLogSyncService() {
        SyncLogFacade syncLogFacade = SyncLogFacade.getInstance();
        syncLogFacade.setUserData(UserManager.getUUID(),
                UserManager.getIdentifiedUserEmail());
        if (getApplicationContext() != null && SettingsManager.getInstance().getAppToken() != null) {
            syncLogFacade.run(getApplicationContext(), SettingsManager.getInstance().getAppToken());
        }
    }


    private void runDataGarbageCollector() {
        ActionsOrchestrator.obtainOrchestrator()
                .addWorkerThreadAction(new DisposeDataAction(FileInspector.create()))
                .orchestrate();
    }

    @WorkerThread
    void start() {
        boolean instabugAvailable = InstabugFeaturesManager.getInstance().isFeatureAvailable(IBGFeature.INSTABUG);
        boolean instabugEnabled = InstabugFeaturesManager.getInstance().getFeatureState(IBGFeature.INSTABUG) == Feature.State.ENABLED;

        if (instabugAvailable && instabugEnabled) {
            startSdk();
        } else {
            setInstabugState(InstabugState.DISABLED);
        }
    }

    @WorkerThread
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    synchronized void startSdk() {
        if (isStarted) {
            return;
        }
        isStarted = true;
        IBGScreenshotsAnalyticsSubscriber.subscribeToCommonAnalyticsCollector();
        CoreServiceLocator.getDevicePerformanceClassConfig().initConfigs();
        CoreServiceLocator.getDataHubController().init(AppLaunchIDProvider.INSTANCE.getSpanId());
        IBGStateEventBusSubscriber.subscribeToIBGState();
        handleInterruptedMigrations();
        AttachmentsUtility.clearInternalAttachments(getApplicationContext());
        AssetsCacheManager.clearExternalCacheDir(getApplicationContext());
        subscribeToSdkEvents();
        IBGNetworkAvailabilityManager.registerNetworkCallbacks(getApplicationContext());
        registerOnSessionCrashedEventBus();
        PluginsManager.startPlugins(getApplicationContext());
        coreSDKChecks.check(Build.VERSION.SDK_INT, BuildConfig.SDK_VERSION);
        restoreFeaturesFromSharedPreferences(getApplicationContext());
        // Drop the files if needed before doing anything else
        checkEncryptorVersion();
        //subscribe to session state events
        notifyVisualUserSteps();
        subscribeToSessionStateEvents();
        initializeExceptionHandler();
        InstabugSDKLogger.d(Constants.LOG_TAG, "Starting Instabug SDK functionality");
        setInstabugState(InstabugState.ENABLED);
        updateInstabugFeatureState(Feature.State.ENABLED);
        SessionManager.getInstance().handleSessionStartEvent();
        IBGSessionManager.INSTANCE.emit(new SessionEvent.Start());
        InstabugSDKLogger.v(Constants.LOG_TAG, "Disposing expired data");
        DataGarbageCollector.deprecatedCollect().disposeAsync();
        InstabugSDKLogger.v(Constants.LOG_TAG, "Running valid migration");
        startMigration();
        InstabugSDKLogger.v(Constants.LOG_TAG, "Registering broadcasts");
        registerBroadcasts();
        InstabugSDKLogger.v(Constants.LOG_TAG, "Preparing user state");
        prepareUserState();
        InstabugSDKLogger.v(Constants.LOG_TAG, "Initializing auto screen recording");
        initializeAutoScreenRecording();
        SessionProfiler.getInstance().start();
        registerActivityLifeCycle();
    }

    private void handleInterruptedMigrations() {
        SharedPreferencesMigrationEngine.runMigrationIfNeeded(
                InstabugFeaturesManager.getInstance().getEncryptionState() == Feature.State.ENABLED,
                getApplicationContext());
        // Drop the files if needed before doing anything else
        checkEncryptorVersion();
        FilesMigrationEngine.runMigration();
    }


    private void registerOnSessionCrashedEventBus() {
        onSessionCrashedDisposable = OnSessionCrashedEventBus.getInstance().subscribe(new Consumer<NDKSessionCrashedEvent>() {
            @Override
            public void accept(NDKSessionCrashedEvent sessionCrashedEvent) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "NDK crashing session found. Sync old sessions");
                PoolProvider.postIOTask(new Runnable() {
                    @Override
                    public void run() {
                        sessionsSyncManager
                                .pushSyncInterval()
                                .evaluateSessions()
                                .sync();
                        IBGSessionServiceLocator
                                .getSyncManger()
                                .sync(getNoneFilter());
                    }
                });
            }
        });
    }

    private void notifyVisualUserSteps() {
        if (getInstabugState() == InstabugState.ENABLED) {
            CoreServiceLocator.getReproStepsProxy().logInstabugEnabledStep();
        } else if (getInstabugState() == InstabugState.DISABLED) {
            CoreServiceLocator.getReproStepsProxy().clean();
            CoreServiceLocator.getReproStepsProxy().reset();
        }
    }

    private void registerActivityLifeCycle() {
        if (!InstabugInternalTrackingDelegate.getInstance().isRegistered()) {
            InstabugInternalTrackingDelegate.getInstance().registerLifecycleListeners(application);
        }
    }

    private void initializeExceptionHandler() {
        //Setup Uncaught Exception Handler
        InstabugSDKLogger.d(Constants.LOG_TAG, "setting Uncaught Exception Handler com.instabug.library.crash.InstabugUncaughtExceptionHandler");
        Thread.setDefaultUncaughtExceptionHandler(new InstabugUncaughtExceptionHandler());
    }

    protected void applyCustomizationsIfAny() {
        if (InstabugFeaturesManager.getInstance().getFeatureState(IBGFeature.VP_CUSTOMIZATION) ==
                Feature.State.ENABLED) {
            Customizations.vpCustomizations();
        }
    }

    public void registerBroadcasts() {
        if (getApplicationContext() == null) {
            InstabugSDKLogger.e(Constants.LOG_TAG,
                    "Unable to register a LocalBroadcast receiver because of a null context");
            return;
        }
        LocalBroadcastManager.getInstance(getApplicationContext()).registerReceiver
                (sdkInvokedBroadcast,
                        new IntentFilter(SDKInvokedBroadcast.SDK_INVOKED));
    }

    private void prepareUserState() {
        UserManager.prepareUserState();
    }

    private void startMigration() {
        if (getApplicationContext() == null) {
            InstabugSDKLogger.e(Constants.LOG_TAG,
                    "Unable to start migration because of a null context");
            return;
        }
        MigrationManager.startMigration(getApplicationContext());
    }

    private void unSubscribeFromSessionStateEvents() {
        if (sessionStateDisposable != null) {
            sessionStateDisposable.dispose();
            sessionStateDisposable = null;
        }
    }


    private void unSubscribeFromOnSessionCrashedEvents() {
        if (onSessionCrashedDisposable != null) {
            onSessionCrashedDisposable.dispose();
            onSessionCrashedDisposable = null;
        }
    }


    protected void stopSdk() {
        if (isEnabled()) {
            stop();
        }
    }

    synchronized private void stop() {
        PoolProvider.postIOTaskWithCheck(() -> {
            IBGNetworkAvailabilityManager.unregisterNetworkCallbacks(getApplicationContext());
            InstabugSDKLogger.d(Constants.LOG_TAG, "Stopping Instabug SDK functionality");
            InstabugCore.setAutoScreenRecordingEnabled(false);
            setInstabugState(InstabugState.DISABLED);
            updateInstabugFeatureState(Feature.State.DISABLED);
            SessionProfiler.getInstance().stop();
            unregisterBroadcasts();
            ScreenshotsAnalyticsEventBus.INSTANCE.post(AnalyticsEvent.ClearScreenShotAnalyticsData.INSTANCE);
            PluginsManager.stopPlugins();
            SessionManager.getInstance().stopCurrentSession();
            IBGSessionManager.INSTANCE.emit(new SessionEvent.Stop());
            InstabugInternalTrackingDelegate.getInstance().unregisterLifecycleListeners(application);
            notifyVisualUserSteps();
            clearDatabase();
            clearStateFiles();
            DiskUtils.deleteSDKDirectory();// unSubscribe from session state events
            IBGScreenshotsAnalyticsSubscriber.unSubscribe();
            unSubscribeFromSessionStateEvents();
            unsubscribeFromSdkEvents();
            unSubscribeFromOnSessionCrashedEvents();
            IBGStateEventBusSubscriber.unsubscribeFromIBGState();
            isStarted = false;
            InstabugMediaProjectionIntent.release();
            new DisposeDataAction(FileInspector.create()).run();
        });
    }

    private void clearStateFiles() {
        DiskUtils.deleteAllStateFiles();
        DiskUtils.deleteNonfatalStateFiles();
    }

    private void clearDatabase() {
        CoreServiceLocator.getSdkCleaningUtil().clearDatabase();
    }

    private boolean isEnabled() {
        return getInstabugState() != InstabugState.NOT_BUILT &&
                InstabugFeaturesManager.getInstance().isFeatureAvailable(IBGFeature.INSTABUG)
                && InstabugFeaturesManager.getInstance().getFeatureState(IBGFeature.INSTABUG) == Feature.State.ENABLED;
    }

    protected synchronized void pauseSdk() {
        if (getInstabugState().equals(InstabugState.ENABLED)) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "Pausing Instabug SDK functionality temporary");
            setInstabugState(InstabugState.DISABLED);

            try {
                IBGNetworkAvailabilityManager.unregisterNetworkCallbacks(getApplicationContext());
                SessionManager.getInstance().stopCurrentSession();
                IBGSessionManager.INSTANCE.emit(new SessionEvent.Stop());
                saveFeaturesToSharedPreferences();
                ScreenshotsAnalyticsEventBus.INSTANCE.post(AnalyticsEvent.ClearScreenShotAnalyticsData.INSTANCE);
                PluginsManager.stopPlugins();
                SessionProfiler.getInstance().stop();
                unregisterBroadcasts();
                unsubscribeFromSdkEvents();
                unSubscribeFromOnSessionCrashedEvents();
                IBGStateEventBusSubscriber.unsubscribeFromIBGState();
                InstabugMediaProjectionIntent.release();
                setInstabugState(InstabugState.DISABLED);
                updateInstabugFeatureState(Feature.State.DISABLED);
                IBGScreenshotsAnalyticsSubscriber.unSubscribe();
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while Pausing Instabug SDK", e);
            }
        }
    }

    protected synchronized void resumeSdk() {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Resuming Instabug SDK");
        setInstabugState(InstabugState.ENABLED);

        try {
            if (Instabug.getApplicationContext() == null) {
                return;
            }
            saveFeaturesToSharedPreferences();
            PluginsManager.startPlugins(Instabug.getApplicationContext());
            SessionProfiler.getInstance().start();
            registerBroadcasts();
            subscribeToSdkEvents();
            IBGNetworkAvailabilityManager.registerNetworkCallbacks(getApplicationContext());
            IBGScreenshotsAnalyticsSubscriber.subscribeToCommonAnalyticsCollector();
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while Resuming Instabug SDK", e);
        }
    }

    private void saveFeaturesToSharedPreferences() {
        Context context = Instabug.getApplicationContext();
        if (context != null) {
            InstabugFeaturesManager.getInstance().saveFeaturesToSharedPreferences(context);
        }
    }

    public void unregisterBroadcasts() {
        if (getApplicationContext() != null) {
            LocalBroadcastManager.getInstance(getApplicationContext()).unregisterReceiver
                    (sdkInvokedBroadcast);
        }
    }

    @Nullable
    public Context getApplicationContext() {
        if (applicationContextWeakReference.get() == null)
            InstabugSDKLogger.e(Constants.LOG_TAG, "Application context instance equal null");
        return applicationContextWeakReference.get();
    }

    public void dismiss() {
        // todo check if we should implement this or not
    }

    private void initializeAutoScreenRecording() {
        InternalAutoScreenRecorderHelper.getInstance().start();
    }

    @SuppressLint("NULL_DEREFERENCE")
    void setLocale(Locale newLocale) {
        Locale oldLocale = SettingsManager.getInstance().getInstabugLocale(getApplicationContext());
        if (!oldLocale.equals(newLocale)) {
            SettingsManager.getInstance().setInstabugLocale(newLocale);
            PluginsManager.notifyPluginsLocaleChanged(oldLocale, newLocale);
        }
    }

    public void addPrivateViews(@NonNull View... views) {
        SettingsManager.getInstance().addPrivateViews(views);
    }

    public void removePrivateViews(@NonNull View... views) {
        SettingsManager.getInstance().removePrivateViews(views);
    }

    public void removeAllPrivateViews() {
        SettingsManager.getInstance().removeAllPrivateViews();
    }

    void setSessionProfilerState(Feature.State state) {
        InstabugFeaturesManager.getInstance().setFeatureState(IBGFeature.SESSION_PROFILER, state);
        if (state == Feature.State.ENABLED && Instabug.isEnabled())
            SessionProfiler.getInstance().start();
        else
            SessionProfiler.getInstance().stop();
    }

    public HashMap<String, String> getAllUserAttributes() {
        return Filters.applyOn(UserAttributeCacheManager.retrieveAll())
                .apply(AttributeFiltersFunctions.nonBackEndMapFilter())
                .thenGet();
    }

    @Nullable
    public String getUserAttribute(String key) {
        return Filters.applyOn(key)
                .apply(AttributeFiltersFunctions.nonBackEndKeyFilter())
                .thenDoReturn(AttributeFiltersFunctions.attributeValueReturnableAction());
    }

    public void setUserAttribute(final String key, final String value) {
        PoolProvider.getUserActionsExecutor().execute(new Runnable() {
            @Override
            public void run() {
                Filters.applyOn(new Pair<>(key, value))
                        .apply(AttributeFiltersFunctions.nonBackEndKeyValuePairFilter())
                        .thenDo(AttributeFiltersFunctions.insertAttributeAction());
            }
        });
    }

    public void removeUserAttribute(final String key) {
        PoolProvider.getUserActionsExecutor().execute(new Runnable() {
            @Override
            public void run() {
                Filters.applyOn(key)
                        .apply(AttributeFiltersFunctions.nonBackEndKeyFilter())
                        .thenDo(AttributeFiltersFunctions.deleteAttributeAction());
            }
        });
    }

    public void clearAllUserAttributes() {
        PoolProvider.getUserActionsExecutor().execute(new Runnable() {
            @Override
            public void run() {
                Filters.applyOn(UserAttributeCacheManager.retrieveAll())
                        .apply(AttributeFiltersFunctions.nonBackEndMapFilter())
                        .thenDo(AttributeFiltersFunctions.deleteAttributesAction());
            }
        });
    }

    private void checkEncryptorVersion() {
        if (InstabugCore.isFirstRunAfterEncryptorUpdate()) {
            PoolProvider.postIOTask(new Runnable() {
                @Override
                public void run() {
                    deleteInstabugFiles();
                }
            });
        }
    }

    /*
     * Due to the latest changes in Encryption/Decryption processing (Previously the process was done for the whole
     * file vs now the Process is done for the first 256 bytes only no matter how big the file is).
     * It was decided to drop any files related to a Bug or a Crash report that exists before this change.*/
    @WorkerThread
    @VisibleForTesting
    void deleteInstabugFiles() {
        if (Instabug.getApplicationContext() != null) {
            File instabugDirectory = DiskUtils.getInstabugDirectory(Instabug.getApplicationContext(), false);
            if (instabugDirectory != null) {
                ArrayList<File> instabugFiles = com.instabug.library.util.DiskUtils.listFilesInDirectory(instabugDirectory);
                for (File file : instabugFiles) {
                    if (FileUtils.isFileRelatedToBugOrCrashReport(file.getPath())) {
                        file.delete();
                    }
                }
            }
            InstabugCore.setFirstRunAfterEncryptorUpdate(false);
        }
    }

    public void reportScreenChange(@Nullable Bitmap screenshot, String screenName) {
        if (screenName == null || screenName.trim().isEmpty()) return;
        CoreServiceLocator.getReproStepsProxy().addVisualUserStep(screenName, screenshot);
        // Fire an event that a CP screen has changed
        IBGSdkCoreEvent sdkCoreEvent = new IBGSdkCoreEvent.CrossPlatformScreenChanged(screenName);
        IBGCoreEventPublisher.post(sdkCoreEvent);
    }

    public void reportCurrentViewChange(String currentView) {
        if (currentView == null || currentView.trim().isEmpty()) return;
        CurrentViewProvider.getInstance().setCrossPlatformCurrentView(currentView);
    }

    public void addExperiments(List<String> experiments) {
        ServiceLocator.getExperimentsManager().addExperiments(experiments);
    }

    public void removeExperiments(List<String> experiments) {
        ServiceLocator.getExperimentsManager().removeExperiments(experiments);
    }

    public void clearAllExperiments() {
        ServiceLocator.getExperimentsManager().clearAllExperiments();
    }

    public void setFullscreen(boolean isFullscreen) {
        SettingsManager.getInstance().setFullScreen(isFullscreen);
    }

    public void addFeatureFLag(IBGFeatureFlag ibgFeatureFlag) {
        FeaturesFlagServiceLocator.getFeatureFlagsManager().add(ibgFeatureFlag);
    }

    public void addFeatureFLags(List<IBGFeatureFlag> ibgFeatureFlag) {
        FeaturesFlagServiceLocator.getFeatureFlagsManager().add(ibgFeatureFlag.toArray(new IBGFeatureFlag[0]));
    }

    public void removeFeatureFlag(List<String> keys) {
        String[] keysArray = keys.toArray(new String[0]);
        FeaturesFlagServiceLocator.getFeatureFlagsManager().remove(keysArray);
    }

    public void removeFeatureFlag(String key) {
        FeaturesFlagServiceLocator.getFeatureFlagsManager().remove(key);
    }

    public void removeAllFeatureFlags() {
        FeaturesFlagServiceLocator.getFeatureFlagsManager().removeAllFeaturesFlags();
    }

    public void captureScreenshot() {
        CoreServiceLocator.getReproStepsManualCaptor().duplicateCurrentParent();
    }

    public void setNetworkAutoMaskingState(Feature.State state) {
        IBGNetworkInterceptionConfigurationProvider configProvider = NetworkInterceptionServiceLocator.getConfigurationProvider();

        if (state.equals(Feature.State.ENABLED) && !configProvider.isAutoMaskingAvailable()) {
            InstabugSDKLogger.e(Constants.LOG_TAG, AUTO_MASKING_ENABLED_BE_FLAG_DISABLED);
        }

        configProvider.setAutoMaskingSDKEnabled(state == Feature.State.ENABLED);
    }

    public void addUserStep(long timestamp, @StepType String stepType, String message,@Nullable String label,String viewType) {
        CoreServiceLocator.getCpUserStepsHandler().addUserStep(timestamp, stepType, message, label, viewType);
    }
    public void shouldDisableNativeUserStepsCapturing(boolean shouldDisableUserSteps){
        SettingsManager.setShouldDisableNativeUserStepsCapturing(shouldDisableUserSteps);
    }
}
