package com.instabug.library;

import static com.instabug.library.Constants.LogPlaceHolders.EMPTY_EMAIL;
import static com.instabug.library.Constants.LogPlaceHolders.EMPTY_USERNAME;
import static com.instabug.library.Constants.LogPlaceHolders.NON_EMPTY_EMAIL;
import static com.instabug.library.Constants.LogPlaceHolders.NON_EMPTY_USERNAME;
import static com.instabug.library.InstabugColorTheme.InstabugColorThemeLight;
import static com.instabug.library.InstabugState.BUILDING;
import static com.instabug.library.InstabugState.DISABLED;
import static com.instabug.library.InstabugState.NOT_BUILT;

import android.annotation.SuppressLint;
import android.app.Application;
import android.content.Context;
import android.graphics.Bitmap;
import android.net.Uri;
import android.os.Build;
import android.view.View;

import androidx.annotation.ColorInt;
import androidx.annotation.DrawableRes;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.annotation.RestrictTo;
import androidx.annotation.VisibleForTesting;
import androidx.core.content.ContextCompat;

import com.instabug.library.apichecker.APIChecker;
import com.instabug.library.apichecker.ReturnableRunnable;
import com.instabug.library.apichecker.VoidRunnable;
import com.instabug.library.core.IBGStateEventBusSubscriber;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventPublisher;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent;
import com.instabug.library.core.plugin.PluginsManager;
import com.instabug.library.diagnostics.customtraces.IBGPendingTraceHandler;
import com.instabug.library.featuresflags.model.IBGFeatureFlag;
import com.instabug.library.firstseen.FirstSeenRequestFetcher;
import com.instabug.library.internal.contentprovider.InstabugApplicationProvider;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.utils.memory.IBGLowMemoryHandler;
import com.instabug.library.internal.video.customencoding.VideoEncoderConfig;
import com.instabug.library.invocation.InstabugInvocationEvent;
import com.instabug.library.invocation.InvocationManagerContract;
import com.instabug.library.logging.InstabugUserEventLogger;
import com.instabug.library.model.IBGTheme;
import com.instabug.library.model.Report;
import com.instabug.library.model.StepType;
import com.instabug.library.networkDiagnostics.model.NetworkDiagnosticsCallback;
import com.instabug.library.networkinterception.NetworkInterceptionServiceLocator;
import com.instabug.library.screenshot.ScreenshotProvider;
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator;
import com.instabug.library.sessioncontroller.SessionManualController;
import com.instabug.library.settings.InstabugMinimalPersistableSettings;
import com.instabug.library.settings.PerSessionSettings;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.ui.onboarding.WelcomeMessage;
import com.instabug.library.ui.onboarding.utils.WelcomeMessageConverter;
import com.instabug.library.user.UserEvent;
import com.instabug.library.user.UserManager;
import com.instabug.library.util.AppVariantHandler;
import com.instabug.library.util.BitmapUtils;
import com.instabug.library.util.CodePushVersionHandler;
import com.instabug.library.util.InstabugDeprecationLogger;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.StringUtility;
import com.instabug.library.util.TimeUtils;
import com.instabug.library.util.threading.PoolProvider;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.Callable;

/**
 * The core Instabug SDK class.
 * <h2>Logging</h2>
 * <p>The Instabug SDK can provide logs that may be helpful if you want to test out the
 * integration or trace the SDK process. To turn it on, just enable debugging by setting
 * <code>Instabug.DEBUG = true;</code>
 * <p>
 * For more info, please refer to
 *
 * @see <a href="https://docs.instabug.com">https://docs.instabug.com</a>
 */
@SuppressWarnings("UnusedDeclaration")
public class Instabug {

    @Nullable
    private volatile static Instabug INSTANCE = null;
    private final InstabugDelegate delegate;
    private static Context appContext;
    static final String REFLECTION_ERROR_MESSAGE = "something went wrong while calling by reflection class not found ";

    private Instabug(@NonNull InstabugDelegate delegate) {
        this.delegate = delegate;
    }


    /**
     * Private api used in CP by reflection, provide network diagnostics data of last active day in the callback
     * if failure rate is greater than or equal 70%
     */
    private static void setNetworkDiagnosticsCallback(@NonNull NetworkDiagnosticsCallback callback) {
        APIChecker.checkAndRunInExecutor("setNetworkDiagnosticsCallback", () -> {
            if (callback != null)
                CoreServiceLocator.getNetworkDiagnosticsManager().onCallbackProvided(callback);
        });
    }

    /**
     * Sets the printed logs priority filter to one of the following levels:
     * <p>
     * - {@link LogLevel#NONE} disables all SDK console logs.
     * <br/>
     * - {@link LogLevel#ERROR} prints errors and warnings, we use this level to let you know if something goes wrong.
     * This is also enabled by default
     * <br/>
     * - {@link LogLevel#DEBUG} use this in case you are debugging an issue.
     * Not recommended for production use.
     * <br/>
     * - {@link LogLevel#VERBOSE} use this only if {@link LogLevel#DEBUG} was not enough and you need more visibility
     * on what is going on under the hood.
     * Similar to the {@link LogLevel#DEBUG} level, this is not meant to be used on production environments.
     * <br/>
     * <p>
     * <br/>
     * More information can be found <a href="<https://docs.instabug.com/docs/debugging>">here</a>
     *
     * @param level the priority level.
     * @see LogLevel
     */
    public static void setSdkDebugLogsLevel(@LogLevel final int level) {
        APIChecker.checkAndRunInExecutor("setSdkDebugLogsLevel", () -> {
            SettingsManager.getInstance().setLogLevel(level);
        });
    }

    /**
     * Get the primary color that the SDK will use to tint certain UI elements in the SDK
     *
     * @see #setPrimaryColor(int)
     * @since 2.0
     */
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public static int getPrimaryColor() {
        return APIChecker.checkAndGet("Instabug.getPrimaryColor", () -> SettingsManager.getInstance().getPrimaryColor(), 0);
    }

    /**
     * Set the primary color that the SDK will use to tint certain UI elements in the SDK
     *
     * @param primaryColorValue The value of the primary color , whatever this color was parsed
     *                          from a resource color or hex color or RGB color values
     * @see #getPrimaryColor()
     * @since 2.0
     * @deprecated use {@link #setTheme(IBGTheme)} ()} instead.
     */
    @Deprecated
    public static void setPrimaryColor(@ColorInt final int primaryColorValue) {
        APIChecker.checkAndRunInExecutor("Instabug.setPrimaryColor", new VoidRunnable() {
            @Override
            public void run() {
                SettingsManager.getInstance().setPrimaryColor(primaryColorValue);
            }
        });
    }

    /**
     * @return current SDK {@link InstabugColorTheme}
     * @since 3.0
     */
    @Nullable
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public static InstabugColorTheme getTheme() {
        return APIChecker.checkAndGet("Instabug.getTheme", new ReturnableRunnable<InstabugColorTheme>() {
            @Override
            public InstabugColorTheme run() {
                return SettingsManager.getInstance().getTheme();
            }
        }, InstabugColorThemeLight);
    }

    /**
     * Adds extra file attachment to be uploaded along with upcoming reports.
     * By using this API you can attach more than one file up to 10 files
     * each file size up to 50MB, more than that will be ignored.
     * Notes:
     * 1) The file will be copied at the reporting time only,
     * So you can update the file whenever you want and you don't need to re-add it.
     * 2) This API will retain only the latest 10 files have been added.
     * 3) This API uses LinkedHashMap collection internally, So it will never accept repeated URIs.
     *
     * @param fileUri               Uri of desired file to be uploaded along upcoming reports
     *                              note:uri shouldn't be null, otherwise it will be ignored
     * @param fileNameWithExtension the file name with extension
     * @since 3.4.0
     */
    public static void addFileAttachment(@NonNull final Uri fileUri, @NonNull final String fileNameWithExtension) {
        APIChecker.checkAndRunInExecutor("Instabug.addFileAttachment", new VoidRunnable() {
            @Override
            public void run() {
                if (fileUri == null) {
                    InstabugSDKLogger.w(Constants.LOG_TAG, "fileUri object passed to Instabug.addFileAttachment() is null");
                    return;
                } else if (fileNameWithExtension == null) {
                    InstabugSDKLogger.w(Constants.LOG_TAG, "fileNameWithExtension passed to Instabug.addFileAttachment() is null");
                    return;
                }

                SettingsManager.getInstance().addExtraAttachmentFile(fileUri, fileNameWithExtension);
                InstabugSDKLogger.d(Constants.LOG_TAG, "addFileAttachment file uri: " + fileUri);
            }
        });
    }

    /**
     * Adds extra data as a file attachment to be uploaded along with upcoming reports.
     * By using this API you can attach more than one file up to 10 files
     * each file size up to 50MB, more than that will be ignored.
     * Notes:
     * This API save the data to new Uri then it adds it using the {@link Instabug#addFileAttachment}
     *
     * @param data                  byte[] of data wil be saved as a file to be uploaded along upcoming reports
     *                              note:data shouldn't be null, otherwise it will be ignored
     * @param fileNameWithExtension the file name with extension
     * @since 4.2.8
     */
    public static void addFileAttachment(@NonNull final byte[] data, @NonNull final String fileNameWithExtension) {
        APIChecker.checkAndRunInExecutor("Instabug.addFileAttachment", new VoidRunnable() {
            @Override
            public void run() {
                if (data == null) {
                    InstabugSDKLogger.w(Constants.LOG_TAG, "data object passed to Instabug.addFileAttachment() is null");
                    return;
                } else if (fileNameWithExtension == null) {
                    InstabugSDKLogger.w(Constants.LOG_TAG, "fileNameWithExtension  passed to Instabug.addFileAttachment() is null");
                    return;
                }
                SettingsManager.getInstance().addExtraAttachmentFile(data, fileNameWithExtension);
                InstabugSDKLogger.d(Constants.LOG_TAG, "addFileAttachment bytes");
            }
        });
    }

    /**
     * Clears all Uris of the attached files.
     * The URIs which added via {@link Instabug#addFileAttachment} API not the physical files.
     */
    public static void clearFileAttachment() {
        APIChecker.checkAndRunInExecutor("Instabug.clearFileAttachment", new VoidRunnable() {
            @Override
            public void run() {
                SettingsManager.getInstance().clearExtraAttachmentFiles();
                InstabugSDKLogger.d(Constants.LOG_TAG, "clearFileAttachment");
            }
        });
    }

    /**
     * @return user data added before using {@link #setUserData(String)}
     */
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public static String getUserData() {
        return APIChecker.checkAndGet("Instabug.getUserData", new ReturnableRunnable<String>() {
            @Override
            public String run() {
                return SettingsManager.getInstance().getUserData();
            }
        }, "");
    }

    /**
     * Adds specific user data that you need to be added to the reports
     *
     * @param userData string representing user data
     * @since 2.0
     */
    public static void setUserData(@NonNull final String userData) {
        APIChecker.checkAndRunInExecutor("Instabug.setUserData", new VoidRunnable() {
            @Override
            public void run() {
                if (InstabugFeaturesManager.getInstance().getFeatureState(IBGFeature.USER_DATA) == Feature
                        .State.ENABLED) {
                    SettingsManager.getInstance().setUserData(StringUtility.trimString(userData, 1000));
                }
            }
        });
    }

    /**
     * @return user's email entered using
     * {@link com.instabug.library.Instabug#identifyUser(String, String)} or in Instabug Feedback
     * form
     */
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public static String getUserEmail() {
        return APIChecker.checkAndGet("Instabug.getUserEmail", new ReturnableRunnable<String>() {
            @Override
            public String run() {
                return UserManager.getIdentifiedUserEmail();
            }
        }, "");
    }

    /**
     * @param callback {@link OnUserUUIDReadyCallback}
     *                 will be called after getting the uuid with the user UUID
     *                 it will be called in main thread
     *                 UUID will be null if the SDK wasn't built or SDK is disabled
     */
    public static void getUserUUID(@NonNull OnUserUUIDReadyCallback callback) {
        APIChecker.checkAndRunInExecutor("Instabug.getUserUUID", () -> {
            String uuid = UserManager.getUUID();
            PoolProvider.postMainThreadTask(() -> {
                try {
                    callback.onUserUUIDReady(uuid);
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while calling OnUserUUIDReadyCallback.onUserUUIDReady", e);
                }
            });
        });
    }

    /**
     * @return application token entered using
     * {@link com.instabug.library.Instabug.Builder#Builder(Application, String)}
     * @see com.instabug.library.Instabug.Builder#Builder(Application, String,
     * InstabugInvocationEvent[])
     */
    @Nullable
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public static String getAppToken() {
        return SettingsManager.getInstance().getAppToken();
    }

    /**
     * Set the user identity.
     * Instabug will pre-fill the user email in reports.
     *
     * @param username Username.
     * @param email    User's default email
     * @since 4.0.0
     * @deprecated {@link #identifyUser(String userName, String email, String id)} should be used instead.
     */
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    @Deprecated
    public static void identifyUser(@Nullable final String username, @NonNull final String email) {
        APIChecker.checkAndRunInExecutor("Instabug.identifyUser", () -> {
            UserManager.identifyUser(username, email, null, false);
            String usernameLog = username == null || username.isEmpty() ? EMPTY_USERNAME : NON_EMPTY_USERNAME;
            String emailLog = email == null || email.isEmpty() ? EMPTY_EMAIL : NON_EMPTY_EMAIL;
            InstabugSDKLogger.i(Constants.LOG_TAG, "identifyUser username: " + usernameLog + " email: " + emailLog);
        });
    }

    /**
     * Set the user identity.
     * Instabug will pre-fill the user email in reports.
     *
     * @param userName Username. (optional)
     * @param email    User's default email (optional)
     * @param id       User's Id (optional)
     */
    public static void identifyUser(@Nullable final String userName, @Nullable final String email, @Nullable String id) {
        APIChecker.checkAndRunInExecutor("Instabug.identifyUser", () -> {
            UserManager.identifyUser(userName, email, id);

        });
    }

    /**
     * @return Holder application of Instabug SDK
     */
    @Nullable
    public static Context getApplicationContext() {
        if (appContext != null) {
            return appContext;
        }
        InstabugApplicationProvider applicationProvider = InstabugApplicationProvider.getInstance();
        if (applicationProvider != null) {
            return applicationProvider.getApplication();
        }
        return null;
    }

    /**
     * Returns the current singleton instance of this class
     *
     * @return singleton instance of Instabug
     */
    @Nullable
    private static Instabug getInstance() {
        InstabugApplicationProvider applicationProvider = InstabugApplicationProvider.getInstance();
        if (INSTANCE == null && applicationProvider != null) {
            InstabugDelegate delegate = InstabugDelegate.getInstance(applicationProvider.getApplication());
            INSTANCE = new Instabug(delegate);
        }
        return INSTANCE;
    }

    /**
     * Logout user.
     *
     * @since 4.0.0
     */
    public static void logoutUser() {
        APIChecker.checkAndRunInExecutor("Instabug.logoutUser", () -> {
            UserManager.logoutUser(true);
            InstabugSDKLogger.d(Constants.LOG_TAG, "logoutUser");
        });
    }

    /**
     * @return {@code true} if Instabug.Builder().build() was called Instabug, {@code false} if not
     * @see {@link Builder#build()
     */
    public static boolean isBuilt() {
        return INSTANCE != null && InstabugStateProvider.getInstance().getState() != NOT_BUILT
                && InstabugStateProvider.getInstance().getState() != BUILDING;
    }

    /**
     * @return {@code true} if Instabug is enabled, {@code false} if it's disabled
     * @throws IllegalStateException if Instabug object wasn't built using {@link
     *                               Instabug.Builder#build()} before this method was called
     *                               * @see #enable()
     * @see #disable()
     */
    public static boolean isEnabled() {
        if (!Instabug.isBuilt())
            return false;
        return InstabugFeaturesManager.getInstance().isFeatureAvailable(IBGFeature.INSTABUG)
                && InstabugFeaturesManager.getInstance().getFeatureState(IBGFeature.INSTABUG) == Feature.State.ENABLED;
    }

    /**
     * Enables all Instabug functionality
     */
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public synchronized static void enable() {

        if (IBGLowMemoryHandler.isSDKPausedOnLowMemory()) {
            return;
        }

        PoolProvider.getApiExecutor().execute(new Runnable() {
            @Override
            public void run() {
                APIChecker.checkBuilt("Instabug.enable", new VoidRunnable() {
                    @Override
                    public void run() {
                        Instabug instance = getInstance();
                        if (instance != null) {
                            instance.delegate.updateInstabugFeatureState(Feature.State.ENABLED);
                            instance.delegate.setInstabugState(InstabugState.ENABLED);
                            instance.delegate.start();
                        }
                        InstabugSDKLogger.d(Constants.LOG_TAG, "enable");
                    }
                });
            }
        });
    }

    /**
     * Disables all Instabug functionality
     */
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public synchronized static void disable() {
        APIChecker.checkAndRunInExecutor("Instabug.disable", new VoidRunnable() {
            @Override
            public void run() {
                if (getInstance() != null) {
                    getInstance().delegate.stopSdk();
                }
                InstabugSDKLogger.d(Constants.LOG_TAG, "disable");
            }
        });
    }

    /**
     * Internal Use Only
     * Pauses all Instabug functionality temp.
     */
    public static void pauseSdk() {
        APIChecker.checkAndRunInExecutor("Instabug.pauseSdk", new VoidRunnable() {
            @Override
            public void run() {
                if (getInstance() != null) {
                    getInstance().delegate.pauseSdk();
                }
                InstabugSDKLogger.d(Constants.LOG_TAG, "pauseSdk");
            }
        });
    }

    /**
     * Internal Use Only
     * Resume instabug state.
     */
    public static void resumeSdk() {
        APIChecker.checkAndRunInExecutor("Instabug.resumeSdk", new VoidRunnable() {
            @Override
            public void run() {
                if (getInstance() != null) {
                    getInstance().delegate.resumeSdk();
                }
                InstabugSDKLogger.d(Constants.LOG_TAG, "resumeSdk");
            }
        });
    }

    /**
     * Change Locale of Instabug UI elements
     *
     * @param locale new locale to use with Instabug (defaults to English)
     */
    public static void setLocale(@NonNull final Locale locale) {
        APIChecker.checkAndRunInExecutor("Instabug.setLocale", new VoidRunnable() {
            @Override
            public void run() {
                if (locale == null) {
                    InstabugSDKLogger.w(Constants.LOG_TAG, "locale object passed to Instabug.setLocale is null");
                    return;
                }
                if (getInstance() != null) {
                    getInstance().delegate.setLocale(locale);
                }
            }
        });
    }

    /**
     * Get currently used Locale of Instabug
     *
     * @return locale used in Instabug
     */
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public static Locale getLocale(@Nullable final Context context) {
        return APIChecker.checkAndGet("Instabug.getLocale", new ReturnableRunnable<Locale>() {
            @Nullable
            @Override
            public Locale run() {
                return SettingsManager.getInstance().getInstabugLocale(context);
            }
        }, Locale.getDefault());
    }

    /**
     * Adds tag(s) to issues before sending them
     *
     * @param tags to be added to issues uploaded from now on till it's reset using
     *             {@link #resetTags()}
     * @see #resetTags()
     */
    public static void addTags(@NonNull final String... tags) {
        APIChecker.checkAndRunInExecutor("Instabug.addTags", new VoidRunnable() {
            @Override
            public void run() {
                SettingsManager.getInstance().addTags(StringUtility.trimStrings(tags));
            }
        });
    }

    /**
     * @return all tags added using {@link #addTags(String...)}
     * @see #resetTags()
     */
    @Nullable
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public static ArrayList<String> getTags() {
        return APIChecker.checkAndGet("Instabug.getTags", new ReturnableRunnable<ArrayList<String>>() {
            @Override
            @Nullable
            public ArrayList<String> run() {
                return SettingsManager.getInstance().getTags();
            }
        }, null);
    }

    /**
     * Reset ALL tags added using {@link #addTags(String...)}
     */
    public static void resetTags() {
        APIChecker.checkAndRunInExecutor("Instabug.resetTags", new VoidRunnable() {
            @Override
            public void run() {
                SettingsManager.getInstance().resetTags();
            }
        });
    }

    /**
     * Allows replacing all the text displayed in instabug with a custom text
     *
     * @param instabugCustomTextPlaceHolder {@link InstabugCustomTextPlaceHolder}
     * @since v3
     */
    public static void setCustomTextPlaceHolders(@NonNull final InstabugCustomTextPlaceHolder instabugCustomTextPlaceHolder) {
        APIChecker.checkAndRunInExecutor("Instabug.setCustomTextPlaceHolders", new VoidRunnable() {
            @Override
            public void run() {
                if (instabugCustomTextPlaceHolder == null) {
                    InstabugSDKLogger.w(Constants.LOG_TAG, "instabugCustomTextPlaceHolder object passed to Instabug.setCustomTextPlaceHolders() is null");
                    return;
                }
                SettingsManager.getInstance().setCustomPlaceHolders(instabugCustomTextPlaceHolder);
            }
        });
    }

    /**
     * Gets all saved user attributes.
     *
     * @return all user attributes as HashMap<String, String>
     * @since 3.2.0
     */
    @Nullable
    public static HashMap<String, String> getAllUserAttributes() {
        return APIChecker.checkAndGet("Instabug.getAllUserAttributes", new ReturnableRunnable<HashMap<String, String>>() {
            @Override
            public HashMap<String, String> run() {
                if (getInstance() != null) {
                    return getInstance().delegate.getAllUserAttributes();
                } else {
                    return new HashMap<>();
                }
            }
        }, null);
    }

    /**
     * Sets user attribute to overwrite it's value or create a new one if it doesn't exist.
     *
     * @param key   the attribute
     * @param value the value
     * @since 3.2.0
     */
    public static void setUserAttribute(@NonNull final String key, @NonNull final String value) {
        APIChecker.checkAndRunInExecutor("Instabug.setUserAttribute", new VoidRunnable() {
            @Override
            public void run() {
                if (getInstance() != null) {
                    getInstance().delegate.setUserAttribute(key, value);
                }
                InstabugSDKLogger.d(Constants.LOG_TAG, "setUserAttribute");
            }
        });
    }

    /**
     * Gets specific user attribute.
     *
     * @param key the attribute key as string
     * @return the desired user attribute
     * @see #setUserAttribute(String, String)
     * @since 3.2.0
     */
    @Nullable
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public static String getUserAttribute(@NonNull final String key) {
        return APIChecker.checkAndGet("Instabug.getUserAttribute", new ReturnableRunnable<String>() {
            @Override
            @Nullable
            public String run() {
                if (getInstance() != null) {
                    return getInstance().delegate.getUserAttribute(key);
                } else {
                    return null;
                }
            }
        }, null);
    }

    /**
     * Removes user attribute if exists.
     *
     * @param key the attribute key as string
     * @see #setUserAttribute(String, String)
     * @since 3.2.0
     */
    public static void removeUserAttribute(@NonNull final String key) {
        APIChecker.checkAndRunInExecutor("Instabug.removeUserAttribute", new VoidRunnable() {
            @Override
            public void run() {
                if (getInstance() != null) {
                    getInstance().delegate.removeUserAttribute(key);
                }
                InstabugSDKLogger.d(Constants.LOG_TAG, "removeUserAttribute");
            }
        });
    }

    /**
     * Clears all user attributes if exists.
     *
     * @since 3.0
     */
    public static void clearAllUserAttributes() {
        APIChecker.checkAndRunInExecutor("Instabug.clearAllUserAttributes", new VoidRunnable() {
            @Override
            public void run() {
                if (getInstance() != null) {
                    getInstance().delegate.clearAllUserAttributes();
                }
                InstabugSDKLogger.d(Constants.LOG_TAG, "clearAllUserAttributes");
            }
        });
    }

    public static Date getFirstRunAt() {
        return SettingsManager.getInstance().getFirstRunAt();
    }

    public static boolean isAppOnForeground() {
        return SettingsManager.getInstance().isAppOnForeground();
    }

    /**
     * Sets the welcome message mode to live, beta or disabled.
     * <p>
     * By default, the welcome message live mode is enabled. and it appears automatically after
     * 30 seconds from the user's first session. You can change it to the beta mode or disable it.
     * <p>
     * The live mode consists of one step to inform the users how to report a bug or feedback.
     * The beta mode consists of three steps to welcome your testers on board, inform them how to
     * report a bug or feedback and to motivate them to always be on the latest app version.
     * <p>
     * Please note, the into message appears only if the invocation event isn't set to none.
     *
     * @param welcomeMessageState An enum to set the welcome message state to live, beta or disabled.
     * @since 4.14.0
     */
    public static void setWelcomeMessageState(@NonNull final WelcomeMessage.State welcomeMessageState) {
        APIChecker.checkAndRunInExecutor("Instabug.setWelcomeMessageState", () -> {
            if (welcomeMessageState == null) {
                InstabugSDKLogger.w(Constants.LOG_TAG, "welcomeMessageState object passed to Instabug.setWelcomeMessageState() is null");
                return;
            }
            PoolProvider.postIOTaskWithCheck(() -> {
                try {
                    int converterState = WelcomeMessageConverter.welcomeMessageConverterState(welcomeMessageState);
                    String bugString = "com.instabug.bug.BugReporting";
                    Method setWelcomeMessageState = Class.forName(bugString).getDeclaredMethod("setWelcomeMessageState", int.class);
                    setWelcomeMessageState.setAccessible(true);
                    setWelcomeMessageState.invoke(Class.forName(bugString), converterState);
                    setWelcomeMessageState.setAccessible(false);
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, REFLECTION_ERROR_MESSAGE + e.getMessage());
                }
            });
        });
    }

    /**
     * Shows the welcome message in a specific mode.
     * <p>
     * By default, the welcome message live mode is enabled. and it appears automatically after
     * 30 seconds from the user's first session. You can change it to the beta mode or disable it.
     * <p>
     * The live mode consists of one step to inform the users how to report a bug or feedback.
     * The beta mode consists of three steps to welcome your testers on board, inform them how to
     * report a bug or feedback and to motivate them to always be on the latest app version.
     * <p>
     * Please note, the into message appears only if the invocation event isn't set to none.
     *
     * @param welcomeMessageState An enum to set the welcome message state to live, beta or disabled.
     * @since 4.14.0
     */
    public static void showWelcomeMessage(@NonNull final WelcomeMessage.State welcomeMessageState) {
        APIChecker.checkAndRunInExecutor("Instabug.showWelcomeMessage", () -> {
            if (welcomeMessageState == null) {
                InstabugSDKLogger.w(Constants.LOG_TAG, "welcomeMessageState object passed to Instabug.showWelcomeMessage() is null");
                return;
            }
            if (!InstabugCore.isForegroundBusy()) {
                try {
                    int converterState = WelcomeMessageConverter.welcomeMessageConverterState(welcomeMessageState);
                    String bugString = "com.instabug.bug.BugReporting";
                    Method showWelcomeMessageMethod = Class.forName(bugString).getDeclaredMethod("showWelcomeMessage", int.class);
                    showWelcomeMessageMethod.setAccessible(true);
                    showWelcomeMessageMethod.invoke(Class.forName(bugString), converterState);
                    showWelcomeMessageMethod.setAccessible(false);
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, REFLECTION_ERROR_MESSAGE + e.getMessage());
                }
            }
            InstabugSDKLogger.i(Constants.LOG_TAG, "showWelcomeMessage: " + welcomeMessageState.name());
        });
    }


    /**
     * Enable/disable session profiler
     *
     * @param state desired state of the session profiler feature
     * @see com.instabug.library.Feature.State
     */
    public static void setSessionProfilerState(@NonNull final Feature.State state) {
        APIChecker.checkAndRunInExecutor("Instabug.setSessionProfilerState", new VoidRunnable() {
            @Override
            public void run() {
                if (state == null) {
                    InstabugSDKLogger.w(Constants.LOG_TAG, "state object passed to Instabug.setSessionProfilerState() is null");
                    return;
                }

                if (getInstance() != null) {
                    getInstance().delegate.setSessionProfilerState(state);
                }
                InstabugSDKLogger.d(Constants.LOG_TAG, "setSessionProfilerState: " + state.name());
            }
        });
    }

    /**
     * Enable/disable automatic user steps tracking
     *
     * @param state desired state of automatic user step tracking feature
     * @see com.instabug.library.Feature.State
     */
    public static void setTrackingUserStepsState(@NonNull final Feature.State state) {
        APIChecker.checkAndRunInExecutor("Instabug.setTrackingUserStepsState", new VoidRunnable() {
            @Override
            public void run() {
                if (state == null) {
                    InstabugSDKLogger.w(Constants.LOG_TAG, "state object passed to Instabug.setTrackingUserStepsState() is null");
                    return;
                }

                InstabugFeaturesManager.getInstance().setFeatureState(IBGFeature.TRACK_USER_STEPS, state);
                InstabugSDKLogger.d(Constants.LOG_TAG, "setTrackingUserStepsState: " + state.name());
            }
        });
    }

    /**
     * Sets a {@link ReproConfigurations} controlling repro steps with each type of report
     *
     * @param configurations the configurations object to be set and used by Instabug.
     * @see ReproConfigurations.Builder
     */
    public static void setReproConfigurations(ReproConfigurations configurations) {
        APIChecker.checkAndRunInExecutor("Instabug.setReproConfigurations", () -> {
            updateAndBroadcastReproConfigurations(configurations);
        });
    }

    /**
     * Set which color theme to use for the SDK's UI
     *
     * @param instabugTheme color theme to apply on the SDK
     * @since 8.0.0
     */
    public static void setColorTheme(@NonNull final InstabugColorTheme instabugTheme) {
        APIChecker.checkAndRunInExecutor("Instabug.setColorTheme", new VoidRunnable() {
            @Override
            public void run() {
                SettingsManager.getInstance().setTheme(instabugTheme);

                switch (instabugTheme) {
                    case InstabugColorThemeDark:
                        SettingsManager.getInstance().setPrimaryColor(PerSessionSettings
                                .PRIMARY_COLOR_DARK);
                        SettingsManager.getInstance().setStatusBarColor(PerSessionSettings.STATUS_BAR_DARK);

                        break;
                    case InstabugColorThemeLight:
                        SettingsManager.getInstance().setPrimaryColor(PerSessionSettings
                                .PRIMARY_COLOR_LIGHT);
                        SettingsManager.getInstance().setStatusBarColor(PerSessionSettings
                                .STATUS_BAR_LIGHT);

                        break;
                }
            }
        });
    }

    /**
     * Set a custom brand logo image that could be displayed at the bottom of Instabug screens
     * when the {@link IBGFeature.WHITE_LABELING} is enabled.
     *
     * @param lightLogoVariant that will be used when the light theme is enabled.
     * @param darkLogoVariant  that will be used when the dark theme is enabled.
     * @since 11.6.0
     */
    public static void setCustomBrandingImage(@DrawableRes int lightLogoVariant, @DrawableRes int darkLogoVariant) {
        APIChecker.checkAndRunInExecutor("Instabug.setCustomBrandingImage", () -> {
            Bitmap lightLogoVariantBitmap = BitmapUtils.drawableToBitmap(
                    ContextCompat.getDrawable(appContext, lightLogoVariant)
            );

            Bitmap darkLogoVariantBitmap = BitmapUtils.drawableToBitmap(
                    ContextCompat.getDrawable(appContext, darkLogoVariant)
            );
            setCustomBrandingImage(lightLogoVariantBitmap, darkLogoVariantBitmap);
        });
    }

    /**
     * Set a custom brand logo image that could be displayed at the bottom of Instabug screens
     * when the {@link IBGFeature.WHITE_LABELING} is enabled.
     *
     * @param lightLogoVariant that will be used when the light theme is enabled.
     * @param darkLogoVariant  that will be used when the dark theme is enabled.
     * @since 11.6.0
     */
    private static void setCustomBrandingImage(Bitmap lightLogoVariant, Bitmap darkLogoVariant) {
        APIChecker.checkAndRunInExecutor("Instabug.setCustomBrandingImage", () -> {
            SettingsManager.getInstance().setLightCustomBrandingLogo(lightLogoVariant);
            SettingsManager.getInstance().setDarkCustomBrandingLogo(darkLogoVariant);
        });
    }

    /**
     * log @{@link UserEvent} associated with the current user
     *
     * @param eventIdentifier identifier to identify this UserEvent
     * @see UserEvent
     * @see Instabug.Builder#build()
     * @since 8.0.0
     */
    public static void logUserEvent(@NonNull final String eventIdentifier) {
        APIChecker.checkAndRunInExecutor("Instabug.logUserEvent", new VoidRunnable() {
            @Override
            public void run() {
                InstabugUserEventLogger.getInstance().logUserEvent(eventIdentifier);
                InstabugSDKLogger.d(Constants.LOG_TAG, "logUserEvent: " + eventIdentifier);
            }
        });
    }

    /**
     * Sets the onReportSubmitHandler that calls
     * {@link Report.OnReportCreatedListener#onReportCreated(Report)}
     * just after creating a bug report or a crash report<br/>
     * Please note that you can add up to 10 Tags, User Attributes, Console Logs and 3 Files.
     * WARNING: This runs on a background thread. Long operations will delay sending the report.
     *
     * @since 8.0.0
     */
    public static void onReportSubmitHandler(final Report.OnReportCreatedListener listener) {
        APIChecker.checkAndRunInExecutor("Instabug.onReportSubmitHandler", new VoidRunnable() {
            @Override
            public void run() {
                SettingsManager.getInstance().setOnReportCreatedListener(listener);
            }
        });
    }

    /**
     * FOR INTERNAL USE ONLY
     *
     * @param isASRAudioEnabled enable of disable audio in Auto screen recording
     */
    @VisibleForTesting
    static void setAutoScreenRecordingAudioCapturingEnabled(final Feature.State isASRAudioEnabled) {
        APIChecker.checkAndRunInExecutor("Instabug.setAutoScreenRecordingAudioCapturingEnabled", new VoidRunnable() {
            @Override
            public void run() {
                if (isASRAudioEnabled != null) {
                    SettingsManager.getInstance().setAutoScreenRecordingAudioCapturingState(isASRAudioEnabled);
                    InstabugSDKLogger.d(Constants.LOG_TAG, "setAutoScreenRecordingAudioCapturingEnabled: " + isASRAudioEnabled.name());
                }
                InstabugSDKLogger.w(Constants.LOG_TAG, "isASRAudioEnabled object passed to Instabug.setAutoScreenRecordingAudioCapturingEnabled() is null");
                return;
            }
        });
    }

    /**
     * Shows Instabug Prompt Options.
     * By default, it contains Report a problem, Suggest an improvement, Ask a question,
     * and a button that navigates to the chats list.
     * To control which options should be enabled, see BugReporting.setState(), Chats.setState(),
     * and Replies.setState().
     *
     * @see Instabug.Builder#build()
     */
    public static void show() {
        APIChecker.checkAndRunInExecutor("Instabug.show", new VoidRunnable() {
            @Override
            public void run() {
                InvocationManagerContract contract = CoreServiceLocator.getInvocationManagerContract();
                if (contract != null) {
                    contract.show();
                }
            }
        });
    }

    /**
     * Marks sensitive views as private.
     *
     * @param views the sensitive views to be masked.
     */
    public static void addPrivateViews(@NonNull final View... views) {
        APIChecker.checkAndRunInExecutor("Instabug.addPrivateViews", new VoidRunnable() {
            @Override
            public void run() {
                if (getInstance() != null && getInstance().delegate != null) {
                    getInstance().delegate.addPrivateViews(views);
                }
            }
        });
    }

    public static void removePrivateViews(@NonNull final View... views) {
        APIChecker.checkAndRunInExecutor("Instabug.removePrivateViews", new VoidRunnable() {
            @Override
            public void run() {
                if (getInstance() != null && getInstance().delegate != null) {
                    getInstance().delegate.removePrivateViews(views);
                }
            }
        });
    }

    /**
     * Removes all added private views
     */
    public static void removeAllPrivateViews() {
        APIChecker.checkAndRunInExecutor("Instabug.removeAllPrivateViews", () -> {
            Instabug instance = getInstance();
            if (instance != null && instance.delegate != null) {
                instance.delegate.removeAllPrivateViews();
            }
        });
    }

    public static boolean isBuilding() {
        return InstabugStateProvider.getInstance().getState() == BUILDING;
    }

    /**
     * For internal SDK Use only.
     * Used to totally disable the SDK and all features in the current app instance.
     * <p>
     * Only used now for NDK .so file not found issue.
     */
    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    private static void disableInternal() {
        APIChecker.checkAndRunInExecutor("Instabug.disableInternal", new VoidRunnable() {
            @Override
            public void run() {
                if (getInstance() != null) {
                    getInstance().delegate.stopSdk();
                    InstabugSDKLogger.d(Constants.LOG_TAG, "disableInternal");
                }
            }
        });
    }

    /**
     * This API is used for reporting any screen change, basically this api is used for RN
     *
     * @param screenshot optional bitmap if found
     * @param screenName mandatory screen name
     */
    private static void reportScreenChange(@Nullable Bitmap screenshot, @NonNull String screenName) {
        if (!Instabug.isBuilt()) {
            return;
        }
        if (getInstance() == null || getInstance().delegate == null) {
            return;
        }

        getInstance().delegate.reportScreenChange(screenshot, screenName);
    }

    /**
     * This API is used for reporting current view, basically used RN side (CP)
     */
    private static void reportCurrentViewChange(@NonNull String currentView) {
        if (!Instabug.isBuilt()) {
            return;
        }
        if (getInstance() == null || getInstance().delegate == null) {
            return;
        }

        getInstance().delegate.reportCurrentViewChange(currentView);
    }
    /**
     * This API checks whether a key from the CP side indicates that native user steps capturing should be disabled
     * @param shouldDisable
     */
    private static void shouldDisableNativeUserStepsCapturing(boolean shouldDisable) {
        APIChecker.checkAndRunInExecutor("Instabug.shouldDisableNativeUserStepsCapturing", () -> {
            if (getInstance() != null && getInstance().delegate != null){
                getInstance().delegate.shouldDisableNativeUserStepsCapturing(shouldDisable);
            }
        });
    }
    /**
     * Used by CP Team
     *
     * @param platform
     */
    private static void setCurrentPlatform(@Platform int platform) {
        SettingsManager.getInstance().setCurrentPlatform(platform);
    }

    /**
     * Api used by flutter to report user step
     *
     * @param timestamp time of the user step
     * @param stepType type of user step
     * @param message message to describe the step
     * @param label text on the view
     * @param viewType type of the view
     */
    private static void addUserStep(long timestamp, @StepType String stepType, String message,@Nullable String label,String viewType) {
        APIChecker.checkAndRun("addUserStep", () -> {
            if (getInstance() == null || getInstance().delegate == null) {
                return;
            }

            getInstance().delegate.addUserStep(timestamp, stepType, message, label, viewType);
        });
    }

    /**
     * Adds experiments to sent with reports
     *
     * @param experiments list of experiments to be added
     * @deprecated use {@link #addFeatureFlag(IBGFeatureFlag ibgFeatureFlag)} or
     * {@link #addFeatureFlags(List ibgFeatureFlag)} instead.
     */
    @Deprecated
    public static void addExperiments(@NonNull final List<String> experiments) {

        APIChecker.checkAndRunInExecutor("Instabug.addExperiments", new VoidRunnable() {
            @Override
            public void run() throws Exception {
                Instabug instance = getInstance();
                if (instance != null) {
                    instance.delegate.addExperiments(experiments);
                }
            }
        });
    }

    /**
     * Removes previously added experiments
     *
     * @param experiments list of experiments to be removed
     * @see Instabug#addExperiments
     * @deprecated use {@link #removeFeatureFlag(List keys)} instead.
     */
    @Deprecated
    public static void removeExperiments(@NonNull final List<String> experiments) {
        APIChecker.checkAndRunInExecutor("Instabug.removeExperiments", new VoidRunnable() {
            @Override
            public void run() throws Exception {
                Instabug instance = getInstance();
                if (instance != null) {
                    instance.delegate.removeExperiments(experiments);
                }
            }
        });
    }

    /**
     * Removes all previously added experiments
     *
     * @see Instabug#addExperiments
     * @deprecated use {@link #removeAllFeatureFlags()} instead.
     */
    @Deprecated
    public static void clearAllExperiments() {
        APIChecker.checkAndRunInExecutor("Instabug.clearAllExperiments", new VoidRunnable() {
            @Override
            public void run() throws Exception {
                Instabug instance = getInstance();
                if (instance != null) {
                    instance.delegate.clearAllExperiments();
                }
            }
        });
    }

    /**
     * @deprecated API and moved to Bug-reporting module
     */
    @Deprecated
    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    public static void setVideoEncoderConfig(@NonNull final VideoEncoderConfig config) {
        try {
            Class<?> bugSettingsClass = Class.forName("com.instabug.bug.settings.BugSettings");
            Method getInstanceMethod = bugSettingsClass.getDeclaredMethod("getInstance");
            Object bugSettingsInstance = getInstanceMethod.invoke(null);
            Method setVideoEncoderConfigMethod = bugSettingsClass.getDeclaredMethod("setVideoEncoderConfig", VideoEncoderConfig.class);
            setVideoEncoderConfigMethod.invoke(bugSettingsInstance, config);
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, REFLECTION_ERROR_MESSAGE + e.getMessage());
        }
    }

    /**
     * Provides a {@link Callable}<{@link Bitmap}> to capture screenshots externally. <br/>
     * Should be only called from a cross-platform SDK (e.g. React Native, Flutter…etc)
     *
     * @param provider when called, returns a bitmap of the screenshot taken
     */
    public static void setScreenshotProvider(@NonNull Callable<Bitmap> provider) {
        APIChecker.checkAndRunInExecutor("Instabug.setScreenshotProvider", () -> {
            if (SettingsManager.getInstance().getCurrentPlatform() == Platform.ANDROID) {
                InstabugSDKLogger.e(Constants.LOG_TAG,
                        "IllegalState in Instabug.setScreenshotProvider",
                        new IllegalStateException("setScreenshotProvider should not be called from the native Android SDK"));
                return;
            }
            ScreenshotProvider.setScreenshotProvider(provider);
        });
    }

    /**
     * Start a new Instabug SDK session manually if the manual session control feature
     * is enabled for your plan and no running sessions.
     * <p>
     * Not publicly available.
     */
    private static void startSession() {
        APIChecker.checkAndRunInExecutor(
                "Instabug.startSessionManually", SessionManualController::startSession
        );
    }

    /**
     * Stop the running Instabug SDK session manually if the manual session control feature
     * is enabled for your plan and there is a running session.
     * <p>
     * Not publicly available.
     */
    private static void stopSession() {
        APIChecker.checkAndRunInExecutor(
                "Instabug.stopSessionManually", SessionManualController::stopSession
        );
    }

    /**
     * an API to auto-mask screenshots.
     *
     * @param types are the masking types on screenshots
     *              by default masking type is {@link MaskingType#MASK_NOTHING}
     * @see MaskingType
     */
    public static void setAutoMaskScreenshotsTypes(@MaskingType int... types) {
        APIChecker.checkAndRunInExecutor("Instabug.setAutoMaskScreenshotsTypes", () ->
                CoreServiceLocator
                        .getUserMaskingFilterProvider()
                        .setAutoMaskScreenshotsTypes(types)
        );
    }

    /**
     * Sets the SDK views to be displayed in fullscreen mode
     * when SDK is in fullscreen mode, status bar and navigation bar will not be displayed when an SDK view appears
     *
     * @param isFullscreen boolean to set the views to be displayed in fullscreen mode
     */
    public static void setFullscreen(boolean isFullscreen) {
        APIChecker.checkAndRunInExecutor("Instabug.setFullscreen", () -> {
            Instabug instance = getInstance();
            if (instance != null) {
                InstabugDelegate delegate = instance.delegate;
                delegate.setFullscreen(isFullscreen);
            }
        });
    }


    /**
     * API to be called from the app when it navigates the user to the play store for app rating.
     */
    public static void willRedirectToStore() {
        long currentTime = TimeUtils.currentTimeMillis();
        APIChecker.checkAndRunInExecutor("Instabug.willRedirectToStore",
                () -> IBGSessionServiceLocator
                        .getManualRatingDetector()
                        .willRedirectToTheStoreAt(currentTime));
    }

    /**
     * Sets Code Push version to be used for all reports.
     * Note: Should only be set for React Native apps
     *
     * @param codePushVersion the Code Push version to work with.
     */
    public static void setCodePushVersion(@Nullable String codePushVersion) {
        APIChecker.checkAndRunInExecutor("Instabug.setCodePushVersion", () -> {
            CodePushVersionHandler.setVersion(codePushVersion, true);
            FirstSeenRequestFetcher.getInstance().fetchFirstSeenRequest(true);
            IBGCoreEventPublisher.post(IBGSdkCoreEvent.CodePushVersionChanged.INSTANCE);


        });
    }

    /**
     * Sets a {@link IBGTheme} used to customize Instabug's ui elements
     *
     * @param theme the theme to be set and used by Instabug.
     * @see IBGTheme.Builder
     */
    public static void setTheme(@Nullable IBGTheme theme) {
        APIChecker.checkAndRunInExecutor("Instabug.setTheme", () ->
                SettingsManager.getInstance().setIBGTheme(theme));
    }


    /**
     * Enable/Disable network logs auto masking
     *
     * @param state desired state of network logs auto masking feature
     */
    public static void setNetworkAutoMaskingState(Feature.State state) {
        APIChecker.checkAndRunInExecutor("Instabug.setNetworkAutoMaskingState", () -> {
            Instabug instance = getInstance();
            if (instance != null) {
                InstabugDelegate delegate = instance.delegate;
                delegate.setNetworkAutoMaskingState(state);
            }
        });
    }
    /**
     * Enable/Disable network Body logs
     *
     * @param shouldEnableBody desired state of disable or enable capturing network body logs from SDK
     */
    public static void setNetworkLogBodyEnabled(boolean shouldEnableBody) {
        NetworkInterceptionServiceLocator.getConfigurationProvider().setNetworkLogBodyEnabledBySDK(shouldEnableBody);
    }
    /**
     * Sets App Variant to be used for all reports.
     * Should send with all requests headers
     *
     * @param appVariant the app variant version to work with.
     */
    public static void setAppVariant(@Nullable String appVariant){
        APIChecker.checkAndRunInExecutor("Instabug.setAppVariant", () -> {
            if(Instabug.isBuilt() && Instabug.isEnabled()){
                InstabugSDKLogger.w(Constants.LOG_TAG,"The AppVariant API is called after SDK initialization which is not recommended");
            }});
        String results = AppVariantHandler.sanitizeAppVariant(appVariant);
        if (results != null) SettingsManager.getInstance().setAppVariant(results);
    }

    public static class Builder {
        private String applicationToken;
        private Context applicationContext;
        @Nullable
        private Application application;
        private int instabugStatusBarColor = PerSessionSettings.STATUS_BAR_LIGHT;
        private InstabugInvocationEvent[] instabugInvocationEvents;

        private Feature.State userDataState = InstabugFeaturesManager.DEFAULT_FEATURE_STATE;
        private Feature.State consoleLogState = InstabugFeaturesManager.DEFAULT_FEATURE_STATE;
        private Feature.State instabugLogState = InstabugFeaturesManager.DEFAULT_FEATURE_STATE;
        private Feature.State inAppMessagingState = InstabugFeaturesManager.DEFAULT_FEATURE_STATE;
        private Feature.State pushNotificationState = InstabugFeaturesManager.DEFAULT_FEATURE_STATE;
        private Feature.State trackingUserStepsState = InstabugFeaturesManager
                .DEFAULT_FEATURE_STATE;
        private Feature.State viewHierarchyState = Feature.State.DISABLED;
        private Feature.State surveysState = InstabugFeaturesManager.DEFAULT_FEATURE_STATE;
        private Feature.State userEventsState = InstabugFeaturesManager.DEFAULT_FEATURE_STATE;
        private Feature.State anrDefaultState = InstabugFeaturesManager.DEFAULT_FEATURE_STATE;

        private boolean emailFieldRequired = true;
        private boolean emailFieldVisibility = true;
        private boolean commentFieldRequired = false;
        private boolean introMessageEnabled = true;
        private boolean shouldPlaySounds = false;
        private boolean successDialogEnabled = true;
        private int floatingButtonOffsetFromTop = -1;
        private boolean isSurveysAutoShowing = true;
        private boolean chatPromptOptionEnable = true;
        private boolean bugPromptOptionEnable = true;
        private boolean feedbackPromptOptionEnable = true;
        private List<Integer> deprecatedMethodsToBeLogedAfterBuild = new ArrayList<>();
        private ReproConfigurations reproConfigurations;
        private static volatile boolean isBuildCalled = false;
        @Nullable
        private String appVariant = null;

        @MaskingType
        private int[] maskingTypes = {};
        @Nullable
        private String codePushVersion = null;
        private boolean isCodePushVersionSetByUser = false;

        @Nullable
        private IBGTheme theme = null;

        /**
         * Initialize Instabug SDK with application token
         *
         * @param application      Application object for initialization of library
         * @param applicationToken Instabug application token, can be obtained from your dashboard
         * @since 2.0
         */
        public Builder(@NonNull Application application, @NonNull String applicationToken) {
            this(application, applicationToken, InstabugInvocationEvent.SHAKE);
        }

        /**
         * Initialize Instabug SDK with application token
         *
         * @param application             Application object for initialization of library
         * @param applicationToken        Instabug application token, can be obtained from your
         *                                dashboard
         * @param instabugInvocationEvent The event Instabug should listen to, to invoke the
         *                                feedback process
         * @since 2.0
         */
        public Builder(@NonNull Application application, @NonNull String applicationToken,
                       @NonNull InstabugInvocationEvent... instabugInvocationEvent) {
            IBGPendingTraceHandler.setBuilderConstructorStartTime(System.currentTimeMillis());
            this.applicationContext = application.getApplicationContext();
            this.instabugInvocationEvents = instabugInvocationEvent;
            this.applicationToken = applicationToken;
            this.application = application;
            this.reproConfigurations = ReproConfigurations.Factory.newDefaultMode();
            IBGPendingTraceHandler.setBuilderConstructorEndTime(System.currentTimeMillis());
        }

        /**
         * Set the invocation events that the SDK should be invoked on
         *
         * @param instabugInvocationEvents the invocation events
         * @see InstabugInvocationEvent
         * @since 8.0.0
         */
        public Builder setInvocationEvents(@NonNull InstabugInvocationEvent...
                                                   instabugInvocationEvents) {
            this.instabugInvocationEvents = instabugInvocationEvents;
            return this;
        }

        /**
         * Sets a {@link IBGTheme} used to customize Instabug's ui elements
         *
         * @param theme the theme to be set and used by Instabug.
         * @see IBGTheme.Builder
         */
        public Builder setTheme(@Nullable IBGTheme theme) {
            this.theme = theme;
            return this;
        }

        /**
         * Enable/disable automatic user steps tracking
         *
         * @param state desired state of automatic user step tracking feature
         * @see com.instabug.library.Feature.State
         * @since 2.0
         */
        public Builder setTrackingUserStepsState(@NonNull Feature.State state) {
            this.trackingUserStepsState = state;
            return this;
        }

        /**
         * Sets a {@link ReproConfigurations} controlling repro steps with each type of report
         *
         * @param configurations the configurations object to be set and used by Instabug.
         * @see ReproConfigurations.Builder
         */
        public Builder setReproConfigurations(ReproConfigurations configurations) {
            reproConfigurations = configurations;
            return this;
        }

        /**
         * Sets the printed logs priority filter to one of the following levels:
         * <p>
         * - {@link LogLevel#NONE} disables all SDK console logs.
         * <br/>
         * - {@link LogLevel#ERROR} prints errors and warnings, we use this level to let you know if something goes wrong.
         * This is also enabled by default
         * <br/>
         * - {@link LogLevel#DEBUG} use this in case you are debugging an issue.
         * Not recommended for production use.
         * <br/>
         * - {@link LogLevel#VERBOSE} use this only if {@link LogLevel#DEBUG} was not enough and you need more visibility
         * on what is going on under the hood.
         * Similar to the {@link LogLevel#DEBUG} level, this is not meant to be used on production environments.
         * <br/>
         * <p>
         * <br/>
         * More information can be found <a href="<https://docs.instabug.com/docs/debugging>">here</a>
         *
         * @param level the priority level.
         * @see LogLevel
         */
        public Builder setSdkDebugLogsLevel(@LogLevel final int level) {
            SettingsManager.getInstance().setLogLevel(level);
            return this;
        }

        /**
         * Enable/disable console log to be added to reports.
         *
         * @param state desired state of console log IBGFeature.
         * @see com.instabug.library.Feature.State
         * @since 2.0
         */
        public Builder setConsoleLogState(@NonNull Feature.State state) {
            this.consoleLogState = state;
            return this;
        }

        /**
         * Enable/disable Instabug log to be added to reports.
         *
         * @param state desired state of Instabug log IBGFeature.
         * @see com.instabug.library.Feature.State
         * @since 2.0
         */
        public Builder setInstabugLogState(@NonNull Feature.State state) {
            this.instabugLogState = state;
            return this;
        }

        /**
         * Enable/disable user data to be added to reports.
         *
         * @param state desired state of user data IBGFeature.
         * @see com.instabug.library.Feature.State
         * @since 2.0
         */
        public Builder setUserDataState(@NonNull Feature.State state) {
            this.userDataState = state;
            return this;
        }

        /**
         * Enable/Disable in-app messaging
         *
         * @param state desired state of in-app messaging feature
         * @since 2.0
         */
        public Builder setInAppMessagingState(@NonNull Feature.State state) {
            this.inAppMessagingState = state;
            return this;
        }

        /**
         * Enable/Disable view hierarchy
         *
         * @param state desired state of view hierarchy feature
         * @since 2.0
         */
        public Builder setViewHierarchyState(@NonNull Feature.State state) {
            this.viewHierarchyState = state;
            return this;
        }

        /**
         * Enable/disable user events.
         *
         * @param state desired state of surveys IBGFeature.
         * @see com.instabug.library.Feature.State
         * @since 4.0
         */
        public Builder setUserEventsState(@NonNull Feature.State state) {
            this.userEventsState = state;
            return this;
        }

        /**
         * an API to auto-mask screenshots.
         *
         * @param types are the masking types on screenshots
         *              by default masking type is {@link MaskingType#MASK_NOTHING}
         * @see MaskingType
         */
        public Builder setAutoMaskScreenshotsTypes(@MaskingType int... types) {
            this.maskingTypes = types;
            return this;
        }


        /**
         * Indicates whether FLAG_SECURE should be ignored or not while capturing screenshots.
         * Default: Disabled
         *
         * @param shouldBeIgnored whether FLAG_SECURE should be ignored.
         */
        public Builder ignoreFlagSecure(boolean shouldBeIgnored) {
            SettingsManager.getInstance().setIgnoreFlagSecure(shouldBeIgnored);
            return this;
        }

        /**
         * Sets Code Push version to be used for all reports.
         * Note: Should only be set for React Native apps
         *
         * @param codePushVersion the Code Push version to work with.
         */
        public Builder setCodePushVersion(@Nullable String codePushVersion) {
            this.codePushVersion = codePushVersion;
            isCodePushVersionSetByUser = true;
            return this;
        }
        /**
         * Sets App Variant to be used for all reports.
         * Should send with all requests headers
         *
         * @param appVariant the app variant version to work with.
         */
        public Builder setAppVariant(@Nullable String appVariant) {
            this.appVariant = AppVariantHandler.sanitizeAppVariant(appVariant);
            return this;
        }

        /**
         * State can be changed dynamically using {@code Instabug.}{@link #enable()} & {@code
         * Instabug.}{@link Instabug#disable()}
         * after building.
         *
         * @return an {@code Instabug} instance with all the chained values set
         * @see Instabug#disable()
         * @see #build(Feature.State)
         */
        @Nullable
        public void build() {
            IBGPendingTraceHandler.setBuilderFGStartTime(System.currentTimeMillis());
            appContext = applicationContext;
            if (isBuildCalled) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "isBuildCalled true returning..");
                return;
            }
            isBuildCalled = true;
            InstabugCore.startVitalComponents(application);
            buildInBG(Feature.State.ENABLED);
            IBGPendingTraceHandler.setBuilderFGEndTime(System.currentTimeMillis());
        }

        /**
         * @param instabugInitialState desired initial state of instabug, {@code Feature.State
         *                             .DISABLED} to disable all Instabug functionality <br/>
         *                             State can be changed dynamically using
         *                             {@code Instabug.}{@link #enable()} & {@code Instabug
         *                             .}{@link Instabug#disable()}
         *                             after building.
         * @return an {@code Instabug} instance with all the chained values set and the desired
         * state
         * @see #enable()
         * @see Instabug#disable()
         * @see #build()
         */
        @Nullable
        public void build(final Feature.State instabugInitialState) {
            IBGPendingTraceHandler.setBuilderFGStartTime(System.currentTimeMillis());
            appContext = applicationContext;
            if (instabugInitialState == Feature.State.DISABLED && (applicationToken == null || applicationToken.isEmpty())) {
                InstabugCore.startVitalComponents(application);
                return;
            }
            if (isBuildCalled) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "isBuildCalled true returning..");
                return;
            }
            isBuildCalled = true;
            InstabugCore.startVitalComponents(application);
            buildInBG(instabugInitialState);
            IBGPendingTraceHandler.setBuilderFGEndTime(System.currentTimeMillis());
        }

        @VisibleForTesting
        void buildInBG(final Feature.State instabugInitialState) {
            PoolProvider.getApiExecutor().execute(() -> {
                        SettingsManager.getInstance().setIBGTheme(theme);
                        InstabugApplicationProvider.init((Application) appContext.getApplicationContext());
                        IBGPendingTraceHandler.setBuilderBGStartTime(System.currentTimeMillis());
                        if (application == null) return;
                        if (applicationToken == null || applicationToken.trim().isEmpty()) {
                            InstabugSDKLogger.w(Constants.LOG_TAG, "Invalid application token. Abort building the SDK");
                            return;
                        }
                        IBGStateEventBusSubscriber.subscribeToIBGState();
                        if (appVariant != null) SettingsManager.getInstance().setAppVariant(appVariant);
                        InstabugSDKLogger.d(Constants.LOG_TAG, "Building Instabug From BG thread, thread name: " + Thread.currentThread().getName());
                        CodePushVersionHandler.setVersion(codePushVersion, isCodePushVersionSetByUser);
                        final InstabugDelegate delegate = InstabugDelegate.getInstance(application);
                        INSTANCE = new Instabug(delegate);
                        InstabugSDKLogger.initLogger(applicationContext);
                        final boolean isInstabugEnabled = (instabugInitialState == Feature.State.ENABLED);
                        InstabugFeaturesManager.getInstance().setFeatureState(IBGFeature.INSTABUG,
                                isInstabugEnabled ? Feature.State.ENABLED : Feature.State.DISABLED);
                        delegate.setInstabugState(BUILDING);

                        logDeprecatedApis();

                        SettingsManager.getInstance().setAppToken(applicationToken);
                        CoreServiceLocator.getNetworkDiagnosticsManager().init();
                        PerSessionSettings.getInstance().setInstabugInvocationEvents(instabugInvocationEvents);
                        PluginsManager.init(applicationContext);

                        new InstabugMinimalPersistableSettings(applicationContext)
                                .setLastSDKStateEnabled(isInstabugEnabled);
                        SessionManager.init(SettingsManager.getInstance());


                        try {
                            CoreServiceLocator
                                    .getUserMaskingFilterProvider()
                                    .setAutoMaskScreenshotsTypes(maskingTypes);
                            updateAndBroadcastReproConfigurations(reproConfigurations);
                            delegate.prepare(applicationContext);
                            delegate.setInstabugState(isInstabugEnabled ? InstabugState.ENABLED : DISABLED);
                            delegate.start();
                            // Set initial Settings

                            // Send a broadcast to notify concerned APIs that the Sdk is built
                            updateFeaturesStates();
                            logFeaturesStates(isInstabugEnabled);
                            InstabugSDKLogger.d(Constants.LOG_TAG, "SDK Built");
                        } catch (Exception e) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while building the sdk: ", e);
                        }
                        IBGPendingTraceHandler.setBuilderBGEndTime(System.currentTimeMillis());
                    }
            );
        }

        private void updateFeaturesStates() {
            InstabugCore.setFeatureState(IBGFeature.USER_DATA, userDataState);
            InstabugCore.setFeatureState(IBGFeature.CONSOLE_LOGS, consoleLogState);
            InstabugCore.setFeatureState(IBGFeature.INSTABUG_LOGS, instabugLogState);
            InstabugCore.setFeatureState(IBGFeature.IN_APP_MESSAGING, inAppMessagingState);
            InstabugCore.setFeatureState(IBGFeature.PUSH_NOTIFICATION, pushNotificationState);
            InstabugCore.setFeatureState(IBGFeature.TRACK_USER_STEPS, trackingUserStepsState);
            InstabugCore.setFeatureState(IBGFeature.VIEW_HIERARCHY_V2, viewHierarchyState);
            InstabugCore.setFeatureState(IBGFeature.SURVEYS, surveysState);
            InstabugCore.setFeatureState(IBGFeature.USER_EVENTS, userEventsState);
        }

        /**
         * A method to log all the deprecated methods after building the sdk.
         */
        private void logDeprecatedApis() {
            for (Integer method : deprecatedMethodsToBeLogedAfterBuild) {
                InstabugDeprecationLogger.getInstance().log(method);
            }
        }

        private void logFeaturesStates(Boolean isInstabugEnabled) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "User data feature state is set to " + userDataState);
            InstabugSDKLogger.v(Constants.LOG_TAG, "Console log feature state is set to " + consoleLogState);
            InstabugSDKLogger.v(Constants.LOG_TAG, "Instabug logs feature state is set to " + instabugLogState);
            InstabugSDKLogger.v(Constants.LOG_TAG, "In-App messaging feature state is set to" +
                    inAppMessagingState);
            InstabugSDKLogger.v(Constants.LOG_TAG, "Push notification feature state is set to " +
                    pushNotificationState);
            InstabugSDKLogger.v(Constants.LOG_TAG, "Tracking user steps feature state is set to " +
                    trackingUserStepsState);
            InstabugSDKLogger.v(Constants.LOG_TAG, "Repro steps feature state is set to " +
                    reproConfigurations.getModesMap());
            InstabugSDKLogger.v(Constants.LOG_TAG, "View hierarchy feature state is set to " +
                    viewHierarchyState);
            InstabugSDKLogger.v(Constants.LOG_TAG, "Surveys feature state is set to " + surveysState);
            InstabugSDKLogger.v(Constants.LOG_TAG, "User events feature state is set to " + userEventsState);
            InstabugSDKLogger.v(Constants.LOG_TAG, "Instabug overall state is set to " + isInstabugEnabled);
        }
    }

    private static void updateAndBroadcastReproConfigurations(ReproConfigurations reproConfigurations) {
        SettingsManager settingsManager = SettingsManager.getInstance();
        if (settingsManager != null) {
            settingsManager.setReproConfigurations(reproConfigurations);
        }
        IBGCoreEventPublisher.post(new IBGSdkCoreEvent.ReproState(reproConfigurations.getModesMap()));
    }

    /**
     * Adds feature flag to sent with reports to BR / CR / SV3
     *
     * @param featureFlag IBGFeatureFlag to be added using this API
     */

    public static void addFeatureFlag(IBGFeatureFlag featureFlag) {
        APIChecker.checkAndRunInExecutor("Instabug.addFeaturesFlag", () -> {
            if (getInstance() != null) {
                getInstance().delegate.addFeatureFLag(featureFlag);
            }

        });
    }


    /**
     * Adds features flags to sent with reports to BR / CR / SV3
     *
     * @param featureFlagsList list of IBGFeatureFlag to be added using this API
     */
    public static void addFeatureFlags(List<IBGFeatureFlag> featureFlagsList) {
        APIChecker.checkAndRunInExecutor("Instabug.addFeaturesFlags", () -> {
            if (getInstance() != null) {
                getInstance().delegate.addFeatureFLags(featureFlagsList);
            }
        });
    }

    /**
     * remove features flags from features flags table on database based on specific keys
     *
     * @param keys list to remove features flags from database based on search this keys
     */
    public static void removeFeatureFlag(List<String> keys) {
        APIChecker.checkAndRunInExecutor("Instabug.removeFeatureFlag", () -> {
            if (getInstance() != null) {
                getInstance().delegate.removeFeatureFlag(keys);
            }
        });
    }

    /**
     * remove all features flags from features flags table on database based on specific keys
     */
    public static void removeAllFeatureFlags() {
        APIChecker.checkAndRunInExecutor("Instabug.removeAllFeatureFlags", () -> {
            if (getInstance() != null) {
                getInstance().delegate.removeAllFeatureFlags();
            }
        });
    }

    /**
     * remove feature flag from features flags table on database based on specific key
     *
     * @param key string  to remove features flags from database based on search this key
     */
    public static void removeFeatureFlag(String key) {
        APIChecker.checkAndRunInExecutor("Instabug.removeFeatureFlag", () -> {
            if (getInstance() != null) {
                getInstance().delegate.removeFeatureFlag(key);
            }
        });
    }

    /**
     * capture dialogs when taking screenshot based on boolean
     *
     * @param shouldCapture boolean to determine if dialogs should be captured when capturing screenshots
     */
    public static void shouldCaptureDialogs(boolean shouldCapture) {
        APIChecker.checkAndRunInExecutor("Instabug.shouldCaptureDialogs", () -> {
            SettingsManager.getInstance().setShouldCaptureDialog(shouldCapture);
        });
    }


    /**
     * capture surface views when taking screenshot without using media projection based on boolean
     *
     * @param shouldCapture boolean to determine if surface views should be captured when capturing screenshots
     */
    public static void shouldCaptureSurfaceView(boolean shouldCapture) {
        SettingsManager.getInstance().setShouldCaptureSurfaceView(shouldCapture);
    }

    /**
     * Capture a screenshot manually and add it to visual user steps used with different Instabug
     * components
     */
    public static void captureScreenshot() {
        APIChecker.checkAndRunInExecutor("Instabug.captureScreenshot", () -> {
            Instabug instance = getInstance();
            if (instance != null) {
                instance.delegate.captureScreenshot();
            }
        });
    }

}
