package com.instabug.library.visualusersteps;

import static com.instabug.library.model.StepType.UNKNOWN;
import static com.instabug.library.visualusersteps.VisualUserStep.ViewOrientation.LANDSCAPE;
import static com.instabug.library.visualusersteps.VisualUserStep.ViewOrientation.Orientation;
import static com.instabug.library.visualusersteps.VisualUserStep.ViewOrientation.PORTRAIT;

import android.annotation.SuppressLint;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.StringDef;

import com.instabug.library.model.StepType;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.ArrayList;

/**
 * @author hossam.
 */

public class VisualUserStep implements Serializable {


    public static final class ViewOrientation {
        public static final String LANDSCAPE = "landscape";
        public static final String PORTRAIT = "portrait";

        @Retention(RetentionPolicy.SOURCE)
        @StringDef({
                LANDSCAPE,
                PORTRAIT
        })
        public @interface Orientation {
        }
    }

    private static final String KEY_PARENT_SCREEN_ID = "parent_screen_identifier";
    private static final String KEY_SCREEN_ID = "screen_identifier";
    private static final String KEY_SCREENSHOT_ID = "screenshot_identifier";
    private static final String KEY_SCREEN_NAME = "screen_name";
    private static final String KEY_EVENT_TYPE = "event_type";
    private static final String KEY_DATE = "date";
    private static final String KEY_VIEW = "view";
    private static final String KEY_VIEW_ORIENTATION = "orientation";
    private static final String KEY_IS_CONTAIN_ICON = "is_contains";
    private static final String KEY_IS_FLAG_SECURE = "is_flag_secure";
    private static final String KEY_BUTTON_ICON = "button_icon";
    private static final String KEY_IS_MANUALLY_INVOKED  = "manually_captured";

    private long date;
    private boolean isContainIcon;
    private boolean isManuallyInvoked;
    @Nullable
    private String parentScreenId;
    @Nullable
    private String screenName;
    @Nullable
    private String screenshotId;
    @Nullable
    private String screenId;
    @Nullable
    @StepType
    private String stepType = UNKNOWN;
    @Nullable
    private String view;
    @Nullable
    private String orientation;
    @Nullable
    private String buttonIcon;

    private boolean isScreenSecure;

    private VisualUserStep(Builder builder) {
        setParentScreenId(builder.parentScreenId);
        setScreenName(builder.screenName);
        setScreenshotId(builder.screenshotId);
        setScreenId(builder.screenId);
        setStepType(builder.stepType);
        setDate(builder.date);
        setView(builder.view);
        setViewOrientation(builder.orientation);
        setButtonIcon(builder.icon);
        setContainIcon(builder.containsIcon);
        setScreenSecure(builder.isScreenSecure);
        setManuallyInvoked(builder.isManuallyInvoked);
    }

    public boolean isScreenSecure() {
        return isScreenSecure;
    }

    public void setScreenSecure(boolean isScreenSecure) {
        this.isScreenSecure = isScreenSecure;
    }

    public void setManuallyInvoked(boolean isManuallyInvoked) {
        this.isManuallyInvoked = isManuallyInvoked;
    }
    public boolean isManuallyInvoked() {
        return this.isManuallyInvoked;
    }

    public static Builder Builder(@Nullable @StepType String stepType) {
        return new Builder(stepType);
    }

    /**
     * From json array list.
     *
     * @param visualUserStepsAsJsonArray the user steps as json array
     * @return the VisualUserSteps array list
     * @throws JSONException the json exception
     */
    public static ArrayList<VisualUserStep> fromJson(JSONArray visualUserStepsAsJsonArray) throws
            JSONException {
        ArrayList<VisualUserStep> visualUserSteps = new ArrayList<>();
        if (visualUserStepsAsJsonArray != null && visualUserStepsAsJsonArray.length() > 0) {
            for (int i = 0; i < visualUserStepsAsJsonArray.length(); i++) {
                VisualUserStep step = VisualUserStep.fromJsonObject(visualUserStepsAsJsonArray
                        .getJSONObject(i));
                visualUserSteps.add(step);
            }
        }
        return visualUserSteps;
    }

    public static VisualUserStep fromJsonObject(JSONObject jsonObject) throws JSONException {
        String stepType = UNKNOWN;

        if (jsonObject.has(KEY_EVENT_TYPE)
                && !JSONObject.NULL.toString().equals(jsonObject.getString(KEY_EVENT_TYPE))) {
            stepType = jsonObject.getString(KEY_EVENT_TYPE).toUpperCase();
        }

        String screenName = null;
        if (jsonObject.has(KEY_SCREEN_NAME)) {
            screenName = jsonObject.getString(KEY_SCREEN_NAME);
        }
        String screenId = null;
        if (jsonObject.has(KEY_SCREEN_ID)) {
            screenId = jsonObject.getString(KEY_SCREEN_ID);
        }
        String screenshotId = null;
        if (jsonObject.has(KEY_SCREENSHOT_ID)) {
            screenshotId = jsonObject.getString(KEY_SCREENSHOT_ID);
        }
        String date = null;
        if (jsonObject.has(KEY_DATE)) {
            date = jsonObject.getString(KEY_DATE);
        }
        String parentScreenID = null;
        if (jsonObject.has(KEY_PARENT_SCREEN_ID)) {
            parentScreenID = jsonObject.getString(KEY_PARENT_SCREEN_ID);
        }
        boolean containsIcon = false;
        if (jsonObject.has(KEY_IS_CONTAIN_ICON)) {
            containsIcon = jsonObject.getBoolean(KEY_IS_CONTAIN_ICON);
        }
        String icon = null;
        if (jsonObject.has(KEY_BUTTON_ICON)) {
            icon = jsonObject.getString(KEY_BUTTON_ICON);
        }
        String view = null;
        if (jsonObject.has(KEY_VIEW)) {
            view = jsonObject.getString(KEY_VIEW);
        }

        String orientation = null;
        if (jsonObject.has(KEY_VIEW_ORIENTATION)) {
            String orientationValue = jsonObject.getString(KEY_VIEW_ORIENTATION);
            switch (orientationValue) {
                case LANDSCAPE:
                    orientation = LANDSCAPE;
                    break;
                default:
                case PORTRAIT:
                    orientation = PORTRAIT;
                    break;
            }
        }
        boolean isFlagSecure = false;
        if (jsonObject.has(KEY_IS_FLAG_SECURE)) {
            isFlagSecure = jsonObject.optBoolean(KEY_IS_FLAG_SECURE, false);
        }
        boolean isManuallyInvoked = false;
        if (jsonObject.has(KEY_IS_MANUALLY_INVOKED)) {
            isManuallyInvoked = jsonObject.optBoolean(KEY_IS_MANUALLY_INVOKED, false);
        }
        if (view == null) {
            view = "";
        }

        return Builder(stepType)
                .screenName(screenName)
                .screenshotId(screenshotId)
                .date(Long.parseLong(date != null ? date : "0"))
                .parentScreenId(parentScreenID)
                .view(view)
                .viewOrientation(orientation)
                .screenId(screenId)
                .isContainIcon(containsIcon)
                .buttonIcon(icon)
                .setScreenSecure(isFlagSecure)
                .setManuallyInvoked(isManuallyInvoked)
                .build();
    }

    public static String toJsonString(@Nullable ArrayList<VisualUserStep> steps) {
        JSONArray jsonArray = new JSONArray();
        if (steps != null && steps.size() > 0) {
            for (VisualUserStep step : steps) {
                jsonArray.put(step.toJson());
            }
        }
        return jsonArray.toString();
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public JSONObject toJson() {

        JSONObject stepJson = new JSONObject();

        final String nullString = "null";

        try {

            stepJson.put(KEY_PARENT_SCREEN_ID,
                    getParentScreenId() == null || getParentScreenId().equals(nullString) ?
                            JSONObject.NULL : getParentScreenId());

            stepJson.put(KEY_SCREEN_NAME,
                    getScreenName() == null || getScreenName().equals(nullString) ?
                            JSONObject.NULL : getScreenName());

            stepJson.put(KEY_SCREENSHOT_ID,
                    getScreenshotId() == null || getScreenshotId().equals
                            (nullString) ? JSONObject.NULL : getScreenshotId());

            stepJson.put(KEY_SCREEN_ID,
                    getScreenId() == null || getScreenId().equals(nullString) ?
                            JSONObject.NULL : getScreenId());

            stepJson.put(KEY_EVENT_TYPE,
                    getStepType() == null || getStepType().equals(UNKNOWN) ?
                            JSONObject.NULL : getStepType().toLowerCase());

            stepJson.put(KEY_DATE, getDate());

            stepJson.put(KEY_VIEW, getView() == null || getView().equals(nullString) ?
                    JSONObject.NULL : getView());

            stepJson.put(KEY_VIEW_ORIENTATION, getViewOrientation() == null
                    || getViewOrientation().equals(nullString) ?
                    JSONObject.NULL : getViewOrientation());

            stepJson.put(KEY_IS_CONTAIN_ICON, isContainIcon());

            stepJson.put(KEY_BUTTON_ICON, getButtonIcon());

            stepJson.put(KEY_IS_FLAG_SECURE, isScreenSecure());

            if (isManuallyInvoked) {
                stepJson.put(KEY_IS_MANUALLY_INVOKED, isManuallyInvoked());
            }

        } catch (JSONException e) {
            e.printStackTrace();
        }
        return stepJson;
    }

    @Nullable
    public String getParentScreenId() {
        return parentScreenId;
    }

    public void setParentScreenId(@Nullable String parentScreenId) {
        this.parentScreenId = parentScreenId;
    }

    @Nullable
    public String getScreenName() {
        return screenName;
    }

    private void setScreenName(@Nullable String screenName) {
        this.screenName = screenName;
    }

    @Nullable
    public String getScreenshotId() {
        return screenshotId;
    }

    public void setScreenshotId(@Nullable String screenshotId) {
        this.screenshotId = screenshotId;
    }

    public boolean hasScreenshot() {
        return screenshotId != null;
    }

    @Nullable
    public String getScreenId() {
        return screenId;
    }

    public void setScreenId(@Nullable String screenId) {
        this.screenId = screenId;
    }

    @Nullable
    @StepType
    public String getStepType() {
        return stepType;
    }

    private void setStepType(@Nullable @StepType String stepType) {
        this.stepType = stepType;
    }

    public long getDate() {
        return date;
    }

    public void setDate(long date) {
        this.date = date;

    }

    void setButtonIcon(@Nullable String buttonIcon) {
        this.buttonIcon = buttonIcon;
    }

    @Nullable
    private String getButtonIcon() {
        return buttonIcon;
    }

    void setContainIcon(boolean containIcon) {
        isContainIcon = containIcon;
    }

    private boolean isContainIcon() {
        return isContainIcon;
    }

    @Nullable
    public String getView() {
        return view;
    }

    public void setView(@Nullable String view) {
        this.view = view;
    }

    @Nullable
    public String getViewOrientation() {
        return orientation;
    }

    public void setViewOrientation(@Nullable @Orientation String orientation) {
        this.orientation = orientation;
    }

    @NonNull
    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String toString() {
        return "VisualUserStep{" +
                "parentScreenId='" + parentScreenId + '\'' +
                ", screenName='" + screenName + '\'' +
                ", screenshotId='" + screenshotId + '\'' +
                ", screenId='" + screenId + '\'' +
                ", eventType='" + stepType + '\'' +
                ", date=" + date +
                ", view='" + view + '\'' +
                '}';
    }

    /**
     * {@code VisualUserStep} builder static inner class.
     */
    public static final class Builder {
        private long date = System.currentTimeMillis();
        private boolean containsIcon;
        private boolean isManuallyInvoked;
        @Nullable
        private String parentScreenId;
        @Nullable
        private String screenName;
        @Nullable
        private String screenshotId;
        @Nullable
        private String screenId;
        @Nullable
        private String view;
        @Nullable
        private String orientation;
        @Nullable
        private String icon;
        @Nullable
        @StepType
        private String stepType;
        private boolean isScreenSecure;

        private Builder(@Nullable @StepType String stepType) {
            this.stepType = stepType;
        }

        /**
         * Sets the {@code parentScreenId} and returns a reference to this Builder so that the
         * methods can be chained together.
         *
         * @param parentScreenId the {@code parentScreenId} to set
         * @return a reference to this Builder
         */
        public Builder parentScreenId(@Nullable String parentScreenId) {
            this.parentScreenId = parentScreenId;
            return this;
        }

        /**
         * Sets the {@code screenName} and returns a reference to this Builder so that the
         * methods can be chained together.
         *
         * @param screenName the {@code screenName} to set
         * @return a reference to this Builder
         */
        public Builder screenName(@Nullable String screenName) {
            this.screenName = screenName;
            return this;
        }

        /**
         * Sets the {@code screenshotId} and returns a reference to this Builder so that the
         * methods can be chained together.
         *
         * @param screenshotId the {@code screenshotId} to set
         * @return a reference to this Builder
         */
        public Builder screenshotId(@Nullable String screenshotId) {
            this.screenshotId = screenshotId;
            return this;
        }

        /**
         * Sets the {@code screenId} and returns a reference to this Builder so that the methods
         * can be chained together.
         *
         * @param screenId the {@code screenId} to set
         * @return a reference to this Builder
         */
        public Builder screenId(@Nullable String screenId) {
            this.screenId = screenId;
            return this;
        }

        /**
         * Sets the {@code date} and returns a reference to this Builder so that the methods can
         * be chained together.
         *
         * @param date the {@code date} to set
         * @return a reference to this Builder
         */
        public Builder date(long date) {
            this.date = date;
            return this;
        }

        /**
         * Sets the {@code view} and returns a reference to this Builder so that the methods can
         * be chained together.
         *
         * @param view the {@code view} to set
         * @return a reference to this Builder
         */
        public Builder view(@Nullable String view) {
            this.view = view;
            return this;
        }

        /**
         * Sets the {@code view orientation} and returns a reference to this Builder so that the
         * methods can
         * be chained together.
         *
         * @param orientation the {@code orientation} to set
         * @return a reference to this Builder
         */
        public Builder viewOrientation(@Nullable @Orientation String orientation) {
            this.orientation = orientation;
            return this;
        }

        /**
         * Sets the {@code view containsIcon} and returns a reference to this Builder so that the
         * methods can
         * be chained together.
         *
         * @param containsIcon the {@code containsIcon} to set
         * @return a reference to this Builder
         */
        public Builder isContainIcon(boolean containsIcon) {
            this.containsIcon = containsIcon;
            return this;
        }

        /**
         * Sets the {@code view orientation} and returns a reference to this Builder so that the
         * methods can
         * be chained together.
         *
         * @param icon the {@code icon} to set
         * @return a reference to this Builder
         */
        public Builder buttonIcon(@Nullable String icon) {
            this.icon = icon;
            return this;
        }

        /**
         * Sets if the activity flag secure is enabled
         *
         * @return a reference to this Builder
         */
        public Builder setScreenSecure(boolean screenSecure) {
            this.isScreenSecure = screenSecure;
            return this;
        }

        public Builder setManuallyInvoked(boolean isManuallyInvoked) {
            this.isManuallyInvoked = isManuallyInvoked;
            return this;
        }

        /**
         * Returns a {@code VisualUserStep} built from the parameters previously set.
         *
         * @return a {@code VisualUserStep} built with parameters of this {@code VisualUserStep
         * .Builder}
         */
        public VisualUserStep build() {
            return new VisualUserStep(this);
        }


    }

}
