package com.instabug.library.util.threading

import androidx.annotation.VisibleForTesting
import com.instabug.library.Feature
import com.instabug.library.IBGFeature
import com.instabug.library.core.InstabugCore
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.settings.PersistableSettings
import com.instabug.library.util.TimeUtils

class IBGQueueMonitoringHelper(
    private val dequeueThreshold: Long, private val completionThreshold: Long
) {
    private val DAY_IN_MILLIS: Long = 24 * 60 * 60 * 1000;

    @VisibleForTesting
    var enqueueTimes = mutableMapOf<String, Long>()

    @VisibleForTesting
    var queueLength: Int = 0


    fun onJobEnqueued(jobId: String?) {
        jobId?.let {
            enqueueTimes[it] = System.currentTimeMillis()
            queueLength++
        }
    }

    fun onJobDequeued(jobId: String?) {
        jobId?.let {
            reportNonFatalIfNecessary(it, System.currentTimeMillis(), JobState.DEQUEUED)
        }
    }

    fun onJobCompleted(jobId: String?) {
        jobId?.let {
            reportNonFatalIfNecessary(it, System.currentTimeMillis(), JobState.COMPLETED)
            queueLength--
            enqueueTimes.remove(it)
        }
    }


    private fun reportNonFatalIfNecessary(jobId: String, timestamp: Long, jobState: JobState) {
        if (TimeUtils.hasXHoursPassed(
                PersistableSettings.getInstance()?.lastReportTime ?: 0L,
                DAY_IN_MILLIS
            )
        ) {
            enqueueTimes[jobId]?.let {
                val elapsedTime = timestamp - it
                val encryptionState = InstabugCore.getFeatureState(IBGFeature.DB_ENCRYPTION)

                when (jobState) {
                    JobState.DEQUEUED -> {
                        if (dequeueThreshold != 0L && elapsedTime > dequeueThreshold) {
                            IBGDiagnostics.reportNonFatal(
                                QueueMonitoringException(),
                                "Job exceeded took $elapsedTime milliseconds. in queue before being ${jobState.name} " +
                                        " Queue length: $queueLength, DB Encryption state: $encryptionState"
                            )
                            PersistableSettings.getInstance()
                                ?.saveLastReportTime(System.currentTimeMillis())
                        }
                    }

                    JobState.COMPLETED -> {
                        if (completionThreshold != 0L && elapsedTime > completionThreshold) {
                            IBGDiagnostics.reportNonFatal(
                                QueueMonitoringException(),
                                "Job exceeded took $elapsedTime milliseconds. in queue before being ${jobState.name} " +
                                        " Queue length: $queueLength, DB Encryption state: $encryptionState"
                            )
                            PersistableSettings.getInstance()
                                ?.saveLastReportTime(System.currentTimeMillis())
                        }
                    }
                }
            }
        }
    }


    private enum class JobState {
        DEQUEUED, COMPLETED
    }

    private class QueueMonitoringException : Exception()
}