package com.instabug.library.screenshot;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ServiceInfo;
import android.graphics.PixelFormat;
import android.hardware.display.DisplayManager;
import android.hardware.display.VirtualDisplay;
import android.media.ImageReader;
import android.media.projection.MediaProjection;
import android.media.projection.MediaProjectionManager;
import android.os.Build;
import android.os.Handler;
import android.os.IBinder;
import android.util.DisplayMetrics;

import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.instabug.library.Constants;
import com.instabug.library.R;
import com.instabug.library.core.eventbus.ScreenCaptureEventBus;
import com.instabug.library.internal.InstabugMediaProjectionIntent;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.NotificationUtils;
import com.instabug.library.util.threading.PoolProvider;

import java.security.SecureRandom;

@RequiresApi(21)
public class ScreenshotCaptureService extends Service {

    private static final String SCREENCAP_NAME = "screencap";
    private static final int VIRTUAL_DISPLAY_FLAGS =
            DisplayManager.VIRTUAL_DISPLAY_FLAG_OWN_CONTENT_ONLY |
                    DisplayManager.VIRTUAL_DISPLAY_FLAG_PUBLIC;
    private static final int SCREENSHOT_NOTIFICATION_ID = new SecureRandom().nextInt();
    public static final String EXTRA_MEDIA_PROJ_INTENT = "instabug.intent.extra.MEDIA_PROJ_INTENT";

    @Nullable
    private MediaProjection mediaProjection;
    @Nullable
    private VirtualDisplay virtualDisplay;
    @Nullable
    private ImageReader imageReader;

    private final MediaProjection.Callback mediaProjectionCallback = new MediaProjection.Callback() {
        @Override
        public void onStop() {
            super.onStop();
            if (virtualDisplay != null) {
                virtualDisplay.release();
                virtualDisplay = null;
            }
            if (mediaProjection != null) {
                mediaProjection.unregisterCallback(this);
                mediaProjection = null;
            }
            imageReader = null;
        }
    };

    public static Intent newIntent(Context context, Intent mediaProjectionIntent) {
        Intent intent = new Intent(context, ScreenshotCaptureService.class);
        intent.putExtra(EXTRA_MEDIA_PROJ_INTENT, mediaProjectionIntent);
        return intent;
    }

    @Override
    public void onCreate() {
        super.onCreate();
        ScreenCaptureEventBus.getInstance().subscribe(screenshotResponse -> stopForeground(true));
    }

    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    @Override
    @SuppressLint("infer")
    public int onStartCommand(Intent intent, int flags, int startId) {
        if (intent != null) {
            InstabugMediaProjectionIntent.setMediaProjectionIntentUsed(true);
            // Creating notification for a MediaProjection service is mandatory in android 10
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                NotificationUtils.createAndShowForegroundNotification(this,
                        R.string.ibg_screen_recording_notification_title, SCREENSHOT_NOTIFICATION_ID, ServiceInfo.FOREGROUND_SERVICE_TYPE_MEDIA_PROJECTION);
            }
            Intent mIntent = intent.getParcelableExtra(EXTRA_MEDIA_PROJ_INTENT);
            if (mIntent == null) {
                stopForeground(true);
                InstabugSDKLogger.e(Constants.LOG_TAG, "Passed media projection intent is null");
            }
            if (mediaProjection != null) {
                mediaProjection.stop();
                mediaProjection = null;
            }
            if (mIntent != null) {
                final MediaProjectionManager mediaProjectionManager =
                        (MediaProjectionManager) getSystemService(Context.MEDIA_PROJECTION_SERVICE);
                mediaProjection = mediaProjectionManager.getMediaProjection(Activity.RESULT_OK, mIntent);
            }
            if (mediaProjection == null) {
                stopForeground(true);
            }
            final DisplayMetrics metrics = new DisplayMetrics();
            Activity currentActivity = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
            if (currentActivity != null) {
                currentActivity.getWindowManager().getDefaultDisplay().getRealMetrics(metrics);
                final int width = metrics.widthPixels, height = metrics.heightPixels;
                final int density = metrics.densityDpi;

                if (mediaProjection != null) {
                    imageReader = ImageReader.newInstance(width, height, PixelFormat.RGBA_8888, 1);
                    mediaProjection.registerCallback(mediaProjectionCallback, null);
                    virtualDisplay = mediaProjection.createVirtualDisplay(
                            SCREENCAP_NAME,
                            width, height, density,
                            VIRTUAL_DISPLAY_FLAGS,
                            imageReader.getSurface(), null, null);

                    imageReader.setOnImageAvailableListener(reader -> {
                        if (imageReader == null) {
                            //The screenshot image has been acquired before.
                            return;
                        }
                        if (mediaProjection != null) {
                            mediaProjection.stop();
                        }
                        BitmapProviderRunnable runnable =
                                new BitmapProviderRunnable(width, height, reader);
                        PoolProvider.postIOTask(runnable);
                    }, new Handler());
                }
            } else {
                stopForeground(true);
            }
        } else {
            stopForeground(true);
        }
        return super.onStartCommand(intent, flags, startId);
    }
}
