package com.instabug.library.networkinterception.config

import com.instabug.library.internal.sharedpreferences.PreferencePropertyFactory
import com.instabug.library.percentagefeatures.PercentageFeature
import com.instabug.library.percentagefeatures.resolvePercentages

class NetworkInterceptionConfigurationProviderImpl(
    preferencesPropertyFactory: PreferencePropertyFactory,
) : IBGNetworkInterceptionConfigurationProvider {

    private val w3CNetworkExternalTraceIdPercentageFeatureDelegate =
        preferencesPropertyFactory.create<PercentageFeature?>(
            KEY_W3C_EXTERNAL_TRACE_ID_AVAILABLE,
            PercentageFeature(),
        )

    private val autoMaskingPercentageFeatureDelegate =
        preferencesPropertyFactory.create<PercentageFeature?>(
            KEY_AUTO_MASKING_AVAILABLE,
            PercentageFeature().apply { beValue = DEFAULT_AUTO_MASKING_PERCENTAGE },
        )

    private var w3CNetworkExternalTraceIdPercentageFeature: PercentageFeature?
        by w3CNetworkExternalTraceIdPercentageFeatureDelegate

    private var autoMaskingPercentageFeature: PercentageFeature?
        by autoMaskingPercentageFeatureDelegate

    override fun setW3CNetworkExternalTraceIdFeatureAvailabilityPercentage(percentage: Double) {
        w3CNetworkExternalTraceIdPercentageFeature =
            w3CNetworkExternalTraceIdPercentageFeature?.apply {
                resolvePercentages(percentage)
            }
    }

    override fun isW3CNetworkExternalTraceIdEnabled(): Boolean =
        w3CNetworkExternalTraceIdPercentageFeature?.isEnabled
            ?: DEFAULT_W3C_EXTERNAL_TRACE_ID_AVAILABLE

    private val isAttachingGeneratedW3CExternalTraceIdEnabledDelegate =
        preferencesPropertyFactory.create(
            KEY_GENERATED_W3C_ATTACHING_AVAILABLE,
            DEFAULT_GENERATED_W3C_ATTACHING_AVAILABLE,
        )
    override var isAttachingGeneratedW3CExternalTraceIdFeatureAvailable: Boolean
        by isAttachingGeneratedW3CExternalTraceIdEnabledDelegate
    override val isAttachingGeneratedW3CExternalTraceIdEnabled: Boolean
        get() =
            isAttachingGeneratedW3CExternalTraceIdFeatureAvailable &&
                isW3CNetworkExternalTraceIdEnabled()

    private val isAttachingCapturedW3CExternalTraceIdEnabledDelegate =
        preferencesPropertyFactory.create(
            KEY_CAPTURED_W3C_ATTACHING_AVAILABLE,
            DEFAULT_CAPTURED_W3C_ATTACHING_AVAILABLE,
        )
    override var isAttachingCapturedW3CExternalTraceIdFeatureAvailable: Boolean
        by isAttachingCapturedW3CExternalTraceIdEnabledDelegate

    override val isAttachingCapturedW3CExternalTraceIdEnabled: Boolean
        get() =
            isAttachingCapturedW3CExternalTraceIdFeatureAvailable &&
                isW3CNetworkExternalTraceIdEnabled()

    override fun setAutoMaskingFeatureAvailabilityPercentage(percentage: Double) {
        val newPercentageFeature = PercentageFeature()
        autoMaskingPercentageFeature?.also { existing -> newPercentageFeature.fromJson(existing.toJson()) }
        autoMaskingPercentageFeature = newPercentageFeature.apply { resolvePercentages(percentage) }
    }

    override fun isAutoMaskingAvailable(): Boolean =
        autoMaskingPercentageFeature?.isEnabled ?: DEFAULT_AUTO_MASKING_BE_ENABLED

    override var isAutoMaskingSDKEnabled: Boolean = DEFAULT_AUTO_MASKING_SDK_ENABLED

    override val isAutoMaskingEnabled: Boolean
        get() = isAutoMaskingAvailable() && isAutoMaskingSDKEnabled

    private val autoMaskingBEHeaderKeysDelegate =
        preferencesPropertyFactory.create(
            KEY_AUTO_MASKING_HEADER_KEYS,
            DEFAULT_SDK_AUTO_MASKING_HEADER_KEYS,
        )

    override var autoMaskingBEHeaderKeys: Set<String>
        by autoMaskingBEHeaderKeysDelegate

    private val autoMaskingBEQueryKeysDelegate =
        preferencesPropertyFactory.create(
            KEY_AUTO_MASKING_QUERY_KEYS,
            DEFAULT_SDK_AUTO_MASKING_QUERY_KEYS,
        )

    override var autoMaskingBEQueryKeys: Set<String>
        by autoMaskingBEQueryKeysDelegate

    override fun resetAutoMasking() {
        autoMaskingPercentageFeatureDelegate.clear()
        autoMaskingBEQueryKeysDelegate.clear()
        autoMaskingBEHeaderKeysDelegate.clear()
    }

    override fun reset() {
        w3CNetworkExternalTraceIdPercentageFeatureDelegate.clear()
        isAttachingGeneratedW3CExternalTraceIdEnabledDelegate.clear()
        isAttachingCapturedW3CExternalTraceIdEnabledDelegate.clear()
        networkBodyCaptureRuleDelegation.clear()
        resetAutoMasking()
    }

    private val networkBodyCaptureRuleDelegation = preferencesPropertyFactory.create(
        KEY_BE_CAPTURE_NETWORK_BODY_LOGS,
        DEFAULT_CAPTURE_NETWORK_BODY_LOG_BE
    )

    override var isNetworkLogBodyEnabledByBE: Boolean
        by networkBodyCaptureRuleDelegation
    override var isNetworkLogBodyEnabledBySDK: Boolean = DEFAULT_CAPTURE_NETWORK_BODY_LOG_SDK
    override val shouldCaptureNetworkBodyLogs: Boolean
        get() =
            isNetworkLogBodyEnabledBySDK && isNetworkLogBodyEnabledByBE
}
