package com.instabug.library.model.v3Session

import com.instabug.library.internal.storage.cache.dbv2.IBGContentValues
import com.instabug.library.internal.storage.cache.dbv2.IBGCursor
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionExperimentEntry.*
import com.instabug.library.map.TwoWayMapper
import com.instabug.library.model.v3Session.SessionsExperimentsKeys.EXPERIMENTS_ARRAY_KEY
import com.instabug.library.model.v3Session.SessionsExperimentsKeys.EXPERIMENTS_DROPPED_COUNT_KEY
import com.instabug.library.util.extenstions.getInt
import com.instabug.library.util.extenstions.getLong
import com.instabug.library.util.extenstions.getString
import com.instabug.library.util.extenstions.toJsonArray
import org.json.JSONObject

val IBGSessionExperiments.asContentValues
    get() = IBGContentValues().apply {
        put(COLUMN_SESSION_SERIAL, sessionSerial, true)
        put(
            COLUMN_EXPERIMENT_ARRAY,
            experiments.let(ExperimentsCacheMapper::mapForwards),
            false
        )
        put(COLUMN_DROPPED_COUNT, droppedCount, false)

    }
val IBGCursor.asSessionExperimentsMap
    get() = use {
        hashMapOf<Long, IBGSessionExperiments>()
            .apply {
                while (moveToNext()) sessionExperiments.also { experiments ->
                    put(experiments.sessionSerial, experiments)
                }
            }
    }
private val IBGCursor.sessionExperiments
    get() = IBGSessionExperiments(
        sessionSerial = getLong(COLUMN_SESSION_SERIAL),
        experiments = getString(COLUMN_EXPERIMENT_ARRAY).let(ExperimentsCacheMapper::mapBackwards),
        droppedCount = getInt(COLUMN_DROPPED_COUNT)
    )
val IBGSessionExperiments.asJsonString
    get() = takeIf { experiments.isNotEmpty() || droppedCount > 0 }
        ?.let { JSONObject() }
        ?.apply {
            put(EXPERIMENTS_ARRAY_KEY, experiments.toJsonArray)
            if (droppedCount > 0) {
                put(EXPERIMENTS_DROPPED_COUNT_KEY, droppedCount)
            }
        }?.toString()

object ExperimentsCacheMapper : TwoWayMapper<List<String>, String> {

    override fun mapForwards(type1: List<String>): String =
        type1.joinToString(separator, transform = ::escapeSpecialCharacters)

    override fun mapBackwards(type2: String): List<String> =
        if (type2.isEmpty())
            emptyList()
        else
            type2.split(splitRegex).map(::undoEscapeSpecialCharacters)

    private fun escapeSpecialCharacters(experiments: String): String =
        experiments.replace(backslash, escapedBackslash).replace(separator, escapedSeparator)

    private fun undoEscapeSpecialCharacters(experiments: String): String =
        experiments.replace(escapedSeparator, separator).replace(escapedBackslash, backslash)

    private const val separator = ","
    private const val escapedSeparator = "\\,"
    private const val backslash = "\\"
    private const val escapedBackslash = "\\/\\"
    private val splitRegex = "((?<!\\\\),)|((?<=\\\\/\\\\),)".toRegex()


}