package com.instabug.library.instacapture.screenshot.pixelcopy;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.graphics.Bitmap;
import android.graphics.Rect;
import android.os.Build;
import android.util.Pair;
import android.view.PixelCopy;
import android.view.View;
import android.view.Window;

import androidx.annotation.IdRes;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.instabug.library.Constants;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.instacapture.exception.ScreenCapturingFailedException;
import com.instabug.library.instacapture.utility.Memory;
import com.instabug.library.instacapture.utility.ViewUtilsKt;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import java.util.HashMap;
import java.util.Map;

import io.reactivexport.Observable;
import io.reactivexport.Single;
import io.reactivexport.SingleEmitter;
import io.reactivexport.SingleOnSubscribe;
import io.reactivexport.SingleSource;
import io.reactivexport.android.schedulers.AndroidSchedulers;
import io.reactivexport.functions.Function;
import io.reactivexport.schedulers.Schedulers;

public class PixelCopyDelegate {

    @RequiresApi(api = Build.VERSION_CODES.O)
    public static Observable<Bitmap> takeScreenshot(@NonNull final Activity activity, @Nullable @IdRes final int[] ignoredViewsIds) {
        return takeWindowScreenShot(activity, ignoredViewsIds).toObservable();
    }

    @RequiresApi(api = Build.VERSION_CODES.O)
    private static Single<Bitmap> takeWindowScreenShot(@NonNull Activity activity, @Nullable @IdRes final int[] ignoredViewsIds) {
        Single<Bitmap> single = Single.create((SingleOnSubscribe<Bitmap>) emitter -> {
                    // create screenshot bitmap
                    Rect bounds = ViewUtilsKt.rootViewBoundsInWindow(activity);
                    if (bounds == null) {
                        emitter.onError(new ScreenCapturingFailedException("Activity is not visual or Top level window decor has not been created yet."));
                        return;
                    }
                    int width = bounds.width();
                    int height = bounds.height();
                    try {
                        final Bitmap screenshot;
                        if ((long) width * height * 4 < Memory.getFreeMemory(activity)) {
                            // ARGB_8888 store each pixel in 4 bytes
                            screenshot = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
                        } else {
                            // RGB_565 store each pixel in 2 bytes
                            screenshot = Bitmap.createBitmap(width, height, Bitmap.Config.RGB_565);
                        }
                        emitter.onSuccess(screenshot);
                    } catch (IllegalArgumentException | OutOfMemoryError e) {
                        // log error
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while capturing " + e.getMessage(), e);
                        emitter.onError(e);
                    }
                }).subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .map((Function<Bitmap, Pair<Bitmap, HashMap<View, Integer>>>) screenshot -> {
                    // hide ignored views
                    try {
                        final HashMap<View, Integer> ignoredViews = hideIgnoredViews(activity, ignoredViewsIds);
                        return new Pair<>(screenshot, ignoredViews);
                    } catch (OutOfMemoryError | Exception e) {
                        IBGDiagnostics.reportNonFatal(e, e.getMessage() != null ? "Something went wrong while hide Ignored Views " : "");
                        return new Pair<>(screenshot, new HashMap<>());
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .doOnSuccess((ignored) -> { /*Waiting visibility to be applied*/ })
                .observeOn(Schedulers.io())
                .flatMap(screenshotCapturingObserver(activity));

        if (SettingsManager.getInstance().shouldCaptureSurfaceView()) {
            return single
                    .observeOn(AndroidSchedulers.mainThread())
                    .flatMap(SurfaceViewPixelCopyHandlerKt.appendSurfaceViewToScreenshotIfPresent(activity,
                            Schedulers.io()));
        } else {
            return single;
        }
    }

    @RequiresApi(api = Build.VERSION_CODES.O)
    @NonNull
    private static Function<Pair<Bitmap, HashMap<View, Integer>>, SingleSource<Bitmap>> screenshotCapturingObserver(@NonNull Activity activity) {
        return bitmapHashMapPair -> Single
                .create(emitter -> {
                    final Bitmap screenshot = bitmapHashMapPair.first;
                    final HashMap<View, Integer> ignoredViews = bitmapHashMapPair.second;
                    try {
                        // capture screenshot
                        Window window = activity.getWindow();
                        requestScreenshotOfWindow(window, screenshot, createFinishedListener(emitter, screenshot, activity, ignoredViews));
                    } catch (OutOfMemoryError | Exception e) {
                        IBGDiagnostics.reportNonFatal(e, e.getMessage() != null ? "Something went wrong while capturing " : "");
                        emitter.onError(e);
                    }
                });
    }

    @RequiresApi(api = Build.VERSION_CODES.O)
    private static PixelCopy.OnPixelCopyFinishedListener createFinishedListener(SingleEmitter<Bitmap> emitter, Bitmap screenshot, Activity activity, HashMap<View, Integer> ignoredViews) {
        return new PixelCopy.OnPixelCopyFinishedListener() {
            @SuppressLint("STARVATION")
            @Override
            public void onPixelCopyFinished(int resultCode) {
                if (resultCode == PixelCopy.SUCCESS) {
//                    // render dialog
                    if (SettingsManager.getInstance().getShouldCaptureDialog()) {
                        renderVisibleDialogs(activity, resultCode, screenshot);
                    }
//
                } else {
                    screenshot.recycle();
                    String errorMessage = "Something went wrong while capturing screenshot using PixelCopy.request resultCode: " + resultCode;
                    InstabugSDKLogger.e(Constants.LOG_TAG, errorMessage);
                    emitter.onError(new Exception(errorMessage));
                    reShowIgnoredViews(ignoredViews);
                    return;
                }
                // reshow ignored views
                reShowIgnoredViews(ignoredViews);
                emitter.onSuccess(screenshot);
            }
        };
    }

    private static HashMap<View, Integer> hideIgnoredViews(@NonNull Activity activity, int[] ignoredViewsIds) {

        HashMap<View, Integer> ignoredViews = new HashMap<>();
        if (activity == null) return ignoredViews;

        if (ignoredViewsIds != null) {
            for (int ignoredViewsId : ignoredViewsIds) {

                final View ignoredView = activity.findViewById(ignoredViewsId);
                if (ignoredView != null) {
                    ignoredViews.put(ignoredView, ignoredView.getVisibility());
                    ignoredView.setVisibility(View.GONE);
                }
            }
        }
        return ignoredViews;
    }

    private static void reShowIgnoredViews(final HashMap<View, Integer> ignoredViews) {
        PoolProvider.postMainThreadTask(() -> {
            if (ignoredViews == null || ignoredViews.size() == 0) return;

            for (Map.Entry<View, Integer> entry : ignoredViews.entrySet()) {
                entry.getKey().setVisibility(entry.getValue());
            }
        });
    }

    @SuppressLint("CheckResult")
    private static void renderVisibleDialogs(Activity activity, int resultCode, Bitmap screenshot) {
        DialogUiRenderer.tryRenderDialogs(activity, screenshot);

    }

    @RequiresApi(api = Build.VERSION_CODES.O)
    private static void requestScreenshotOfWindow(Window window, Bitmap screenshot, PixelCopy.OnPixelCopyFinishedListener listener) {
        PixelCopy.request(window, screenshot, listener, PixelCopyHandlerProvider.getHandler());

    }
}
