package com.instabug.library.visualusersteps

import android.graphics.Bitmap
import android.view.View
import com.instabug.library.interactionstracking.IBGUINode
import com.instabug.library.model.StepType
import java.lang.ref.WeakReference
import java.util.concurrent.Future

/**
 * A contract for any party implementing repro steps capturing capabilities.
 */
interface ReproStepsCaptor : ReproStepsManualCaptor {

    /**
     * Sets the last view manually to be used internally while capturing.
     * @param viewRef a [WeakReference] of the last assignable [View].
     */
    fun setLastView(viewRef: WeakReference<View>?)

    /**
     * Returns the current parent currently exists internally.
     * @return a [Parent] object representing the current parent. Could be null.
     */
    fun getCurrentParent(): Parent?

    /**
     * Adds a repro step.
     * @param screenName the screen name to be used while adding the step.
     * @param bitmap the [Bitmap] representing screenshot captured using another party.
     */
    fun addVisualUserStep(screenName: String, bitmap: Bitmap?)

    /**
     * Adds a repro step.
     * @param parent the [Parent] that this parent should be added as a child of.
     * @param stepType the kind of step to be added.
     * @param screenName the screen name to be used while adding the step.
     * @param view the view class to be used while add the step.
     * @param icon the icon path if captured.
     */
    fun addVisualUserStep(
        @StepType stepType: String,
        screenName: String?,
        view: String?,
        icon: String?
    )

    fun addVisualUserStep(
        @StepType stepType: String,
        screenName: String?,
        finalTarget: IBGUINode,
        touchedView: Future<TouchedView?>
    )

    /**
     * Logs a keyboard event
     * @param isKeyboardOpen whether the event representing a keyboard open or close.
     */
    fun logKeyboardEvent(isKeyboardOpen: Boolean)

    /**
     * Logs a change in focus.
     * @param oldFocus the [View] that was focused before receiving the change event.
     * @param newFocus the [View] that has been focused to override the old focused view.
     */
    fun logFocusChange(oldFocus: View?, newFocus: View?)

    /**
     * Logs a step telling that Instabug has been enabled.
     */
    fun logInstabugEnabledStep()

    /**
     * Logs a step telling that the app is now in foreground.
     */
    fun logForegroundStep()

    /**
     * Logs a step telling that the app is now in background.
     */
    fun logBackgroundStep()

    /**
     * Removes a specific screenshot using its Uri.
     * @param screenshotUri a [String] representing the screenshot Uri to be removed.
     */
    fun removeScreenshotId(screenshotUri: String)

    /**
     * Removes the last added step if it's a [StepType.TAP].
     */
    fun removeLastTapStep()

    /**
     * Fetches currently available steps.
     * @return an [ArrayList] of [VisualUserStep].
     */
    fun fetch(): ArrayList<VisualUserStep?>

    /**
     * Removes all the steps added previously.
     */
    fun clean()

    /**
     * Resets the underlying state of the captor.
     */
    fun reset()
}