package com.instabug.library.util;

import android.app.Activity;
import android.graphics.Rect;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.view.View;
import android.view.ViewTreeObserver;
import android.widget.EditText;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.util.threading.PoolProvider;
import com.instabug.library.view.ViewUtils;

import java.lang.ref.WeakReference;

public class KeyboardEventListener implements ViewTreeObserver.OnGlobalFocusChangeListener {

    private static final String OPEN_KEYBOARD_TASK_KEY = "open_keyboard_task";
    private final ViewTreeObserver.OnGlobalLayoutListener listener;
    private final WeakReference<Activity> activityRef;
    private final KeyboardCallback callback;

    public KeyboardEventListener(@NonNull final Activity activity, @NonNull KeyboardCallback callback) {
        this.activityRef = new WeakReference<>(activity);
        this.callback = callback;
        this.listener = new ViewTreeObserver.OnGlobalLayoutListener() {
            private boolean lastState;

            public void onGlobalLayout() {
                isKeyboardOpen(activityRef.get(), keyboardOpen -> {
                    if (keyboardOpen != this.lastState) {
                        dispatchKeyboardEvent(keyboardOpen);
                        this.lastState = keyboardOpen;

                    }
                });
            }

            {
                isKeyboardOpen(activityRef.get(), keyboardOpen -> this.lastState = keyboardOpen);
            }
        };
        registerKeyboardListener(activityRef.get());
        registerFocusChangeListener(activityRef.get());
    }


    private void registerFocusChangeListener(@Nullable Activity activity) {
        View view = getRootView(activity);
        if (view != null) {
            view.getViewTreeObserver().addOnGlobalFocusChangeListener(this);
        }
    }

    private void registerKeyboardListener(@Nullable Activity activity) {
        View view = getRootView(activity);
        if (view != null) {
            view.getViewTreeObserver().addOnGlobalLayoutListener(this.listener);
        }
    }

    private void dispatchKeyboardEvent(boolean isOpen) {
        callback.isOpen(isOpen);
    }

    public final void unregisterKeyboardListener() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            if (activityRef != null) {
                View view = getRootView(activityRef.get());
                if (view != null) {
                    view.getViewTreeObserver().removeOnGlobalLayoutListener(listener);
                    view.getViewTreeObserver().removeOnGlobalFocusChangeListener(this);
                }
            }
        }
    }

    public final void isKeyboardOpen(@Nullable Activity activity, OpenKeyBoardCallback mainThreadOpenKeyBoardCallback) {
        PoolProvider.postOrderedIOTask(OPEN_KEYBOARD_TASK_KEY, () -> {
            boolean isOpen;
            if (activity != null) {
                View view = getRootView(activity);
                if (view != null) {
                    Rect visibleBounds = new Rect();
                    view.getWindowVisibleDisplayFrame(visibleBounds);
                    int heightDiff = activity.getWindow().getDecorView().getRootView().getHeight() - visibleBounds.height();
                    int marginOfError = ViewUtils.convertDpToPx(activity, 100.0F);
                    isOpen = heightDiff > marginOfError;
                } else {
                    isOpen = false;
                }
            } else {
                isOpen = false;
            }
            new Handler(Looper.getMainLooper()).post(() -> mainThreadOpenKeyBoardCallback.keyboardOpen(isOpen));
        });

    }


    @Nullable
    View getRootView(@Nullable Activity activity) {
        if (activity != null) {
            View rootView = null;
            try {
                rootView = activity.getWindow().getDecorView().getRootView();
            } catch (Exception exception) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "Couldn't find activity root view while registering keyboard listener due to: " + exception.getMessage());
            }
            return rootView;
        } else
            return null;
    }

    @Override
    public void onGlobalFocusChanged(View oldFocus, View newFocus) {
        if (newFocus == null) {
            return;
        }
        if (newFocus instanceof EditText) {
            if (oldFocus == null || oldFocus != newFocus) {
                CoreServiceLocator.getReproStepsProxy().setLastView(new WeakReference<>(newFocus));
                CoreServiceLocator.getReproStepsProxy().logFocusChange(oldFocus, newFocus);
            }
        } else {
            CoreServiceLocator.getReproStepsProxy().setLastView(null);
            CoreServiceLocator.getReproStepsProxy().logKeyboardEvent(false);
        }
    }


    public interface KeyboardCallback {
        void isOpen(boolean isKeyboardOpen);
    }

    public interface OpenKeyBoardCallback {
        void keyboardOpen(Boolean keyboard);
    }
}
