package com.instabug.library.sessionreplay.model

import android.graphics.Bitmap
import com.instabug.library.sessionreplay.LOG_TYPE_KEY
import com.instabug.library.sessionreplay.TIMESTAMP_KEY
import com.instabug.library.sessionreplay.bitmap.BitmapScaler
import com.instabug.library.sessionreplay.model.SRLogType.Companion.SCREENSHOT
import com.instabug.library.visualusersteps.VisualUserStep.ViewOrientation.Orientation
import org.json.JSONObject

private const val SCREENSHOT_ID = "screenshot_identifier"

private const val SCREEN_NAME = "screen_name"
private const val SCREEN_LONG_NAME = "screen_long_name"

private const val VIEW_ORIENTATION = "orientation"
private const val IS_FLAG_SECURE = "is_flag_secure"
private const val IS_MANUALLY_INVOKED = "manually_captured"

private const val SR_FILE_NAME = "sr_%s.jpeg"
private const val NULL_SCREEN_FALLBACK = "NA"

private const val PACKAGE_SEPARATOR = "."

class SRScreenshotLog private constructor(
    bitmap: Bitmap?,
    val fileName: String,
    val screenName: String,//View
    val fullScreenName: String,//com.example.View
    val isScreenSecure: Boolean,
    @Orientation val viewOrientation: String,
    override val timestamp: Long,
    val isMetadataStored: Boolean,
    private var isManuallyInvoked: Boolean
) : SRLog {
    class Builder {
        private var screenshotBitmap: Bitmap? = null
        private var fullScreenName: String? = null

        @Orientation
        private var viewOrientation: String? = null
        private var timestamp: Long = 0L
        private var isScreenSecured = false
        private var isMetadataStored = false
        private var fileName: String? = null
        private var screenName: String? = null
        private var isManuallyInvoked = false

        fun setBitmap(bitmap: Bitmap?) = apply { screenshotBitmap = bitmap }

        fun setScreenView(fullScreenName: String?) = apply { this.fullScreenName = fullScreenName }

        fun setViewOrientation(@Orientation viewOrientation: String) =
            apply { this.viewOrientation = viewOrientation }

        fun setTimestamp(timestamp: Long) = apply { this.timestamp = timestamp }

        fun setScreenSecure(isScreenSecure: Boolean) =
            apply { this.isScreenSecured = isScreenSecure }

        fun setMetadataStored(isMetadataStored: Boolean) =
            apply { this.isMetadataStored = isMetadataStored }

        fun setManuallyInvoked(manuallyInvoked: Boolean) = apply {
            this.isManuallyInvoked = manuallyInvoked
        }

        private val isValid
            get() = fullScreenName != null &&
                    viewOrientation != null &&
                    timestamp != 0L

        fun fillFromInstance(origin: SRScreenshotLog) = apply {
            screenshotBitmap = origin.bitmap
            isMetadataStored = origin.isMetadataStored
            timestamp = origin.timestamp
            fullScreenName = origin.fullScreenName
            viewOrientation = origin.viewOrientation
            fileName = origin.fileName
            screenName = origin.screenName
            isScreenSecured = origin.isScreenSecure
        }

        fun build(): SRScreenshotLog? {
            return takeIf { isValid }?.let {
                val fileName = fileName ?: SR_FILE_NAME.format(timestamp)
                val screenName = screenName ?: run {
                    fullScreenName
                        ?.split(PACKAGE_SEPARATOR)
                        ?.lastOrNull()
                        ?: NULL_SCREEN_FALLBACK
                }

                SRScreenshotLog(
                    bitmap = screenshotBitmap,
                    fileName = fileName,
                    screenName = screenName,
                    fullScreenName = this.fullScreenName ?: NULL_SCREEN_FALLBACK,
                    viewOrientation = this.viewOrientation!!,
                    timestamp = this.timestamp,
                    isScreenSecure = isScreenSecured,
                    isMetadataStored = isMetadataStored,
                    isManuallyInvoked =  isManuallyInvoked
                )
            }
        }
    }

    var bitmap = bitmap
        private set
    override val logType: String = SCREENSHOT
    override val srJsonRep: JSONObject
        get() = JSONObject().apply {
            put(TIMESTAMP_KEY, timestamp)
            put(LOG_TYPE_KEY, logType)
            put(SCREENSHOT_ID, fileName)
            put(SCREEN_NAME, screenName)
            put(SCREEN_LONG_NAME, fullScreenName)
            put(VIEW_ORIENTATION, viewOrientation)
            put(IS_FLAG_SECURE, isScreenSecure)
            takeIf { isManuallyInvoked }?.let {
                put(IS_MANUALLY_INVOKED, isManuallyInvoked)
            }
        }

    fun scaleBitmap(scaler: BitmapScaler) {
        bitmap = bitmap?.let { scaler.scale(it) }
    }

    fun nullifyBitmap() {
        bitmap = null
    }
}