package com.instabug.library.sessionreplay

import com.instabug.library.sessionreplay.SRLoggingController.Status
import com.instabug.library.sessionreplay.bitmap.BitmapCompressor
import com.instabug.library.sessionreplay.bitmap.BitmapScaler
import com.instabug.library.sessionreplay.model.SRLog
import com.instabug.library.sessionreplay.model.SRScreenshotLog
import com.instabug.library.sessionreplay.monitoring.SRLoggingMonitor
import com.instabug.library.util.extenstions.defensiveExecute
import com.instabug.library.util.extenstions.getOrLogAndReport
import com.instabug.library.util.extenstions.runOrLogAndReport
import com.instabug.library.util.extenstions.runOrLogError
import com.instabug.library.util.threading.OrderedExecutorService

fun interface SRLogStore {
    operator fun invoke(log: SRLog)
}

interface BlockingSRLogStore : SRLogStore {
    fun blockingStore(log: SRLog): Boolean
}

class ControlledSRLogStore(
    private val executor: OrderedExecutorService,
    private val filesDirectory: SRFilesDirectory,
    private val loggingController: SRLoggingController,
    private val loggingMonitor: SRLoggingMonitor
) : BlockingSRLogStore {
    override fun invoke(log: SRLog) {
        executor.defensiveExecute(
            key = SRExecutionQueues.Main,
            errorMessage = "Failure while storing log",
            tag = SR_LOG_TAG
        ) { internalInvoke(log) }
    }

    override fun blockingStore(log: SRLog): Boolean = internalInvoke(log)

    private fun internalInvoke(log: SRLog): Boolean {
        loggingController.getStatusForLog(log)
            .also { outcome -> loggingMonitor.trackLog(log, outcome) }
            .takeIf { outcome -> outcome == Status.EligibleForStoring } ?: return false
        return runCatching {
            log.let(::WriteSRLogOperation)
                .let(filesDirectory::operateOnCurrent).get()
                ?.also(loggingController::onLogStored)
                ?.let { true } ?: false
        }.onFailure { t -> loggingMonitor.analyzeAndTrackException(t) }
            .getOrLogAndReport(false, "Error while storing log in SR", tag = SR_LOG_TAG)
    }
}

fun interface SRScreenshotStore {
    operator fun invoke(log: SRScreenshotLog)
}

class ControlledSRScreenshotStore(
    private val sessionReplayStore: BlockingSRLogStore,
    private val scalar: BitmapScaler,
    private val sessionReplayDirectory: SRFilesDirectory,
    private val compressor: BitmapCompressor,
    private val executor: OrderedExecutorService,
    private val loggingController: SRLoggingController,
    private val loggingMonitor: SRLoggingMonitor
) : SRScreenshotStore {
    override fun invoke(log: SRScreenshotLog) {
        runOrLogError(errorMessage = SOMETHING_WENT_WRONG_WHILE_SAVING_SR_SCREENSHOT_ERROR_MESSAGE) {
            log.scaleBitmap(scalar)
            executor.defensiveExecute(
                key = SRExecutionQueues.Main,
                errorMessage = "Failure while storing screenshot",
                tag = SR_LOG_TAG
            ) { internalInvoke(log) }
        }
    }

    private fun internalInvoke(log: SRScreenshotLog) {
        val isMetadataStored = log.isMetadataStored || sessionReplayStore.blockingStore(log)
        val isScreenshotEligibleForStoring =
            loggingController.getStatusForScreenshot(log)
                .also { outcome -> loggingMonitor.trackScreenshot(outcome) }
                .let { outcome -> outcome == Status.EligibleForStoring }
        if (!isMetadataStored || !isScreenshotEligibleForStoring) return
        runCatching {
            sessionReplayDirectory.operateOnCurrent(SaveScreenshotOp(log, compressor))
                .get()?.let(loggingController::onScreenshotStored)
        }.onFailure { t -> loggingMonitor.analyzeAndTrackException(t) }
            .runOrLogAndReport("Error while storing screenshot in SR", tag = SR_LOG_TAG)
    }
}