package com.instabug.library.networkDiagnostics.manager

import androidx.annotation.VisibleForTesting
import androidx.annotation.WorkerThread
import com.instabug.library.Constants
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventSubscriber
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.core.eventbus.eventpublisher.IBGDisposable
import com.instabug.library.networkDiagnostics.NetworkDiagnosticsConstants
import com.instabug.library.networkDiagnostics.caching.NetworkDiagnosticsCachingManager
import com.instabug.library.networkDiagnostics.configuration.NetworkDiagnosticsConfigurationProvider
import com.instabug.library.networkDiagnostics.model.NetworkDiagnostics
import com.instabug.library.networkDiagnostics.model.NetworkDiagnosticsCallback
import com.instabug.library.networkDiagnostics.model.NetworkDiagnosticsWrapper
import com.instabug.library.networkDiagnostics.model.getTotalRequestsCount
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.DateUtils
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.threading.OrderedExecutorService
import com.instabug.library.util.toFormattedString
import java.util.concurrent.Executor
import java.util.concurrent.ScheduledExecutorService
import java.util.concurrent.ScheduledFuture
import java.util.concurrent.TimeUnit

interface NetworkDiagnosticsManager {
    fun init()
    fun onNetworkRequestFailed()
    fun onNetworkRequestSucceeded()
    fun onSessionEnded(crashed: Boolean)
    fun onCallbackProvided(callback: NetworkDiagnosticsCallback)
    fun onNetworkDiagnosticsCallbacksDisabled()
}

class NetworkDiagnosticsManagerImpl(
    private val executor: OrderedExecutorService,
    private val configurationProvider: NetworkDiagnosticsConfigurationProvider,
    private val cachingManager: NetworkDiagnosticsCachingManager,
    private val scheduledExecutor: ScheduledExecutorService,
    private val mainThreadExecutor: Executor
) : NetworkDiagnosticsManager {

    companion object {
        const val EXECUTOR_KEY = "NetworkDiagnostics"
        const val REQUESTS_NUMBER_THRESHOLD = 5
        const val TIME_THRESHOLD_MILLIS = 10 * 1000L
        const val FAILURE_PERCENTAGE_THRESHOLD = 0.7F
        const val DEFAULT_DATE_FORMAT = "dd-MM-yyyy"
    }

    @VisibleForTesting
    var networkDiagnosticsWrapper: NetworkDiagnosticsWrapper? = null

    @VisibleForTesting
    var pendingRequestToBeDumpedCount = 0

    @VisibleForTesting
    var scheduledFuture: ScheduledFuture<*>? = null

    @VisibleForTesting
    var coreEventsDisposable: IBGDisposable? = null

    override fun init() = postOnExecutor {
        if (configurationProvider.networkDiagnosticsEnabled) {
            InstabugSDKLogger.v(Constants.LOG_TAG, NetworkDiagnosticsConstants.LogMessages.LOADING_CACHED_DATA)
            networkDiagnosticsWrapper = cachingManager.loadFromDisk()
            subscribeOnSessionEvents()
            migrateTodayIfNeeded()
        }
    }

    override fun onNetworkRequestFailed() = postOnExecutor {
        if (configurationProvider.networkDiagnosticsEnabled) {
            networkDiagnosticsWrapper?.let {
                pendingRequestToBeDumpedCount++
                networkDiagnosticsWrapper = it.copy(
                    today = it.today.copy(failCount = it.today.failCount + 1)
                )
                InstabugSDKLogger.d(Constants.LOG_TAG, "ND: Number of failed requests increased: ${networkDiagnosticsWrapper?.today}")

                dumpDataIfNeeded()
            }

        }
    }

    override fun onNetworkRequestSucceeded() = postOnExecutor {
        if (configurationProvider.networkDiagnosticsEnabled) {
            networkDiagnosticsWrapper?.let {
                pendingRequestToBeDumpedCount++
                networkDiagnosticsWrapper = it.copy(
                    today = it.today.copy(successCount = it.today.successCount + 1)
                )
                InstabugSDKLogger.d(Constants.LOG_TAG, "ND: Number of succeeded requests increased: ${networkDiagnosticsWrapper?.today}")
                dumpDataIfNeeded()
            }

        }
    }

    override fun onSessionEnded(crashed: Boolean) {
        InstabugSDKLogger.v(
            Constants.LOG_TAG,
            NetworkDiagnosticsConstants.LogMessages.SESSION_ENDED
        )
        InstabugSDKLogger.d(Constants.LOG_TAG, "ND: $networkDiagnosticsWrapper")

        val dumpDataRunnable = {
            cancelTimedDumping()
            dumpDataToDisk()
        }
        if (crashed) {
            dumpDataRunnable.invoke()
        } else {
            postOnExecutor(dumpDataRunnable)
        }
    }

    override fun onCallbackProvided(callback: NetworkDiagnosticsCallback) {
        networkDiagnosticsWrapper?.let { wrapper ->
            val totalRequestsCount = wrapper.lastActiveDay.getTotalRequestsCount()
            if (isFailurePercentageThresholdReached(totalRequestsCount, wrapper.lastActiveDay.failCount)) {
                val dateString = wrapper.lastActiveDay.date.toFormattedString(DEFAULT_DATE_FORMAT)
                InstabugSDKLogger.v(
                    Constants.LOG_TAG,
                    NetworkDiagnosticsConstants.LogMessages.CALLING_ON_READY
                        .replace("\"s1\"", dateString)
                        .replace("\"s2\"", "$totalRequestsCount")
                        .replace("\"s3\"", "${wrapper.lastActiveDay.failCount}")
                )

                mainThreadExecutor.execute {
                    callback.onReady(
                        date = dateString, totalRequestsCount, wrapper.lastActiveDay.failCount
                    )
                }
                networkDiagnosticsWrapper = wrapper.copy(
                    lastActiveDay = wrapper.lastActiveDay.copy(successCount = 0, failCount = 0)
                )
            }
        }
    }

    private fun isFailurePercentageThresholdReached(totalCount: Int, failCount: Int): Boolean {
        return totalCount > 0 && failCount.toFloat() / totalCount >= FAILURE_PERCENTAGE_THRESHOLD
    }

    override fun onNetworkDiagnosticsCallbacksDisabled() {
        cachingManager.clearCachedDate()
    }

    private fun migrateTodayIfNeeded() {
        networkDiagnosticsWrapper?.let { wrapper ->
            if (wrapper.today.date != DateUtils.getTodayDateWithoutTime()) {
                networkDiagnosticsWrapper = NetworkDiagnosticsWrapper(
                    today = NetworkDiagnostics(),
                    lastActiveDay = wrapper.today
                )
            }
        }
    }


    @VisibleForTesting
    fun subscribeOnSessionEvents() {
        coreEventsDisposable = IBGCoreEventSubscriber.subscribe(this::handleSDKCoreEvents)
    }

    private fun handleSDKCoreEvents(coreEvent: IBGSdkCoreEvent) {
        if (coreEvent == IBGSdkCoreEvent.V3Session.V3SessionFinished) {
            onSessionEnded(SettingsManager.getInstance().isCrashedSession)
        }
    }

    private fun postOnExecutor(runnable: Runnable) {
        executor.execute(EXECUTOR_KEY, runnable)
    }

    private fun dumpDataIfNeeded() {
        cancelTimedDumping()
        if (pendingRequestToBeDumpedCount >= REQUESTS_NUMBER_THRESHOLD) {
            InstabugSDKLogger.v(
                Constants.LOG_TAG,
                NetworkDiagnosticsConstants.LogMessages.REQUEST_THRESHOLD_DUMPING
            )
            InstabugSDKLogger.d(Constants.LOG_TAG, "ND: $networkDiagnosticsWrapper")
            dumpDataToDisk()
        } else {
            scheduleTimedDumping()
        }
    }

    @WorkerThread
    private fun dumpDataToDisk() {
        if (pendingRequestToBeDumpedCount > 0) {
            networkDiagnosticsWrapper?.let {
                cachingManager.dumpToDisk(it)
                resetPendingRequestsNumber()
            }
        }
    }


    private fun resetPendingRequestsNumber() {
        pendingRequestToBeDumpedCount = 0

    }

    private fun cancelTimedDumping() {
        scheduledFuture?.cancel(false)
        scheduledFuture = null
    }

    @VisibleForTesting
    fun scheduleTimedDumping() {
        scheduledFuture = scheduledExecutor.schedule({
            postOnExecutor {
                InstabugSDKLogger.v(Constants.LOG_TAG, NetworkDiagnosticsConstants.LogMessages.TIMED_DUMPING)
                InstabugSDKLogger.d(Constants.LOG_TAG, "ND: $networkDiagnosticsWrapper")
                dumpDataToDisk()
            }
        }, TIME_THRESHOLD_MILLIS, TimeUnit.MILLISECONDS)
    }
}

