package com.instabug.library.migration;

import android.content.Context;

import com.instabug.library.BuildConfig;
import com.instabug.library.Constants;
import com.instabug.library.internal.storage.cache.Cache;
import com.instabug.library.internal.storage.cache.CacheManager;
import com.instabug.library.internal.storage.cache.UserAttributesCacheManager;
import com.instabug.library.internal.storage.cache.db.userAttribute.UserAttributeCacheManager;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;

import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Map;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import io.reactivexport.Observable;
import io.reactivexport.ObservableEmitter;
import io.reactivexport.ObservableOnSubscribe;

import static com.instabug.library.internal.storage.cache.UserAttributesCacheManager.USER_ATTRIBUTES_DISK_CACHE_KEY;
import static com.instabug.library.internal.storage.cache.UserAttributesCacheManager.USER_ATTRIBUTES_MEMORY_CACHE_KEY;

/**
 Created by Barakat on 22/03/2019
 */
public class UserAttributeMigration extends AbstractMigration {
    private static final String USER_ATTR_MIGRATION = "user_attributes_migration";
    @Nullable
    private WeakReference<Context> context;

    public UserAttributeMigration() {
        super(USER_ATTR_MIGRATION);
    }

    @Override
    public int getMigrationVersion() {
        return 1;
    }

    @Override
    public void initialize(@NonNull Context context) {
        this.context = new WeakReference<>(context);
    }

    @Override
    public boolean shouldMigrate() {
        boolean isOldCacheExist = UserAttributesCacheManager.getAll() != null;
        InstabugSDKLogger.v(Constants.LOG_TAG, "Checking if old cache is existing and it's returning " + isOldCacheExist);
        return isOldCacheExist;
    }

    @Override
    public Observable<AbstractMigration> migrate() {
        return Observable.create(new ObservableOnSubscribe<AbstractMigration>() {
            @Override
            public void subscribe(ObservableEmitter<AbstractMigration> emitter) {
                if (context != null && context.get() != null) {
                    migrateOldCacheToDB();
                    emitter.onNext(UserAttributeMigration.this);
                    emitter.onComplete();
                }
            }
        });
    }

    private void migrateOldCacheToDB() {
        HashMap<String, String> userAttributes = UserAttributesCacheManager.getAll();
        if (userAttributes != null) {
            for (Map.Entry<String, String> entry : userAttributes.entrySet()) {
                UserAttributeCacheManager.insert(entry.getKey(), entry.getValue());
            }
        }
        Cache cache = CacheManager.getInstance().getCache(USER_ATTRIBUTES_MEMORY_CACHE_KEY);
        if (cache != null) {
            CacheManager.getInstance().deleteCache(cache.getId());
        }

        Cache diskCache = CacheManager.getInstance().getCache(USER_ATTRIBUTES_DISK_CACHE_KEY);
        if (diskCache != null) {
            CacheManager.getInstance().deleteCache(diskCache.getId());
            CacheManager.getInstance().invalidateCache(diskCache);
        }
    }

    @Override
    public void doPreMigration() {

    }

    @Override
    public void doAfterMigration() {
        SettingsManager.getInstance().setCurrentSDKVersion(BuildConfig.SDK_VERSION);
    }
}
