package com.instabug.library.logging.disklogs

import android.annotation.SuppressLint
import android.text.format.DateUtils
import com.instabug.library.internal.resolver.LoggingSettingResolver
import com.instabug.library.model.LoggingSettings
import com.instabug.library.util.FileUtils
import com.instabug.library.util.TimeUtils
import java.io.File
import java.io.IOException

object LogFilesHelper {

    private const val FILE_SEPARATOR = "/"


    @JvmStatic
    fun fileReachedSingleFileSizeLimit(logFile: File): Boolean {
        val singleFileLimit: Long =
            LoggingSettingResolver.getInstance().loggingSettings?.singleFileLimit
                ?: LoggingSettings.DEFAULT_SIZE_LIMIT / LoggingSettings.DEFAULT_TODAY_FILE_COUNT
        return FileUtils.getSize(logFile) >= singleFileLimit
    }

    @JvmStatic
    fun directoryReachedSizeLimit(logsDirectory: File?): Boolean {
        if(logsDirectory != null) {
            val directoryLimit =
                LoggingSettingResolver.getInstance().loggingSettings?.sizeLimit
                    ?: LoggingSettings.DEFAULT_SIZE_LIMIT
            return FileUtils.getSize(logsDirectory) >= directoryLimit
        }
        return false
    }

    @JvmStatic
    fun deleteOldestFileFromDirectory(logsDirectory: File?) {
        val files = logsDirectory?.listFiles()
        if (files != null && files.isNotEmpty()) {
            val filesList = mutableListOf(*files)
            FileUtils.sortByLastModifiedAsc(filesList)
            filesList[0].delete()
        }
    }


    /**
     * A method to get the timestamp from the file name.
     *
     * @param file
     * @return file timestamp if a proper file provided, otherwise, return -1
     */

    @JvmStatic
    fun getTimeStampFromFileName(file: File?): Long {
        return if (file == null) -1 else try {
            var fileName = file.name
            if (fileName.contains(LoggingFileProvider.LOGS_EXTENSION))
                fileName = fileName.replace(LoggingFileProvider.LOGS_EXTENSION, "")
            fileName.toLong()
        } catch (ex: Exception) {
            -1
        }
    }

    /**
     * A method to check if the file is today's file by checking the file timestamp.
     *
     * @param file
     * @return true if it's today's file, false otherwise.
     */
    @JvmStatic
    fun isTodayFile(file: File?): Boolean {
        return if(file != null) {
            val fileCreatedAt = getTimeStampFromFileName(file)
            DateUtils.isToday(fileCreatedAt)
        } else {
            false
        }
    }


    @JvmStatic
    fun getTodayFileFromDirectory(logsDir: File?): File? {
        if (logsDir != null) {
            val files = logsDir.listFiles()
            if (files != null && files.isNotEmpty()) {
                val filesList = mutableListOf(*files)
                FileUtils.sortByLastModifiedAsc(filesList)
                if(isTodayFile(filesList[filesList.lastIndex])) {
                    return filesList[filesList.lastIndex]
                }
            }
            return createTodayFile(logsDir)
        } else {
            return null
        }
    }


    /**
     * A method to create today file if the file doesn't exist then create new file with the timestamp as
     * its name, otherwise, return the existing file.
     *
     * @param dir the directory that holds the file.
     * @return
     * @throws IOException
     */
    @SuppressLint("VisibleForTests")
    @JvmStatic
    @Throws(IOException::class)
    fun createTodayFile(dir: File?): File? {
        if (dir != null) {
            val file = File(
                dir.absolutePath
                        + FILE_SEPARATOR + TimeUtils.currentTimeMillis()
                    .toString() + LoggingFileProvider.LOGS_EXTENSION
            )
            if (!file.exists()) {
                val newFile = file.createNewFile()
                return if (newFile) file else null
            }
            return file
        }
        return null
    }
}