package com.instabug.library.logging;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserEventEntry.COLUMN_EVENT_IDENTIFIER;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserEventEntry.COLUMN_EVENT_LOGGING_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserEventEntry.COLUMN_IS_ANONYMOUS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserEventEntry.COLUMN_UUID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserEventEntry.TABLE_NAME;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.Constants;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.HashMap;
import java.util.Map;

public final class UserEventsDbHelper {

    private UserEventsDbHelper() {
        // Utility classes should not be instantiatable
    }

    public static long insertOrUpdate(@NonNull String identifier, int count, @NonNull String uuid, boolean anonymous) {
        long rowId = insertWithOnConflict(identifier, count, uuid, anonymous);
        if (rowId < 1) {
            rowId = update(identifier, count, uuid, anonymous);
        }
        return rowId;
    }

    public static int getOccurrencesCount(@NonNull String identifier, @NonNull String uuid) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String[] columns = {COLUMN_EVENT_LOGGING_COUNT};
        String selection = COLUMN_EVENT_IDENTIFIER + " = ? AND " + COLUMN_UUID + " = ? ";
        String[] selectionArgs = {identifier, uuid};
        int count = 0;
        Cursor cursor = null;
        try {
            cursor = db.query(TABLE_NAME, columns, selection, selectionArgs, null, null, null);
            if (cursor != null && cursor.getCount() > 0) {
                cursor.moveToFirst();
                int countColumnIndex = cursor.getColumnIndex(COLUMN_EVENT_LOGGING_COUNT);
                count = cursor.getInt(countColumnIndex);
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "retrieve occurrences count failed: ", e);
            IBGDiagnostics.reportNonFatal(e, "retrieve user events occurrences count failed: " + e.getMessage());
        } finally {
            if (cursor != null) {
                cursor.close();
            }
            db.close();
        }
        return count;
    }

    public static Map<String, Integer> retrieveAll(@NonNull String uuid) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String[] columns = {COLUMN_EVENT_IDENTIFIER, COLUMN_EVENT_LOGGING_COUNT};
        String selection = COLUMN_UUID + " = ? ";
        String[] selectionArgs = {uuid};
        Map<String, Integer> userEvents = new HashMap<>();
        Cursor cursor = null;
        try {
            cursor = db.query(TABLE_NAME, columns, selection, selectionArgs, null, null, null);
            if (cursor != null && cursor.getCount() > 0) {
                int identifierIndex = cursor.getColumnIndex(COLUMN_EVENT_IDENTIFIER);
                int countIndex = cursor.getColumnIndex(COLUMN_EVENT_LOGGING_COUNT);
                while (cursor.moveToNext()) {
                    String identifier = cursor.getString(identifierIndex);
                    int count = cursor.getInt(countIndex);
                    userEvents.put(identifier, count);
                }
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Retrieving user events failed due to: " + e.getMessage());
            IBGDiagnostics.reportNonFatal(e, "Retrieving user events failed due to: " + e.getMessage());
        } finally {
            if (cursor != null) {
                cursor.close();
            }
            db.close();
        }
        return userEvents;
    }

    public static Map<String, Integer> retrieveAllAnonymous() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String[] columns = {COLUMN_EVENT_IDENTIFIER, COLUMN_EVENT_LOGGING_COUNT};
        String selection = COLUMN_IS_ANONYMOUS + " = ? ";
        String[] selectionArgs = {"1"};
        Map<String, Integer> userEvents = new HashMap<>();
        Cursor cursor = null;
        try {
            cursor = db.query(TABLE_NAME, columns, selection, selectionArgs, null, null, null);
            if (cursor != null && cursor.getCount() > 0) {
                int identifierIndex = cursor.getColumnIndex(COLUMN_EVENT_IDENTIFIER);
                int countIndex = cursor.getColumnIndex(COLUMN_EVENT_LOGGING_COUNT);
                while (cursor.moveToNext()) {
                    String identifier = cursor.getString(identifierIndex);
                    int count = cursor.getInt(countIndex);
                    userEvents.put(identifier, count);
                }
            }
        } catch (Exception e){
            InstabugSDKLogger.e(Constants.LOG_TAG, "Retrieving anonymous user events failed due to: " + e.getMessage());
            IBGDiagnostics.reportNonFatal(e, "Retrieving anonymous user events failed due to: " + e.getMessage());
        } finally {
            if (cursor != null) {
                cursor.close();
            }
            db.close();
        }
        return userEvents;
    }

    public static void deleteAll() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();
            db.delete(TABLE_NAME, null, null);
            db.setTransactionSuccessful();
        } catch (Exception e){
            InstabugSDKLogger.e(Constants.LOG_TAG, "Deleting user events failed due to: " + e.getMessage());
            IBGDiagnostics.reportNonFatal(e, "Deleting user events failed due to: " + e.getMessage());
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    public static void deleteAll(String uuid) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_UUID + "=?";
        String[] whereArgs = new String[]{uuid};
        try {
            db.beginTransaction();
            db.delete(TABLE_NAME, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } catch (Exception e){
            InstabugSDKLogger.e(Constants.LOG_TAG, "Deleting user events failed due to: " + e.getMessage());
            IBGDiagnostics.reportNonFatal(e, "Deleting user events failed due to: " + e.getMessage());
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    public static void deleteAllAnonymous() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_IS_ANONYMOUS + " = ? ";
        String[] whereArgs = new String[]{"1"};
        try {
            db.beginTransaction();
            db.delete(TABLE_NAME, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } catch (Exception e){
            InstabugSDKLogger.e(Constants.LOG_TAG, "Deleting anonymous user events failed due to: " + e.getMessage());
            IBGDiagnostics.reportNonFatal(e, "Deleting anonymous user events failed due to: " + e.getMessage());
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    public static boolean exists(@NonNull String identifier, @NonNull String uuid) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String[] columns = {COLUMN_EVENT_LOGGING_COUNT};
        String selection = COLUMN_EVENT_IDENTIFIER + " = ? AND " + COLUMN_UUID + " = ? ";
        String[] selectionArgs = {identifier, uuid};
        Cursor cursor = null;
        try {
            cursor = db.query(TABLE_NAME, columns, selection, selectionArgs, null, null, null);
            return cursor != null && cursor.getCount() > 0;
        } catch (Exception e){
            InstabugSDKLogger.e(Constants.LOG_TAG, "Retrieving user events failed due to: " + e.getMessage());
            IBGDiagnostics.reportNonFatal(e, "Retrieving user events failed due to: " + e.getMessage());
        } finally {
            if (cursor != null) {
                cursor.close();
            }
            db.close();
        }

        return false;
    }

    @VisibleForTesting
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    static long insertWithOnConflict(String identifier, int count, String uuid, boolean anonymous) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            ContentValues values = map(identifier, count, uuid, anonymous);
            db.beginTransaction();
            long rowId = db.insertWithOnConflict(TABLE_NAME, null, values);
            db.setTransactionSuccessful();
            return rowId;
        } catch (Exception e){
            InstabugSDKLogger.e(Constants.LOG_TAG, "Inserting user events failed due to: " + e.getMessage());
            IBGDiagnostics.reportNonFatal(e, "Inserting user events failed due to: " + e.getMessage());
        } finally {
            db.endTransaction();
            db.close();
        }

        return -1;
    }

    @VisibleForTesting
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    static long update(String identifier, int count, String uuid, boolean anonymous) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_EVENT_IDENTIFIER + "=? AND "
                + COLUMN_UUID + "=?";
        String[] whereArgs = new String[]{identifier, uuid};
        try {
            db.beginTransaction();
            ContentValues values = map(identifier, count, uuid, anonymous);
            long rowId = db.update(TABLE_NAME, values, whereClause, whereArgs);
            db.setTransactionSuccessful();
            return rowId;
        } catch (Exception e){
            InstabugSDKLogger.e(Constants.LOG_TAG, "Updating user event failed due to: " + e.getMessage());
            IBGDiagnostics.reportNonFatal(e, "Updating user event failed due to: " + e.getMessage());
        } finally {
            db.endTransaction();
            db.close();
        }
        return -1;
    }

    @VisibleForTesting
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    static ContentValues map(String identifier, int count, String uuid, boolean anonymous) {
        ContentValues contentValues = new ContentValues();
        contentValues.put(COLUMN_EVENT_IDENTIFIER, identifier);
        contentValues.put(COLUMN_EVENT_LOGGING_COUNT, count);
        contentValues.put(COLUMN_UUID, uuid);
        contentValues.put(COLUMN_IS_ANONYMOUS, anonymous);
        return contentValues;
    }
}
