package com.instabug.library.internal.storage.cache.db.migrations;

import android.database.sqlite.SQLiteDatabase;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.internal.storage.cache.db.InstabugDBVersions;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.COMMA_SEP;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.INTEGER_TYPE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_APP_TOKEN;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_APP_VERSION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_CRASH_REPORTING_ENABLED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_DEVICE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_DURATION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_OS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_SDK_VERSION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_STARTED_AT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_SYNC_STATUS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_USERS_PAGE_ENABLED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_USER_ATTRIBUTES;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_USER_ATTRIBUTES_KEYS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_USER_EMAIL;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_USER_EVENTS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_USER_EVENTS_KEYS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_USER_NAME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_UUID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.TEXT_TYPE;

/**
 * Allows the database to migrate from version 16 to version 17
 * <p>
 * Schemas are defined in {@link InstabugDbContract}. Any changes to these schemas will require
 * a migration.
 *
 * @see InstabugDBVersions
 */
public class Migration_19_20 implements Migration {

    @Override
    public void migrate(SQLiteDatabase db) {
        // SQLite does not fully support ALTER TABLE statements. The workaround to altering an
        // existing column's type is to create a new table with a temporary name, copy everything to
        // the new table, drop the old table and then rename the new table to its original name.
        String oldTableName = SessionEntry.TABLE_NAME;
        String newTableName = SessionEntry.TABLE_NAME + "_temp";
        // create new table with temp name.
        createTable(db, newTableName);
        // copy values from old table to new table.
        copyValues(db, oldTableName, newTableName, getColumnsCommaSeparated());
        // drop old table.
        dropTable(db, oldTableName);
        // rename the new table to its original name.
        renameTable(db, newTableName, oldTableName);
    }

    @VisibleForTesting
    public void createTable(@NonNull SQLiteDatabase db, @NonNull String tableName) {
        db.execSQL("CREATE TABLE IF NOT EXISTS " + tableName +
                " ( " + COLUMN_ID + /*new column type*/TEXT_TYPE + COMMA_SEP
                + COLUMN_STARTED_AT + INTEGER_TYPE + COMMA_SEP
                + COLUMN_DURATION + INTEGER_TYPE + COMMA_SEP
                + COLUMN_USER_ATTRIBUTES + TEXT_TYPE + COMMA_SEP
                + COLUMN_USER_EVENTS + TEXT_TYPE + COMMA_SEP
                + COLUMN_USER_ATTRIBUTES_KEYS + TEXT_TYPE + COMMA_SEP
                + COLUMN_USER_EVENTS_KEYS + TEXT_TYPE + COMMA_SEP
                + COLUMN_USER_EMAIL + TEXT_TYPE + COMMA_SEP
                + COLUMN_UUID + TEXT_TYPE + COMMA_SEP
                + COLUMN_USER_NAME + TEXT_TYPE + COMMA_SEP
                + COLUMN_OS + TEXT_TYPE + COMMA_SEP
                + COLUMN_APP_TOKEN + TEXT_TYPE + COMMA_SEP
                + COLUMN_DEVICE + TEXT_TYPE + COMMA_SEP
                + COLUMN_SDK_VERSION + TEXT_TYPE + COMMA_SEP
                + COLUMN_APP_VERSION + TEXT_TYPE + COMMA_SEP
                + COLUMN_CRASH_REPORTING_ENABLED + INTEGER_TYPE + COMMA_SEP
                + COLUMN_USERS_PAGE_ENABLED + INTEGER_TYPE + COMMA_SEP
                + COLUMN_SYNC_STATUS + INTEGER_TYPE + " )");
    }

    @VisibleForTesting
    public void copyValues(SQLiteDatabase db, String oldTable, String newTable, String columns) {
        db.execSQL("INSERT INTO " + newTable
                + " SELECT " + columns
                + " FROM " + oldTable);
    }

    @VisibleForTesting
    public void dropTable(SQLiteDatabase db, String tableName) {
        db.execSQL("DROP TABLE IF EXISTS " + tableName);
    }

    @VisibleForTesting
    public void renameTable(SQLiteDatabase db, String oldTableName, String newTableName) {
        db.execSQL("ALTER TABLE " + oldTableName + " RENAME TO " + newTableName);
    }

    @VisibleForTesting
    public String getColumnsCommaSeparated() {
        return COLUMN_ID + COMMA_SEP
                + COLUMN_STARTED_AT + COMMA_SEP
                + COLUMN_DURATION + COMMA_SEP
                + COLUMN_USER_ATTRIBUTES + COMMA_SEP
                + COLUMN_USER_EVENTS + COMMA_SEP
                + COLUMN_USER_ATTRIBUTES_KEYS + COMMA_SEP
                + COLUMN_USER_EVENTS_KEYS + COMMA_SEP
                + COLUMN_USER_EMAIL + COMMA_SEP
                + COLUMN_UUID + COMMA_SEP
                + COLUMN_USER_NAME + COMMA_SEP
                + COLUMN_OS + COMMA_SEP
                + COLUMN_APP_TOKEN + COMMA_SEP
                + COLUMN_DEVICE + COMMA_SEP
                + COLUMN_SDK_VERSION + COMMA_SEP
                + COLUMN_APP_VERSION + COMMA_SEP
                + COLUMN_CRASH_REPORTING_ENABLED + COMMA_SEP
                + COLUMN_USERS_PAGE_ENABLED + COMMA_SEP
                + COLUMN_SYNC_STATUS;
    }
}
