package com.instabug.library.internal.storage.cache.db;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.DatabaseUtils;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.Arrays;

/**
 * Created by tarek on 3/15/17.
 */

public class SQLiteDatabaseWrapper {

    @Nullable
    private SQLiteDatabase database;
    private final SQLiteOpenHelper databaseHelper;
    @Nullable
    private Boolean databaseTransactionsEnabled;

    SQLiteDatabaseWrapper(SQLiteOpenHelper helper) {
        databaseHelper = helper;
    }

    // Synchronized by the caller
    // todo move this to the DB init and remove this method from all sub classes
    void open() {
        // Opening new database
        try {
            if (database != null && database.isOpen()) return;

            database = databaseHelper.getWritableDatabase();
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while opening the DB: " + e.getMessage(), e);
            IBGDiagnostics.reportNonFatal(e, "Error while opening the DB: " + e.getMessage());
        } catch (OutOfMemoryError oom) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while opening the DB: " + oom.getMessage(), oom);
            IBGDiagnostics.reportNonFatal(oom, "Error while opening the DB: " + oom.getMessage());
        }
    }

    // todo to be removed from all sub classes
    public synchronized void close() {
    }

    @SuppressWarnings("ConstantConditions")
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public void beginTransaction() {
        try {
            if (database != null && database.isOpen()) {
                if (isDatabaseTransactionsEnabled()) {
                    database.beginTransaction();
                }
            } else {
                logOperationFailedWarning("DB transaction failed");
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "DB transaction failed due to: " + ex.getMessage());
            logOperationFailedWarning("DB transaction failed due to:" + ex.getMessage());
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, "DB transaction failed due to: " + oom.getMessage());
            logOperationFailedWarning("\"DB transaction failed due to an Exception due to: " + oom.getMessage());
        }
    }

    private synchronized boolean isDatabaseTransactionsEnabled() {
        if (databaseTransactionsEnabled == null && Instabug.getApplicationContext() != null) {
            databaseTransactionsEnabled = !InstabugCore.isDatabaseTransactionDisabled();
        }
        return databaseTransactionsEnabled != null ? databaseTransactionsEnabled : false;
    }

    private synchronized void logOperationFailedWarning(String message) {
        if (database == null) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Attempted to do operation on an uninitialized " +
                    "database. Falling back silently");
        } else if (!database.isOpen()) {
            InstabugSDKLogger.e(Constants.LOG_TAG
                    , "Attempted to do operation on a closed database. " +
                            "Falling back silently");
        } else {
            InstabugSDKLogger.e(Constants.LOG_TAG, message + " ,Falling back silently");
        }
    }

    @SuppressWarnings("ConstantConditions")
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public long insert(@NonNull String table, @Nullable String nullColumnHack,
                       @NonNull ContentValues values) {
        try {
            if (database != null && database.isOpen()) {
                return database.insert(table, nullColumnHack, values);
            } else {
                logOperationFailedWarning("DB insertion failed");
                return -1;
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "DB insertion failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            logOperationFailedWarning("DB insertion failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            return -1;
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, "DB insertion failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            logOperationFailedWarning("DB insertion failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            return -1;
        }
    }

    @SuppressWarnings("ConstantConditions")
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public long insertWithOnConflict(@NonNull String tableName,
                                     @Nullable String nullColumnHack,
                                     @NonNull ContentValues values) {
        try {
            if (database != null && database.isOpen()) {
                return database.insertWithOnConflict(tableName, nullColumnHack, values, SQLiteDatabase.CONFLICT_IGNORE);
            } else {
                logOperationFailedWarning("DB insertion with on conflict failed");
                return -1;
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "DB insertion with on conflict failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            logOperationFailedWarning("DB insertion with on conflict failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            return -1;
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, "DB insertion with on conflict failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            logOperationFailedWarning("DB insertion with on conflict failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            return -1;
        }
    }

    @SuppressWarnings("ConstantConditions")
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public long insertWithOnConflictReplace(@NonNull String tableName,
                                            @Nullable String nullColumnHack,
                                            @NonNull ContentValues values) {
        try {
            if (database != null && database.isOpen()) {
                return database.insertWithOnConflict(tableName, nullColumnHack, values, SQLiteDatabase.CONFLICT_REPLACE);
            } else {
                logOperationFailedWarning("DB insertion with on conflict replace failed");
                return -1;
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "DB insertion with on conflict replace failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            logOperationFailedWarning("DB insertion with on conflict replace failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            return -1;
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, "DB insertion with on conflict replace failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            logOperationFailedWarning("DB insertion with on conflict replace failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            return -1;
        }
    }

    @SuppressWarnings("ConstantConditions")
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public void execSQL(@NonNull String sql) {
        try {
            if (database != null && database.isOpen()) {
                database.execSQL(sql);
            } else {
                logOperationFailedWarning("DB execution a sql failed");
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            logOperationFailedWarning("DB execution a sql failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, "DB execution a sql failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            logOperationFailedWarning("DB execution a sql failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
        }
    }

    @SuppressWarnings("ConstantConditions")
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public void execSQL(@NonNull String sql, @NonNull Object[] args) {
        try {
            if (database != null && database.isOpen()) {
                database.execSQL(sql, args);
            } else {
                logOperationFailedWarning("DB execution a sql failed");
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            logOperationFailedWarning("DB execution a sql failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, "DB execution a sql failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            logOperationFailedWarning("DB execution a sql failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
        }
    }

    @SuppressWarnings("ConstantConditions")
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public void setTransactionSuccessful() {
        try {
            if (database != null && database.isOpen()) {
                if (isDatabaseTransactionsEnabled()) {
                    database.setTransactionSuccessful();
                }
            } else {
                logOperationFailedWarning("DB transaction not successful");
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "DB transaction not successful due to: " + ex.getMessage());
            logOperationFailedWarning("DB transaction not successful due to: " + ex.getMessage());
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, "DB transaction not successful due to: " + oom.getMessage());
            logOperationFailedWarning("DB transaction not successful due to: " + oom.getMessage());
        }
    }

    @SuppressWarnings("ConstantConditions")
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public synchronized void endTransaction() {
        try {
            if (database != null && database.isOpen()) {
                if (isDatabaseTransactionsEnabled()) {
                    database.endTransaction();
                }
            } else {
                logOperationFailedWarning("DB end transaction not successful");
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "DB end transaction not successful due to: " + ex.getMessage());
            logOperationFailedWarning("DB end transaction not successful due to: " + ex.getMessage());
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, "DB end transaction not successful due to: " + oom.getMessage());
            logOperationFailedWarning("B end transaction not successful due to: " + oom.getMessage());
        }
    }

    @Nullable
    @SuppressWarnings("ConstantConditions")
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public Cursor rawQuery(@NonNull String sql, @Nullable String[] selectionArgs) {
        try {
            if (database != null && database.isOpen()) {
                return database.rawQuery(sql, selectionArgs);
            } else {
                logOperationFailedWarning("DB raw query failed");
                return null;
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "DB raw query failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            logOperationFailedWarning("DB raw query failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            return null;
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, "DB raw query failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            logOperationFailedWarning("DB raw query failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            return null;
        }
    }

    @SuppressWarnings("ConstantConditions")
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public int delete(@NonNull String table,
                      @Nullable String whereClause,
                      @Nullable String[] whereArgs) {
        try {
            if (database != null && database.isOpen()) {
                return database.delete(table, whereClause, whereArgs);
            } else {
                logOperationFailedWarning("DB deletion failed");
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "DB raw query failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            logOperationFailedWarning("DB deletion failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, "DB raw query failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            logOperationFailedWarning("DB deletion failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
        }
        return 0;
    }

    @SuppressWarnings("ConstantConditions")
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public int update(@NonNull String table,
                      @NonNull ContentValues values,
                      @Nullable String whereClause,
                      @Nullable String[] whereArgs) {
        try {
            if (database != null && database.isOpen()) {
                return database.update(table, values, whereClause, whereArgs);
            } else {
                logOperationFailedWarning("DB update failed");
                return -1;
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "DB update failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            logOperationFailedWarning("DB update failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            return -1;
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, "DB update failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            logOperationFailedWarning("DB update failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            return -1;
        }
    }

    @Nullable
    @SuppressWarnings("ConstantConditions")
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public Cursor query(String table, @Nullable String[] columns,
                        @Nullable String selection,
                        @Nullable String[] selectionArgs,
                        @Nullable String groupBy,
                        @Nullable String having,
                        @Nullable String orderBy) {
        try {
            if (database != null && database.isOpen()) {
                return database.query(table, columns, selection, selectionArgs, groupBy, having, orderBy);
            } else {
                logOperationFailedWarning("DB query failed");
                return null;
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "DB query failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            logOperationFailedWarning("DB query failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            return null;
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, "DB query failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            logOperationFailedWarning("DB query failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            return null;
        }
    }

    @Nullable
    @SuppressWarnings("ConstantConditions")
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public Cursor query(String table,
                        @Nullable String[] columns,
                        @Nullable String selection,
                        @Nullable String[] selectionArgs,
                        @Nullable String groupBy,
                        @Nullable String having,
                        @Nullable String orderBy,
                        @Nullable String limit) {
        try {
            if (database != null && database.isOpen()) {
                return database.query(table, columns, selection, selectionArgs, groupBy, having, orderBy,
                        limit);
            } else {
                logOperationFailedWarning("DB query failed");
                return null;
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "DB query failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            logOperationFailedWarning("DB query failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            return null;
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, "DB query failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            logOperationFailedWarning("DB query failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            return null;
        }
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public long queryNumEntries(@NonNull String tableName) {
        try {
            if (database != null && database.isOpen()) {
                return DatabaseUtils.queryNumEntries(database, tableName);
            } else {
                logOperationFailedWarning("DB query num entries failed");
                return -1;
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "DB query num entries failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            logOperationFailedWarning("DB query num entries failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            return -1;
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, "DB query num entries failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            logOperationFailedWarning("DB query num entries failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            return -1;
        }
    }

    @VisibleForTesting
    public void closeDB() {
        if (databaseHelper != null) {
            databaseHelper.close();
        }
        if (database != null) {
            database.close();
            database = null;
        }
    }

    public boolean deleteDatabase(Context context) {
        closeDB();
        return context.deleteDatabase(databaseHelper.getDatabaseName());
    }
}
