package com.instabug.library.internal.storage.cache;

import android.annotation.SuppressLint;

import com.instabug.library.Constants;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import static com.instabug.library.internal.storage.cache.UserAttributesCacheManager.USER_ATTRIBUTES_CACHE_KEY;
import static com.instabug.library.internal.storage.cache.UserAttributesCacheManager.USER_ATTRIBUTES_DISK_CACHE_FILE_NAME;
import static com.instabug.library.internal.storage.cache.UserAttributesCacheManager.USER_ATTRIBUTES_DISK_CACHE_KEY;
import static com.instabug.library.internal.storage.cache.UserAttributesCacheManager.USER_ATTRIBUTES_MEMORY_CACHE_KEY;

/**
 * @author mSobhy
 */
public class CacheManager {
    /**
     * Key of default <b>In-memory</b> cache to be used to store non-specific cache values
     */
    public static final String DEFAULT_IN_MEMORY_CACHE_KEY = "DEFAULT_IN_MEMORY_CACHE_KEY";

    /**
     * List of caches that currently exist
     */
    private final List<Cache> caches;
    /**
     * Singleton instance of {@code CacheManager}
     */
    private static CacheManager INSTANCE;

    /**
     * Default constructor instantiating CacheManager with default {@link InMemoryCache}
     */
    public CacheManager() {
        caches = new ArrayList<>();
        Cache<String, String> defaultInMemoryCache = new InMemoryCache<>
                (DEFAULT_IN_MEMORY_CACHE_KEY);
        caches.add(defaultInMemoryCache);
    }

    /**
     * Creates instance <b>if non-exists</b>
     *
     * @return singleton value of CacheManager
     */
    public synchronized static CacheManager getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new CacheManager();
        }
        return INSTANCE;
    }

    /**
     * Find and get cache with {@code cacheId} if exists
     *
     * @param cacheId cache id to get
     * @return {@link Cache} object if a Cache with this ID exists, null otherwise
     * @see Cache
     */
    @Nullable
    public Cache getCache(String cacheId) {
        synchronized (caches) {
            for (Cache cache : caches) {
                if (cache.getId().equals(cacheId)) {
                    return cache;
                }
            }
        }
        InstabugSDKLogger.v(Constants.LOG_TAG, "No cache with this ID was found " + cacheId + " returning null");
        return null;
    }

    /**
     * Add a new {@link Cache} to the existing caches
     *
     * @param cache cache to be added
     * @return existing {@code Cache} object if another cache with the same ID exists or same
     * object if it doesn't exist
     * @see Cache
     */
    public Cache addCache(Cache cache) {
        final Cache existingCache = getCache(cache.getId());
        if (existingCache != null) {
            return existingCache;
        } else {
            synchronized (caches) {
                caches.add(cache);
            }
            return cache;
        }
    }

    /**
     * Removes cache matching {@code cacheId} if it exists in {@link #caches}
     *
     * @param cacheId cache id to remove
     * @return try if it was successfully removed, false otherwise
     */
    public boolean deleteCache(String cacheId) {
        final Cache existingCache = getCache(cacheId);
        if (existingCache != null) {
            synchronized (caches) {
                return caches.remove(existingCache);
            }
        } else {
            InstabugSDKLogger.v(Constants.LOG_TAG, "No cache was this ID was found " + cacheId + " to be " +
                    "deleted");
            return false;
        }
    }

    /**
     * Check if cache with this ID exists or not
     *
     * @param cacheId cache id to check for
     * @return true if cache with id = cacheId exists, false otherwise
     */
    public boolean cacheExists(String cacheId) {
        return getCache(cacheId) != null;
    }

    /**
     * Subscribes to updates on Cache matching ID = cacheID
     *
     * @param cacheId              cache to subscribe to
     * @param cacheChangedListener listener to be notified about updates
     * @return true if subscribed, false if its already subscribed
     * @throws IllegalArgumentException if the cache doesn't exist
     * @see #cacheExists(String)
     */
    public boolean subscribe(String cacheId, CacheChangedListener cacheChangedListener) {
        if (cacheExists(cacheId)) {
            Cache cache = getCache(cacheId);
            if (cache != null) {
                return cache.addOnCacheChangedListener(cacheChangedListener);
            }
        }
        throw new IllegalArgumentException("No cache exists with this ID to subscribe to");
    }

    /**
     * Un-subscribes from updates to Cache matching ID = cacheID
     *
     * @param cacheId              cache to un-subscribe from
     * @param cacheChangedListener listener to be removed
     * @return true if un-subscribed, false if cache doesn't exist
     */
    public boolean unSubscribe(String cacheId, CacheChangedListener cacheChangedListener) {
        if (cacheExists(cacheId)) {
            Cache cache = getCache(cacheId);
            if (cache != null) {
                return cache.removeOnCacheChangedListener(cacheChangedListener);
            }
        }
        return false;
    }

    /**
     * Remove <b>ALL</b> (On-Disk, In-Memory, etc...) caches
     */
    public void invalidateAllCaches() {
        synchronized (caches) {
            for (Cache cache : caches) {
                invalidateCache(cache);
            }
        }
        InstabugSDKLogger.v(Constants.LOG_TAG, "All caches have been invalidated");
    }

    /**
     * Remove On-Disk caches
     */
    public void invalidateCache(Cache cache) {
        if (cache != null) {
            cache.invalidate();
            InstabugSDKLogger.v(Constants.LOG_TAG, "Cache with the ID " + cache.getId() + " have been invalidated");
        }
    }

    /**
     * Remove <b>ALL</b> (On-Disk, In-Memory, etc...) caches
     * except user attributes
     */
    public void invalidateAllCachesForIdentifyingUsers() {
        synchronized (caches) {
            for (Cache cache : caches) {
                if (cache.getId().equals(USER_ATTRIBUTES_MEMORY_CACHE_KEY)
                        || cache.getId().equals(USER_ATTRIBUTES_DISK_CACHE_KEY)) {
                    continue;
                }
                invalidateCache(cache);
            }
        }
        InstabugSDKLogger.v(Constants.LOG_TAG, "All caches have been invalidated except user attributes cache");
    }

    /**
     * Remove <b>ALL</b> (On-Disk, In-Memory, etc...) caches
     */
    public void invalidateAllCachesButUserAttributes() {
        synchronized (caches) {
            for (Cache cache : caches) {
                if (!cache.getId().equalsIgnoreCase(USER_ATTRIBUTES_CACHE_KEY)
                        && !cache.getId().equalsIgnoreCase(USER_ATTRIBUTES_DISK_CACHE_FILE_NAME)
                        && !cache.getId().equalsIgnoreCase(USER_ATTRIBUTES_DISK_CACHE_KEY)
                        && !cache.getId().equalsIgnoreCase(USER_ATTRIBUTES_MEMORY_CACHE_KEY)) {
                    invalidateCache(cache);
                }
            }
        }
        InstabugSDKLogger.v(Constants.LOG_TAG, "All caches have been invalidated");
    }

    /**
     * Migrates all values from cache whose key is {@code migratingFromCacheKey} to cache whose
     * key is/will be {@code migratingToCacheKey}
     *
     * @param migratingFromCacheKey key of cache to migrate data from
     * @param migratingToCacheKey   key of cache to migrate data to <b>will be created if doesn't
     *                              exist</b>
     * @param extractor             an instance of
     *                              {@link com.instabug.library.internal.storage.cache.CacheManager.KeyExtractor} handling how
     *                              to extract keys from values
     * @param <K>                   Cache key class type
     * @param <V>                   Cache value class type
     * @throws IllegalArgumentException if no cache matching {@code migratingFromCacheKey} was found
     */
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public <K, V> void migrateCache(@NonNull String migratingFromCacheKey, @NonNull String
            migratingToCacheKey, KeyExtractor<K, V> extractor) throws IllegalArgumentException {
        Cache<K, V> migratingFromCache = getCache(migratingFromCacheKey);
        Cache<K, V> migratingToCache = getCache(migratingToCacheKey);
        InstabugSDKLogger.v(Constants.LOG_TAG, "Caches to be migrated " + migratingFromCache + " - " +
                migratingToCache);
        if (migratingFromCache == null) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "No cache with these key(" + migratingFromCacheKey + ") was found to migrate from");
            return;
        }
        if (migratingToCache == null) {
            migratingToCache = new InMemoryCache<>(migratingToCacheKey);
            addCache(migratingToCache);
        }
        migrateCache(migratingFromCache, migratingToCache, extractor);
    }

    /**
     * Migrates all values from cache whose key is {@code migratingFromCacheKey} to cache whose
     * key is/will be {@code migratingToCacheKey}
     *
     * @param migratingFromCache cache to migrate data from
     * @param migratingToCache   cache to migrate data to <b>will be created if doesn't exist</b>
     * @param extractor          an instance of
     *                           {@link com.instabug.library.internal.storage.cache.CacheManager.KeyExtractor} handling how
     *                           to extract keys from values
     * @param <K>                Cache key class type
     * @param <V>                Cache value class type
     * @throws IllegalArgumentException if no cache matching {@code migratingFromCacheKey} was found
     */
    public <K, V> void migrateCache(@NonNull Cache<K, V> migratingFromCache, @NonNull Cache<K, V>
            migratingToCache, KeyExtractor<K, V> extractor) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Invalidated migratingTo cache");

        if (migratingToCache == null || migratingFromCache == null) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "cache migration process got failure," + " migratingToCache: "
                    + migratingToCache + ", migratingFromCache: " + migratingFromCache);
            return;
        }

        migratingToCache.invalidate();

        final List<V> migratingFromCachedValues = migratingFromCache.getValues();
        if (migratingFromCachedValues == null || migratingFromCachedValues.isEmpty()) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Cache to migrate from doesn't contain any elements, not " +
                    "going further with the migration");
            return;
        }

        for (V value : migratingFromCachedValues) {
            if (value != null) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "Adding value " + value + " with key " + extractor
                        .extractKey(value));
                migratingToCache.put(extractor.extractKey(value), value);
            }
        }
    }

    public static abstract class KeyExtractor<K, V> {
        public abstract K extractKey(V value);
    }
}
