package com.instabug.library.internal.storage.cache;

import android.annotation.SuppressLint;
import android.content.Context;
import android.os.Environment;

import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.model.AssetEntity;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.service.FilesService;
import com.instabug.library.util.FileUtils;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.File;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author mesbah
 */
@SuppressLint("RV_RETURN_VALUE_IGNORED_BAD_PRACTICE")
public class AssetsCacheManager {

    private static final String ASSETS_MEMORY_CACHE_KEY = "assets_memory_cache";
    private static final Map<String, DownloadingEntity> currentDownloadingFiles = new ConcurrentHashMap<>();

    @Nullable
    public static AssetCache getCache() {
        if (!CacheManager.getInstance().cacheExists(ASSETS_MEMORY_CACHE_KEY)) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "In-memory assets cache not found, create it");
            CacheManager.getInstance().addCache(new AssetCache(ASSETS_MEMORY_CACHE_KEY));
            InstabugSDKLogger.v(Constants.LOG_TAG, "In-memory assets created successfully");
        }
        InstabugSDKLogger.v(Constants.LOG_TAG, "In-memory assets cache found");
        return (AssetCache) CacheManager.getInstance().getCache(ASSETS_MEMORY_CACHE_KEY);
    }

    public static AssetEntity createEmptyEntity(Context context, String fileUrl, AssetEntity.AssetType assetType) {
        File file = new File(getCacheDirectory(context), String.valueOf(fileUrl.hashCode()));
        return new AssetEntity(String.valueOf(fileUrl.hashCode()), assetType, fileUrl, file);
    }

    public static void getAssetEntity(AssetEntity assetEntity, OnDownloadFinished onDownloadFinished) {
        AssetCache assetCache = getCache();
        AssetEntity retrievedAssetEntity;
        if (assetCache != null) {
            retrievedAssetEntity = assetCache.get(assetEntity.getKey());
        } else {
            retrievedAssetEntity = null;
        }

        if (retrievedAssetEntity != null && retrievedAssetEntity.getFile() != null && retrievedAssetEntity.getFile().exists()) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Get file from cache");
            onDownloadFinished.onSuccess(retrievedAssetEntity);
        } else if (isDownloading(assetEntity.getKey())) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "File currently downloading, wait download to finish");
            waitDownloadToFinish(assetEntity, onDownloadFinished);
        } else {
            InstabugSDKLogger.v(Constants.LOG_TAG, "File not exist download it");
            downloadAssetEntity(assetEntity, onDownloadFinished);
        }
    }

    public static void downloadAssetEntity(final AssetEntity assetEntity, OnDownloadFinished onDownloadFinished) {
        if (assetEntity != null && onDownloadFinished != null) {
            DownloadingEntity downloadingEntity = new DownloadingEntity();
            downloadingEntity.setAssetEntity(assetEntity);
            List<WeakReference<OnDownloadFinished>> waitingList = downloadingEntity.getWaitingList();
            waitingList.add(new WeakReference<>(onDownloadFinished));
            downloadingEntity.setWaitingList(waitingList);
            AssetEntity entity = downloadingEntity.getAssetEntity();
            if (entity != null) {
                currentDownloadingFiles.put(entity.getKey(), downloadingEntity);
            }
            FilesService.getInstance().downloadFile(
                    assetEntity, new Request.Callbacks<AssetEntity, Throwable>() {
                        @Override
                        public void onSucceeded(@Nullable AssetEntity responseAssetEntity) {
                            if (responseAssetEntity != null) {
                                addAssetEntity(responseAssetEntity);
                                notifyDownloadFinishedSuccessfully(responseAssetEntity);
                            }
                        }

                        @Override
                        public void onFailed(Throwable error) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "downloading asset entity got error: ", error);
                            notifyDownloadFailed(assetEntity, error);
                        }
                    });
        }
    }


    public static void addAssetEntity(@Nullable AssetEntity assetEntity) {
        AssetCache assetCache = getCache();
        if (assetCache != null && assetEntity != null) {
            assetCache.put(assetEntity.getKey(), assetEntity);
        }
    }

    public static boolean isDownloading(String key) {
        return (currentDownloadingFiles.get(key) != null);
    }

    public static void waitDownloadToFinish(AssetEntity assetEntity, OnDownloadFinished onDownloadFinished) {
        DownloadingEntity downloadingEntity = currentDownloadingFiles.get(assetEntity.getKey());
        if (downloadingEntity != null) {
            List<WeakReference<OnDownloadFinished>> waitingList = downloadingEntity.getWaitingList();
            waitingList.add(new WeakReference<>(onDownloadFinished));
            downloadingEntity.setWaitingList(waitingList);
        }
    }

    public static void notifyDownloadFinishedSuccessfully(@Nullable AssetEntity assetEntity) {
        if (assetEntity != null) {
            DownloadingEntity downloadingEntity = currentDownloadingFiles.get(assetEntity.getKey());
            if (downloadingEntity != null) {
                for (WeakReference<OnDownloadFinished> onDownloadFinishedReference : downloadingEntity.getWaitingList()) {
                    if (onDownloadFinishedReference != null) {
                        OnDownloadFinished onDownloadFinished = onDownloadFinishedReference.get();
                        if (onDownloadFinished != null) {
                            onDownloadFinished.onSuccess(assetEntity);
                            currentDownloadingFiles.remove(assetEntity.getKey());
                        }
                    }
                }
            }
        }
    }

    public static void notifyDownloadFailed(AssetEntity assetEntity, Throwable error) {
        DownloadingEntity downloadingEntity = currentDownloadingFiles.get(assetEntity.getKey());
        if (downloadingEntity != null) {
            for (WeakReference<OnDownloadFinished> onDownloadFinishedReference : downloadingEntity.getWaitingList()) {
                if (onDownloadFinishedReference != null) {
                    OnDownloadFinished onDownloadFinished = onDownloadFinishedReference.get();
                    if (onDownloadFinished != null) {
                        onDownloadFinished.onFailed(error);
                        currentDownloadingFiles.remove(assetEntity.getKey());
                    }
                }
            }
        }
    }

    public static void clearRedundantFiles(Context context) {
        try {
            File[] files = getCacheDirectory(context).listFiles();
            if (files == null)
                return;
            for (File f : files)
                f.delete();
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while cleaning up cache directory", e);
        }
    }

    public static void cleanUpCache(Context context) {
        if (CacheManager.getInstance().cacheExists(ASSETS_MEMORY_CACHE_KEY)) {
            Cache cache = CacheManager.getInstance().getCache(ASSETS_MEMORY_CACHE_KEY);
            if (cache != null) {
                cache.invalidate();
            }
        }
        clearRedundantFiles(context);
    }

    public static File getCacheDirectory(Context context) {
        String directoryPath = context.getCacheDir().getAbsolutePath() + "/instabug/assetCache";
        File directory = new File(directoryPath);
        if (!directory.exists()) {
            boolean isCreated = directory.mkdirs();
            File noMediaFile = new File(directory, ".nomedia");
            try {
                noMediaFile.createNewFile();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return directory;
    }

    /**
     * Delete assets cache's external directory if it is exist.
     * This was introduced after migrating assets cache from external to internal storage.
     */
    public static void clearExternalCacheDir(Context context) {
        if (context != null
                && Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED)
                && context.getExternalCacheDir() != null
        ) {
            String directoryPath = context.getExternalCacheDir().getPath() + "/instabug";
            File externalCacheFile = new File(directoryPath);
            if (externalCacheFile.exists()) {
                FileUtils.deleteDirectory(externalCacheFile);
            }
        }
    }

    public static class DownloadingEntity {
        @Nullable
        public AssetEntity assetEntity;

        public List<WeakReference<OnDownloadFinished>> waitingList;

        public DownloadingEntity() {
            waitingList = new ArrayList<>();
        }

        public DownloadingEntity setAssetEntity(AssetEntity assetEntity) {
            this.assetEntity = assetEntity;
            return this;
        }

        public DownloadingEntity setWaitingList(List<WeakReference<OnDownloadFinished>> waitingList) {
            this.waitingList = waitingList;
            return this;
        }

        @Nullable
        public AssetEntity getAssetEntity() {
            return assetEntity;
        }

        public List<WeakReference<OnDownloadFinished>> getWaitingList() {
            return waitingList;
        }
    }

    public interface OnDownloadFinished {
        void onSuccess(AssetEntity assetEntity);

        void onFailed(Throwable error);
    }

}
