package com.instabug.library.featuresflags.mappers

import com.instabug.library.featuresflags.model.IBGFeatureFlag
import com.instabug.library.internal.storage.cache.dbv2.IBGCursor
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionFeaturesFlagsEntry.COLUMN_FEATURES_FLAGS_ARRAY
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionFeaturesFlagsEntry.COLUMN_SESSION_SERIAL
import com.instabug.library.util.extenstions.getLong
import com.instabug.library.util.extenstions.getString
import org.json.JSONArray
import org.json.JSONObject

fun IBGFeatureFlag.asJsonObject(): JSONObject {
    return JSONObject().apply {
        put("key", key)
        put("value", value)
    }
}

fun IBGCursor.toSessionsFeaturesFlags(): Map<Long, JSONArray> {
    val sessionFeatureFlagsMap = hashMapOf<Long, JSONArray>()
    while (moveToNext()) {
        val sessionSerial = getLong(COLUMN_SESSION_SERIAL)
        val featuresFlagsString = getString(COLUMN_FEATURES_FLAGS_ARRAY)
        val featuresFlags = JSONArray(featuresFlagsString)
        sessionFeatureFlagsMap[sessionSerial] = featuresFlags
    }
    close()
    return sessionFeatureFlagsMap
}

/**
 * Map the given features flags array to experiments using delimiter " -> " in case the
 * feature flag object has a key and value or a use key only if there is no value.
 * @param ffArray features flags JSON array that need to be mapped.
 *
 * @return A list of a unique string experiments.
 */
internal fun featuresFlagsAsExperiments(ffArray: JSONArray): List<String> {
    val featuresFlagsAsExperiments = mutableListOf<String>()
    for (elementIndex in 0 until ffArray.length()) {
        val featureFlagObject = ffArray.optJSONObject(elementIndex)
        val featureFlagKey = featureFlagObject.optString("key")
        val featureFlagValue =
            if (featureFlagObject.optString("value").isEmpty()) ""
            else " -> " + featureFlagObject.optString("value")
        featuresFlagsAsExperiments.add(featureFlagKey + featureFlagValue)
    }
    return featuresFlagsAsExperiments
}

/**
 * Map the given features flags array to a JSON object of unique elements.
 * @param ffArray features flags JSON array that need to be mapped.
 *
 * @return JSON object of unique features flags in a string representation.
 */
fun featuresFlagsAsJson(ffArray: JSONArray): JSONObject {
    val featuresFlags = JSONObject()
    for (elementIndex in 0 until ffArray.length()) {
        ffArray.optJSONObject(elementIndex).let {
            featuresFlags.put(it.optString("key"), it.optString("value"))
        }
    }
    return featuresFlags
}