package com.instabug.library.encryption.iv

import android.os.Build
import android.security.KeyPairGeneratorSpec
import android.util.Base64.*
import androidx.annotation.RequiresApi
import com.instabug.library.Constants
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.encryption.EncryptionPreferences
import com.instabug.library.internal.contentprovider.InstabugApplicationProvider
import com.instabug.library.util.InstabugSDKLogger
import java.io.ByteArrayInputStream
import java.io.ByteArrayOutputStream
import java.math.BigInteger
import java.security.KeyPairGenerator
import java.security.KeyStore
import java.security.SecureRandom
import java.util.*
import javax.crypto.Cipher
import javax.crypto.CipherInputStream
import javax.crypto.CipherOutputStream
import javax.security.auth.x500.X500Principal

object Post18IVGenerator {

    private const val RSA_MODE = "RSA/ECB/PKCS1Padding"
    private const val RSA_KEY_ALIAS = "iv_rsa_keys"
    private const val androidKeyStore = "AndroidKeyStore"

    private var keystore: KeyStore? = null

    init {
        if (keystore == null) {
            try {
                keystore = KeyStore.getInstance(androidKeyStore)
                keystore?.load(null)
            } catch (e: Exception) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Error while instantiating keystore", e)
                keystore = null
            }
        }
    }

    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    private fun generateRSAKeys() {
        InstabugApplicationProvider.getInstance()?.application?.let {
            try {
                val start: Calendar = Calendar.getInstance()
                val end: Calendar = Calendar.getInstance()
                //Keys will expire in this date
                end.add(Calendar.YEAR, 30)
                val spec: KeyPairGeneratorSpec = KeyPairGeneratorSpec.Builder(it)
                    .setAlias(RSA_KEY_ALIAS)
                    .setSubject(X500Principal("CN=$RSA_KEY_ALIAS"))
                    .setSerialNumber(BigInteger.ONE)
                    .setStartDate(start.time)
                    .setEndDate(end.time)
                    .build()
                val generator: KeyPairGenerator =
                    KeyPairGenerator.getInstance("RSA", androidKeyStore)

                generator.initialize(spec)
                generator.generateKeyPair()
            } catch (e: java.lang.Exception) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Error while generating RSA keys", e)
            }
        }
    }

    private fun rsaEncrypt(secret: ByteArray): String? {
        val privateKeyEntry: KeyStore.PrivateKeyEntry
        keystore?.let { keystore ->
            try {
                privateKeyEntry = keystore.getEntry(
                    RSA_KEY_ALIAS,
                    null
                ) as KeyStore.PrivateKeyEntry
                val inputCipher: Cipher =
                    Cipher.getInstance(RSA_MODE)
                inputCipher.init(Cipher.ENCRYPT_MODE, privateKeyEntry.certificate.publicKey)
                val outputStream = ByteArrayOutputStream()
                val cipherOutputStream = CipherOutputStream(outputStream, inputCipher)
                cipherOutputStream.write(secret)
                cipherOutputStream.close()
                val encrypted: ByteArray = outputStream.toByteArray()
                return encodeToString(encrypted, DEFAULT)

            } catch (exception: java.lang.Exception) {
                InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "Error while encrypting IV using RSA",
                    exception
                )
                IBGDiagnostics.reportNonFatal(exception, "Error while encrypting IV using RSA")
            } catch (oom: OutOfMemoryError) {
                InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "OOM while encrypting IV using RSA",
                    oom
                )
                IBGDiagnostics.reportNonFatal(oom, "OOM while encrypting IV using RSA")
            }
        }
        return ""
    }

    private fun rsaDecrypt(encrypted: String): ByteArray? {
        keystore?.let { keystore ->
            return try {
                val encryptedBytes: ByteArray = decode(encrypted, DEFAULT)
                val privateKeyEntry =
                    keystore.getEntry(RSA_KEY_ALIAS, null) as KeyStore.PrivateKeyEntry
                val output: Cipher = Cipher.getInstance(RSA_MODE)
                output.init(Cipher.DECRYPT_MODE, privateKeyEntry.privateKey)
                val cipherInputStream = CipherInputStream(
                    ByteArrayInputStream(encryptedBytes), output
                )
                val values: ArrayList<Byte> = ArrayList()
                var nextByte: Int
                while (cipherInputStream.read().also { nextByte = it } != -1) {
                    values.add(nextByte.toByte())
                }
                val bytes = ByteArray(values.size)
                for (i in bytes.indices) {
                    bytes[i] = values[i]
                }
                bytes
            } catch (e: java.lang.Exception) {
                InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "Error while decrypting encryption IV using RSA",
                    e
                )
                IBGDiagnostics.reportNonFatal(e, "Error while decrypting encryption IV using RSA")
                null
            } catch (oom: OutOfMemoryError) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "OOM while decrypting IV using RSA", oom)
                IBGDiagnostics.reportNonFatal(oom, "OOM while decrypting encryption IV using RSA")
                null
            }
        }
        return null
    }

    @JvmStatic
    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    fun getIV(): ByteArray? {
        if (EncryptionPreferences.getEncryptionIV().isEmpty()) {
            generateIV()
        }

        val decryptedIvBytes = rsaDecrypt(EncryptionPreferences.getEncryptionIV())
        return if (decryptedIvBytes != null && decryptedIvBytes.isNotEmpty()) {
            decryptedIvBytes
        } else {
            null
        }
    }

    @Synchronized
    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    private fun generateIV() {
        if (keystore != null && keystore?.containsAlias(RSA_KEY_ALIAS) != true) {
            generateRSAKeys()
        }

        val iv = ByteArray(12)
        SecureRandom().nextBytes(iv)

        val encryptedIV = rsaEncrypt(iv)
        EncryptionPreferences.saveEncryptionIV(encryptedIV ?: "")
    }
}