package com.instabug.library.diagnostics.nonfatals;

import android.content.Context;
import android.net.Uri;

import androidx.annotation.NonNull;

import com.instabug.library.Constants;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.diagnostics.nonfatals.cache.NonFatalCacheManager;
import com.instabug.library.diagnostics.nonfatals.di.ServiceLocator;
import com.instabug.library.diagnostics.nonfatals.model.NonFatal;
import com.instabug.library.diagnostics.nonfatals.model.Occurrence;
import com.instabug.library.diagnostics.nonfatals.settings.NonFatalSettings;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.internal.storage.operation.ReadStateFromFileDiskOperation;
import com.instabug.library.model.State;
import com.instabug.library.util.FileUtils;
import com.instabug.library.util.InstabugSDKLogger;

import org.jetbrains.annotations.NotNull;

import java.io.File;
import java.util.Iterator;
import java.util.List;


/**
 * Created by Shannan on 06/09/2021.
 */
public class NonFatalsManagerImpl implements NonFatalsManager {

    private static final String NON_FATAL_STATE_FILES_PREFIX = "files:non_fatal_state:";
    private static final String NON_FATAL_STATE = "non_fatal_state:";


    @NonNull
    private final NonFatalCacheManager cacheManager;
    @NonNull
    private final NonFatalSettings nonFatalSettings;

    public NonFatalsManagerImpl(@NonNull NonFatalCacheManager cacheManager,
                                @NonNull NonFatalSettings Settings) {
        this.cacheManager = cacheManager;
        this.nonFatalSettings = Settings;
    }

    @Override
    public void saveNonFatal(@NonNull @NotNull NonFatal nonFatal) {
        if (!nonFatalSettings.isEnabled()) {
            return;
        }
        if (IgnoreListHelper.shouldBeIgnored(nonFatal, nonFatalSettings.getIgnoreList())) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "NonFatal " + nonFatal.getExceptionType() + " - " + nonFatal.getFileName() + " was ignored");
            return;
        }
        cacheManager.saveNonFatal(nonFatal);
    }

    @Override
    public void clearCache() {
        DiskUtils.deleteNonfatalStateFiles();
        cacheManager.clearCache();
    }

    @Override
    public List<NonFatal> getNonFatals() {
        List<NonFatal> nonFatals = cacheManager.getAllNonFatals();
        try {
            Iterator<NonFatal> it = nonFatals.iterator();
            while (it.hasNext()) {
                NonFatal nonFatal = it.next();
                // Remove ignored non-fatals
                if (IgnoreListHelper.shouldBeIgnored(nonFatal, nonFatalSettings.getIgnoreList())) {
                    InstabugSDKLogger.v(Constants.LOG_TAG, "NonFatal " + nonFatal.getExceptionType() + " - "
                            + nonFatal.getFileName() + " won't be synced, as it is present in ignore list");
                    it.remove();
                } else {
                    List<Occurrence> occurrences = cacheManager.getNonFatalOccurrences(nonFatal.getId());
                    State nonFatalState = null;
                    for (Occurrence occurrence : occurrences) {
                        if (loadState(occurrence)) {
                            nonFatalState = occurrence.getState();
                            nonFatal.addOccurrence(occurrence);
                            nonFatal.setState(nonFatalState);
                        } else {
                            cacheManager.deleteOccurrence(occurrence.getStateFileUri());
                        }
                    }
                }
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "error while preparing non-fatals for sync", e);
        }
        return nonFatals;
    }

    private boolean loadState(Occurrence occurrence) {
        try {
            Context context = ServiceLocator.getContext();
            if (context != null && occurrence.getStateFileUri() != null) {
                State state = new State();
                Uri uri = Uri.parse(occurrence.getStateFileUri());
                String stateJson = DiskUtils.with(context)
                        .readOperation(new ReadStateFromFileDiskOperation(uri))
                        .execute();
                state.fromJson(stateJson);
                occurrence.setState(state);
                return true;
            }
            return false;
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while loading state for non fatal", e);
            return false;
        }
    }

    @Override
    public void cleanStaleStateFiles() {
        List<File> files = FileUtils.getStateFiles(NON_FATAL_STATE_FILES_PREFIX);
        List<String> cachedNonFatalsStateFiles = cacheManager.getStateFilesForAllOccurrences();
        if (!files.isEmpty()) {
            for (File stateFile : files) {
                try {
                    boolean isFileExistsInDB = false;
                    for (String filename : cachedNonFatalsStateFiles) {
                        if (filename.contains(stateFile
                                .getName()
                                .substring(stateFile.getName()
                                        .indexOf(NON_FATAL_STATE)
                                        + NON_FATAL_STATE.length() + 1))) {
                            isFileExistsInDB = true;
                            break;
                        }
                    }
                    if (!isFileExistsInDB) {
                        boolean isDeleted = stateFile.delete();
                        if (isDeleted) {
                            InstabugSDKLogger.v(Constants.LOG_TAG, "file " + stateFile.getName() + " is deleted");
                        } else {
                            InstabugSDKLogger.v(Constants.LOG_TAG, "file " + stateFile.getName() + " is not deleted");
                        }
                    }
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Error: " + e.getMessage() + " while cleaning stale non fatals state files");
                    IBGDiagnostics.reportNonFatal(e, "can't clean Stale non fatals State Files");
                }
            }

        }
    }

    @Override
    public void cleanNonFatalsWithNoOccurrences() {
        List<NonFatal> nonFatals = cacheManager.getAllNonFatals();
        for (NonFatal nonFatal: nonFatals) {
            List<Occurrence> occurrences = cacheManager.getNonFatalOccurrences(nonFatal.getId());
            if(occurrences.isEmpty()) {
                cacheManager.deleteNonFatal(nonFatal.getId());
            }
        }
    }
}
