package com.instabug.library.view

import android.content.Context
import android.graphics.Color
import android.view.View
import android.view.WindowManager
import android.widget.ProgressBar
import android.widget.TextView
import androidx.annotation.ColorInt
import androidx.appcompat.app.AlertDialog
import com.instabug.library.InstabugColorTheme
import com.instabug.library.R
import com.instabug.library.core.InstabugCore
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.UiUtils
import com.instabug.library.util.extenstions.setColor

/**
 * a replacement dialog for the deprecated {@link ProgressDialog}
 */
interface IBGProgressDialog {
    /**
     * @return true if the dialog is visible false other wise
     * */
    val isShowing: Boolean

    /**
     * shows the progress dialog if not shown
     * */
    fun show()

    /**
     *dismiss the progress dialog if is shown
     */
    fun dismiss()

    /**
     * sets the progress dialog message
     */
    fun setMessage(message: String)

    /**
     * sets progress bar color
     * the default color is {@link Color.BLACK} and {@link Color.LTGRAY} if dark mode enabled
     */
    fun setProgressColor(@ColorInt color: Int)

    class Builder {
        private var message: String = ""
        private var color: Int? = null
        private var isCancelable: Boolean = false
        private var theme: Int = R.style.InstabugDialogStyle

        fun setMessage(message: String): Builder = apply {
            this.message = message
        }

        fun setProgressColor(@ColorInt color: Int) = apply {
            this.color = color
        }

        fun setCancelable(isCancelable: Boolean) = apply {
            this.isCancelable = isCancelable
        }

        fun setTheme(theme: Int) = apply {
            this.theme = theme
        }

        fun build(context: Context): IBGProgressDialog = IBGProgressDialogImpl(
            context = context,
            theme = theme,
            progressMessage = message,
            progressColor = color,
            isCancelable = isCancelable
        )
    }
}

class IBGProgressDialogImpl(
    context: Context,
    progressColor: Int? = null,
    private val theme: Int = R.style.InstabugDialogStyle,
    private var progressMessage: String = "",
    private var isCancelable: Boolean = false
) : IBGProgressDialog {
    private var progressColor: Int = progressColor ?: if (isDarkMode) Color.LTGRAY else Color.BLACK
    private var progressBar: ProgressBar? = null
    private var progressMessageTV: TextView? = null

    private val isDarkMode
        get() = SettingsManager.getInstance().theme.let { it == InstabugColorTheme.InstabugColorThemeDark }
    private val dialog: AlertDialog by lazy {
        AlertDialog.Builder(context, theme).prepareDialog(context).create()
    }
    override val isShowing: Boolean get() = dialog.isShowing

    private fun AlertDialog.Builder.prepareDialog(context: Context) = apply {
        val view = View.inflate(context, R.layout.instabug_progress_dialog, null)
        setView(view)
        setCancelable(isCancelable)
        initViews(view)
    }

    private fun initViews(view: View) {
        progressMessageTV = view.findViewById(R.id.ib_progress_text)
        progressMessageTV?.text = progressMessage
        progressBar = view.findViewById(R.id.ib_progress_bar)
        progressBar?.setColor(progressColor)
    }

    override fun show() {
        if(InstabugCore.isFullScreen()) {
            dialog.window?.setFlags(
                WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE,
                WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE
            )
        }
        dialog.takeUnless { isShowing }?.show()
        if(InstabugCore.isFullScreen()) {
            dialog.window?.let {
                UiUtils.showInFullscreen(it)
                it.clearFlags(WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE)
            }
        }

    }


    override fun dismiss() {
        dialog.takeIf { isShowing }?.dismiss()
    }

    override fun setMessage(message: String) {
        message.takeUnless { message == progressMessage }?.also { progressMessage = message }
            ?.let { progressMessageTV?.setText(it) }
    }

    override fun setProgressColor(@ColorInt color: Int) {
        progressColor = color
        progressBar?.setColor(color)
    }

}
