package com.instabug.library.util;

import android.annotation.SuppressLint;
import android.graphics.Bitmap;
import android.media.MediaMetadataRetriever;
import android.text.TextUtils;

import com.instabug.library.Constants;
import com.instabug.library.util.threading.PoolProvider;

import java.io.File;
import java.io.IOException;

import androidx.annotation.Nullable;

/**
 * @author mSobhy
 */
public class VideoManipulationUtils {

    /**
     * Extracts {@link Bitmap} describing video frame at the exact time provided
     *
     * @param videoPath Path of video to extract bitmap from (P.S: unless this video is internal, you need to be granted WRITE_EXTERNAL_STORAGE permission)
     * @return Bitmap describing the requested frame (might be null)
     */
    @Nullable
    @SuppressLint("RESOURCE_LEAK")
    public static Bitmap extractVideoFrame(String videoPath) {
        Bitmap frame = null;
        MediaMetadataRetriever mediaMetadataRetriever = new MediaMetadataRetriever();
        try {
            mediaMetadataRetriever.setDataSource(videoPath);
            /*
            This method finds a
            representative frame at any time position if possible,
            and returns it as a bitmap. This is useful for generating a thumbnail
            */
            frame = mediaMetadataRetriever.getFrameAtTime();
        } catch (IllegalArgumentException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while extracting video frame", e);
        } finally {
            try {
                mediaMetadataRetriever.release();
            } catch (IOException e) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Error while releasing mediaMetadataRetriever" + e);
            }
        }
        return frame;
    }

    /**
     * Extracts {@link Bitmap} describing first video frame
     *
     * @param videoPath Path of video to extract bitmap from (P.S: unless this video is internal, you need to be granted WRITE_EXTERNAL_STORAGE permission)
     */
    public static void extractFirstVideoFrame(String videoPath, OnVideoFrameReady onVideoFrameReady) {
        PoolProvider.postIOTask(() -> {
            File videoFile = new File(videoPath);

            Bitmap frame = null;
            if (videoFile.exists()) {
                frame = extractVideoFrame(videoPath);
            }

            final Bitmap finalFrame = frame;
            PoolProvider.postMainThreadTask(() -> onVideoFrameReady.onReady(finalFrame));
        });
    }

    /**
     * Extracts duration describing the given video path
     *
     * @param videoPath Path of video to extract duration from (P.S: unless this video is internal, you need to be granted WRITE_EXTERNAL_STORAGE permission)
     * @return video duration or -1 if failed to extract it
     */
    @SuppressLint("RESOURCE_LEAK")
    public static long extractVideoDuration(String videoPath) throws IllegalArgumentException {
        MediaMetadataRetriever mediaMetadataRetriever = new MediaMetadataRetriever();
        long timeMilli = -1;
        try {
            mediaMetadataRetriever.setDataSource(videoPath);
            String time = mediaMetadataRetriever.extractMetadata(MediaMetadataRetriever.METADATA_KEY_DURATION);
            if (!TextUtils.isEmpty(time)) {
                timeMilli = Long.parseLong(time);
            }
        } catch (Exception e) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "Error while extracting video duration" + e);
        } finally {
            try {
                mediaMetadataRetriever.release();
            } catch (IOException e) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Error while releasing mediaMetadataRetriever" + e);
            }
        }
        return timeMilli;
    }
}
