package com.instabug.library.util

import android.os.Build
import android.view.View
import android.view.Window
import android.view.WindowInsets
import android.view.WindowInsetsController
import android.view.WindowManager
import android.widget.PopupWindow
import android.widget.Spinner
import com.instabug.library.Constants
import com.instabug.library.diagnostics.IBGDiagnostics

object UiUtils {

    @JvmStatic
    fun showInFullscreen(window: Window) {
        kotlin.runCatching {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
                val windowInsetsController = window.insetsController
                windowInsetsController?.let {
                    it.hide(WindowInsets.Type.statusBars())
                    it.hide(WindowInsets.Type.navigationBars())
                    it.systemBarsBehavior =
                        WindowInsetsController.BEHAVIOR_SHOW_TRANSIENT_BARS_BY_SWIPE
                    val decorView = window.decorView
                    decorView.setOnApplyWindowInsetsListener { v: View?, insets: WindowInsets ->
                        if (insets.isVisible(WindowInsets.Type.statusBars())
                            || insets.isVisible(WindowInsets.Type.navigationBars())
                        ) {
                            // Status bar is visible, block it again
                            it.hide(WindowInsets.Type.statusBars())
                            it.hide(WindowInsets.Type.navigationBars())
                        }
                        insets
                    }
                }
            } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                val decorView = window.decorView
                // Hide the status bar.
                val uiOptions =
                    View.SYSTEM_UI_FLAG_FULLSCREEN or View.SYSTEM_UI_FLAG_HIDE_NAVIGATION or View.SYSTEM_UI_FLAG_IMMERSIVE;

                decorView.systemUiVisibility = uiOptions
                decorView.setOnSystemUiVisibilityChangeListener { visibility ->
                    if (visibility and View.SYSTEM_UI_FLAG_FULLSCREEN == 0) {
                        // Immersive mode lost, reapply
                        decorView.systemUiVisibility = uiOptions
                    }
                }
            } else {
                window.setFlags(
                    WindowManager.LayoutParams.FLAG_FULLSCREEN,
                    WindowManager.LayoutParams.FLAG_FULLSCREEN
                )
            }
        }.onFailure {
            InstabugSDKLogger.w(
                Constants.LOG_TAG,
                "Couldn't enable fullscreen mode ${it.cause?.message}"
            )
            IBGDiagnostics.reportNonFatal(it, "Couldn't enable fullscreen mode")
        }
    }

    @JvmStatic
    fun avoidDropdownFocus(spinner: Spinner) {
        kotlin.runCatching {
            val isAppCompat = spinner is androidx.appcompat.widget.AppCompatSpinner
            val spinnerClass =
                if (isAppCompat) androidx.appcompat.widget.AppCompatSpinner::class.java else Spinner::class.java
            val popupWindowClass =
                if (isAppCompat) androidx.appcompat.widget.ListPopupWindow::class.java else android.widget.ListPopupWindow::class.java

            val listPopup = spinnerClass
                .getDeclaredField("mPopup")
                .apply { isAccessible = true }
                .get(spinner)
            if (popupWindowClass.isInstance(listPopup)) {
                val popup = popupWindowClass
                    .getDeclaredField("mPopup")
                    .apply { isAccessible = true }
                    .get(listPopup)
                if (popup is PopupWindow) {
                    popup.isFocusable = false
                }
            }
        }.onFailure {
            InstabugSDKLogger.w(
                Constants.LOG_TAG,
                "Couldn't disable dropdown focus ${it.cause?.message}"
            )
            IBGDiagnostics.reportNonFatal(it, "Couldn't disable dropdown focus")
        }
    }
}