package com.instabug.library.util;

import static com.instabug.library.core.InstabugCore.canPrintLog;

import android.content.Context;
import android.util.Log;

import androidx.annotation.NonNull;

import com.instabug.library.Constants;
import com.instabug.library.LogLevel;
import com.instabug.library.logging.disklogs.InstabugSDKDiskLogger;
import com.instabug.library.model.LogDescriptor;
import com.instabug.library.model.LoggingSettings;

public final class InstabugSDKLogger {

    private static final String LOG_TAG = "IB-";
    private volatile static InstabugSDKDiskLogger instabugSDKDiskLogger;

    private InstabugSDKLogger() {
    }

    public static void initLogger(@NonNull Context context) {
        if (instabugSDKDiskLogger == null) {
            instabugSDKDiskLogger = new InstabugSDKDiskLogger(context);
        }
    }

    /**
     * Adds protected log message to SDK logs without displaying
     * it in logcat
     *
     * @param tag        log tag
     * @param logMessage message to logged
     */
    public static void p(@NonNull String tag, @NonNull String logMessage) {
        if (logMessage == null) {
            return;
        }
        if (instabugSDKDiskLogger != null)
            instabugSDKDiskLogger.logBaseLevel(tag, logMessage, Thread.currentThread().getName(), System.currentTimeMillis());
    }

    /**
     * Prints a verbose message using Android {@link Log}
     *
     * @param tag        log tag
     * @param logMessage message to logged <b>will be chunk-ed if too long</b>
     */
    public static void v(@NonNull String tag, @NonNull String logMessage) {
        if (logMessage == null) {
            return;
        }

        try {
            if (canPrintLog(LogLevel.VERBOSE)) {
                if (logMessage.length() > 4000) {
                    int chunkCount = logMessage.length() / 4000;
                    Log.v(tag, "logMessage length = " + logMessage.length() + " divided to " + (chunkCount + 1) + " chunks");
                    for (int i = 0; i <= chunkCount; i++) {
                        int max = 4000 * (i + 1);
                        String message;
                        if (max >= logMessage.length()) {
                            message = logMessage.substring(4000 * i);
                        } else {
                            message = logMessage.substring(4000 * i, max);
                        }
                        Log.v(tag, "chunk " + (i + 1) + " of " + (chunkCount + 1) + ":\n" + message);
                    }
                } else {
                    Log.v(tag, logMessage);
                }
            }
            if (instabugSDKDiskLogger != null)
                instabugSDKDiskLogger.logVerboseLogs(tag, logMessage, Thread.currentThread().getName(), System.currentTimeMillis());
        } catch (Exception e) {
            e(LOG_TAG + "InstabugSDKLogger", e.getMessage(), e);
        }
    }

    /**
     * Prints a debug message using Android {@link Log}
     *
     * @param tag        log tag
     * @param logMessage message to logged <b>will be chunk-ed if too long</b>
     */
    public static void d(@NonNull String tag, @NonNull String logMessage) {
        if (logMessage == null) {
            return;
        }

        try {
            if (canPrintLog(LogLevel.DEBUG)) {
                if (logMessage.length() > 4000) {
                    int chunkCount = logMessage.length() / 4000;
                    Log.d(tag, "logMessage length = " + logMessage.length() + " divided to " + (chunkCount + 1) + " chunks");
                    for (int i = 0; i <= chunkCount; i++) {
                        int max = 4000 * (i + 1);
                        String message;
                        if (max >= logMessage.length()) {
                            message = logMessage.substring(4000 * i);
                        } else {
                            message = logMessage.substring(4000 * i, max);
                        }
                        Log.d(tag, "chunk " + (i + 1) + " of " + (chunkCount + 1) + ":\n" + message);
                    }
                } else {
                    Log.d(tag, logMessage);
                }
            }
            p(tag, logMessage);
        } catch (Exception e) {
            e(LOG_TAG + "InstabugSDKLogger", e.getMessage(), e);
        }
    }


    /**
     * Prints an info message using Android {@link Log}<br/>
     * <b>Should NOT be used for debug purposes as it's unstoppable</b>
     *
     * @param tag        log tag
     * @param logMessage message to logged
     */
    public static void i(@NonNull String tag, @NonNull String logMessage) {
        if (logMessage == null) {
            return;
        }
        try {
            if (canPrintLog(LogLevel.DEBUG)) {
                Log.i(tag, logMessage);
            }
            p(tag, logMessage);
        } catch (Exception e) {
            e(LOG_TAG + "InstabugSDKLogger", e.getMessage(), e);
        }
    }

    /**
     * Prints a warning message using Android {@link Log}<br/>
     * <b>Should NOT be used for debug purposes as it's unstoppable</b>
     *
     * @param tag        log taag
     * @param logMessage message to logged
     */
    public static void w(@NonNull String tag, @NonNull String logMessage) {
        if (logMessage == null) {
            return;
        }
        try {
            if (canPrintLog(LogLevel.ERROR)) {
                Log.w(tag, logMessage);
            }
            p(tag, logMessage);
        } catch (Exception e) {
            e(LOG_TAG + "InstabugSDKLogger", e.getMessage(), e);
        }
    }

    /**
     * Prints an error message using Android {@link Log}<br/>
     * <b>Should NOT be used for debug purposes as it's unstoppable</b>
     *
     * @param tag        log tag
     * @param logMessage message to logged
     */
    public static void e(@NonNull String tag, @NonNull String logMessage) {
        if (logMessage == null) {
            return;
        }

        try {
            if (canPrintLog(LogLevel.ERROR)) {
                Log.e(tag, logMessage);
            }
            p(tag, logMessage);
        } catch (Exception e) {
            Log.e(Constants.LOG_TAG, e.getMessage(), e);
        }
    }

    /**
     * Same as {@link #e(String, String)} but with a {@code throwable} to log exceptions as well<br/>
     * <b>Tip:</b> Exceptions should always be accompanied by an explanation of why it occurred<br/>
     * <b>Should NOT be used for debug purposes as it's unstoppable</b>
     *
     * @param tag        log tag
     * @param logMessage message to logged
     * @param throwable  a throwable object to log stacktrace of
     * @see #e(String, String)
     */
    public static void e(@NonNull String tag, @NonNull String logMessage, @NonNull Throwable throwable) {
        if (logMessage == null) {
            return;
        }
        try {
            if (canPrintLog(LogLevel.ERROR)) {
                Log.e(tag, logMessage, throwable);
            }
            p(tag, logMessage);
        } catch (Exception e) {
            Log.e(Constants.LOG_TAG, e.getMessage(), e);
        }
    }

    /**
     * Prints an WTF (What a Terrible Failure) message using Android {@link Log}<br/>
     * <b>Should NOT be used for debug purposes as it's unstoppable</b>
     *
     * @param tag        log tag
     * @param logMessage message to logged
     */
    public static void wtf(@NonNull String tag, @NonNull String logMessage) {
        if (logMessage == null) {
            return;
        }
        try {
            if (canPrintLog(LogLevel.ERROR)) {
                Log.wtf(tag, logMessage);
            }
            p(tag, logMessage);
        } catch (Exception e) {
            Log.e(Constants.LOG_TAG, e.getMessage(), e);
        }
    }

    public static void logSessionDetails(LogDescriptor logDescriptor) {
        try {
            if (instabugSDKDiskLogger != null) {
                instabugSDKDiskLogger.logSessionDetails(logDescriptor);
            }
        } catch (Exception e) {
            Log.e(Constants.LOG_TAG, e.getMessage(), e);
        }
    }

    /**
     * Same as {@link #wtf(String, String)} but with a {@code throwable} to log exceptions as well<br/>
     * <b>Tip:</b> Exceptions should always be accompanied by an explanation of why it occurred<br/>
     * <b>Should NOT be used for debug purposes as it's unstoppable</b>
     *
     * @param tag        log tag
     * @param logMessage message to logged
     * @param throwable  a throwable object to log stacktrace of
     * @see #wtf(String, String)
     */
    public static void wtf(@NonNull String tag, @NonNull String logMessage, @NonNull Throwable throwable) {
        if (logMessage == null) {
            return;
        }
        try {
            if (canPrintLog(LogLevel.ERROR)) {
                Log.wtf(tag, logMessage, throwable);
            }
            p(tag, logMessage);
        } catch (Exception e) {
            Log.e(Constants.LOG_TAG, e.getMessage(), e);
        }
    }

    public static void logEndSession(long timeStamp) {
        try {
            if (instabugSDKDiskLogger != null) {
                instabugSDKDiskLogger.logEndSession(timeStamp);
            }
        } catch (Exception e) {
            Log.e(Constants.LOG_TAG, e.getMessage(), e);
        }
    }

    public static void onDiskLoggingLevelChanged(@LoggingSettings.LogLevels int level) {
        if (instabugSDKDiskLogger != null) {
            instabugSDKDiskLogger.onDiskLoggingLevelChanged(level);
        }
    }

    public static void privateVerbose(String tag, String message) {
        if (instabugSDKDiskLogger != null) {
            instabugSDKDiskLogger.logVerboseLogs(tag, message, Thread.currentThread().getName(), TimeUtils.currentTimeMillis());
        }
    }
}
