package com.instabug.library.util;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.util.Log;

import com.instabug.library.Instabug;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

import androidx.annotation.IntDef;
import androidx.annotation.VisibleForTesting;

import static com.instabug.library.util.InstabugDeprecationLogger.MigrationSection.METHOD_SET_CHATS_STATE;
import static com.instabug.library.util.InstabugDeprecationLogger.MigrationSection.METHOD_SHOW_CHATS;


/**
 * This class is to print migration guide message to console log.
 * it should print 1 msg every 20 seconds.
 * <p>
 * Created by mzelzoghbi on 5/15/18.
 */
public class InstabugDeprecationLogger {

    private static final String PREFIX_SECTION = "#section-";
    @VisibleForTesting
    static String baseUrl = "https://docs.instabug.com/docs/android-sdk-8-6-migration-guide";
    private static InstabugDeprecationLogger INSTANCE;

    private InstabugDeprecationLogger() {
    }

    public static InstabugDeprecationLogger getInstance() {
        if (INSTANCE == null)
            INSTANCE = new InstabugDeprecationLogger();
        return INSTANCE;
    }

    @VisibleForTesting
    @SuppressLint("ERADICATE_FIELD_NOT_NULLABLE")
    static void tearDown() {
        INSTANCE = null;
    }

    private static void setBaseUrl(String baseUrl) {
        if (baseUrl != null && baseUrl.trim().length() > 0) {
            InstabugDeprecationLogger.baseUrl = baseUrl;
        }
    }

    public void log(@MigrationSection int section) {
        if (Instabug.isBuilt() && Instabug.isEnabled()) {
            Context context = Instabug.getApplicationContext();
            if (context != null) {
                boolean isDebuggable = ((context.getApplicationInfo().flags &
                        ApplicationInfo.FLAG_DEBUGGABLE) != 0);
                if (isDebuggable) {
                    if (section == METHOD_SET_CHATS_STATE ||
                            section == METHOD_SHOW_CHATS) {
                        Log.i("INSTABUG", "\n\n" +
                                "In this release, we’re improving the in-app communication " +
                                "experience. Now, your end user will have a unified experience while sending you a " +
                                "report independently from its type. Whether it is a bug, improvement, or question," +
                                " they’ll see the same experience.\n\n" +
                                "The Chats class and its methods have been deprecated, and while they still function, " +
                                "they will be completely removed in a future release. For more details about this API’s" +
                                " replacement, check the docs here: " + getMigrationPageUrl(section) + ".\n\n" +
                                "If you have any questions please reach out to us through contactus@instabug.com.");
                    }
                }
            }
        }
    }

    public void log(String message) {
        if (Instabug.isBuilt() && Instabug.isEnabled()) {
            Context context = Instabug.getApplicationContext();
            if (context != null) {
                boolean isDebuggable = ((context.getApplicationInfo().flags &
                        ApplicationInfo.FLAG_DEBUGGABLE) != 0);
                if (isDebuggable) {
                    Log.i("INSTABUG", message);
                }
            }
        }
    }

    @VisibleForTesting
    String getMigrationPageUrl() {
        return baseUrl;
    }

    @VisibleForTesting
    String getMigrationPageUrl(@MigrationSection int section) {
        return baseUrl + PREFIX_SECTION + getSection(section);
    }

    private String getSection(@MigrationSection int section) {
        switch (section) {
            case METHOD_SHOW_CHATS:
                return "show";
            case METHOD_SET_CHATS_STATE:
                return "setstate";
            default:
                return "";
        }
    }

    @IntDef({
            METHOD_SHOW_CHATS,
            METHOD_SET_CHATS_STATE})
    @Retention(RetentionPolicy.SOURCE)
    public @interface MigrationSection {
        int METHOD_SHOW_CHATS = 18;
        int METHOD_SET_CHATS_STATE = 19;
    }
}
