package com.instabug.library.util

import androidx.annotation.VisibleForTesting
import androidx.annotation.WorkerThread
import com.instabug.library.Constants
import com.instabug.library.internal.sharedpreferences.corePref
import com.instabug.library.internal.sharedpreferences.PreferencesProperty

object CodePushVersionHandler {

    @VisibleForTesting
    const val LENGTH_LIMIT = 30

    @VisibleForTesting
    const val NOT_PROVIDED_ERROR =
        "Code push string is empty. All sessions and reports will be sent with the default app version."

    @VisibleForTesting
    const val LIMIT_EXCEEDED_ERROR =
        "Code push string exceeds the $LENGTH_LIMIT character limit. Extra trailing characters will be trimmed."

    @VisibleForTesting
    const val PREF_KEY_IB_CODE_PUSH_VERSION = "ib_code_push_version"

    /**
     * Added as a workaround to solve unsupported nullability in [PreferencesProperty]
     */
    @VisibleForTesting
    const val PREF_VALUE_NOT_SET = "IBG-CPV-NOT-SET"

    @VisibleForTesting
    const val WITH_CODE_PUSH_FORMAT = "%s+codepush:%s"

    @VisibleForTesting
    var storedCodePushVersion by corePref(PREF_KEY_IB_CODE_PUSH_VERSION, PREF_VALUE_NOT_SET)

    /**
     * Stores the provided Code Push version or clear the stored one if the new one is not valid.
     * validation criteria includes not being null or blank (consists only of whitespaces).
     * @param version the newly provided Code Push version.
     * @param isSetByUser Indicates whether the user have used the API to set the provided [version]
     * or didn't use it at all.
     */
    @WorkerThread
    @JvmStatic
    fun setVersion(version: String?, isSetByUser: Boolean) {
        version?.takeIf(this::isValid)
            ?.let(this::trimAndLogIfNeeded)
            ?.also(this::storeVersion)
            ?: clearVersionAndLog(isSetByUser)
    }

    /**
     * Checks if the provided Code Push version is valid or not.
     * @return true if the provided version is not null & not blank and false otherwise.
     */
    private fun isValid(version: String?): Boolean = !version.isNullOrBlank()

    /**
     * Trims the provided Code Push version if its length exceeds the specified limit (currently 30 chars).
     * In addition, logs an error notifying the user that the provided version exceeds the specified
     * limit and a trimmed version shall be used instead.
     * @param version the provided Code Push version
     * @return the [String] provided in the input if its length didn't exceed the specified limit or
     * a [String] represents the first n chars of the input (trimmed version).
     */
    private fun trimAndLogIfNeeded(version: String): String =
        version.trim().let { trimmed ->
            if (trimmed.length <= LENGTH_LIMIT) return trimmed
            InstabugSDKLogger.w(Constants.LOG_TAG, LIMIT_EXCEEDED_ERROR)
            return trimmed.substring(0, LENGTH_LIMIT)
        }

    /**
     * Stores the provided Code Push version.
     * @param version the provided Code Push version.
     */
    private fun storeVersion(version: String) {
        storedCodePushVersion = version
    }

    /**
     * Clears the stored Code Push version and logs an error notifying the user that there's no
     * Code Push version provided and we'll proceed with the default app version if the user has initially
     * used the API to set a Code Push version.
     */
    private fun clearVersionAndLog(isSetByUser: Boolean) {
        if (isSetByUser) {
            InstabugSDKLogger.w(Constants.LOG_TAG, NOT_PROVIDED_ERROR)
        }
        storedCodePushVersion = PREF_VALUE_NOT_SET
    }

    /**
     * Returns the app version with Code Push version appended to it in the format:
     * {version_name} ({version_code})+codepush:{code_push_version}.
     * If there's no code push version stored, the app version itself is returned with no extensions.
     * @param appVersion the original application version in the format {version_name} ({version_code}).
     * @return a [String] representing the result. If there's a Code Push version stored, then the [String]
     * will be in the format {version_name} ({version_code})+codepush:{code_push_version}.
     * Else, the originally provided app version will be returned in the format {version_name} ({version_code})
     */
    @JvmStatic
    fun appendCodePushVersionIfExists(appVersion: String): String =
        storedCodePushVersion.takeUnless { it == PREF_VALUE_NOT_SET }
            ?.let { storedVersion -> WITH_CODE_PUSH_FORMAT.format(appVersion, storedVersion) }
            ?: appVersion
}
