package com.instabug.library.settings;

import android.content.Context;
import android.graphics.Bitmap;
import android.net.Uri;
import android.os.Build;
import android.view.View;

import androidx.annotation.ColorInt;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Feature;
import com.instabug.library.InstabugColorTheme;
import com.instabug.library.InstabugCustomTextPlaceHolder;
import com.instabug.library.LogLevel;
import com.instabug.library.OnSdkDismissCallback;
import com.instabug.library.OnSdkInvokedCallback;
import com.instabug.library.Platform;
import com.instabug.library.ReproConfigurations;
import com.instabug.library.internal.storage.AttachmentsUtility;
import com.instabug.library.invocation.InstabugInvocationEvent;
import com.instabug.library.invocation.OnInvokeCallback;
import com.instabug.library.invocation.util.InstabugVideoRecordingButtonPosition;
import com.instabug.library.model.IBGTheme;
import com.instabug.library.model.Report;
import com.instabug.library.performanceclassification.DevicePerformanceClassUtils;
import com.instabug.library.ui.onboarding.WelcomeMessage;
import com.instabug.library.util.InstabugAnnotations;
import com.instabug.library.util.ThemeApplier;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.WeakHashMap;

/**
 * Contains SDK settings that should be maintained in-memory until application
 * is terminated
 *
 * @author mSobhy
 */
public class PerSessionSettings {

    public final static int PRIMARY_COLOR_LIGHT = 0XFF0D7AFF;
    public final static int PRIMARY_COLOR_DARK = 0XFF6DCFF6;
    public final static int DEFAULT_ORIENTATION = -2;
    public final static int STATUS_BAR_DARK = 0XFF0A0A0A;
    public final static int STATUS_BAR_LIGHT = 0XFFC5C6C7;
    private final static int MAX_FILE_ATTACHMENTS_COUNT = 3;

    @Nullable
    private static volatile PerSessionSettings perSessionSettings;
    private int primaryColor = PRIMARY_COLOR_LIGHT;
    private int statusBarColor = STATUS_BAR_LIGHT;
    private long sessionStartedAt;
    private boolean isPromptOptionsScreenShown;
    @Nullable
    private Locale instabugLocale;
    @Nullable
    private Locale appLocale;
    @Nullable
    private ArrayList<String> tags;
    @Nullable
    private InstabugCustomTextPlaceHolder instabugCustomTextPlaceHolder;
    @Nullable
    private final LinkedHashMap<Uri, String> extraAttachmentFiles;
    @Nullable
    private OnInvokeCallback onInvokeCallback;
    @Nullable
    private OnSdkDismissCallback onSdkDismissCallback;
    @Nullable
    private OnSdkInvokedCallback onSdkInvokedCallback;
    private InstabugColorTheme instabugColorTheme = InstabugColorTheme.InstabugColorThemeLight;
    @Nullable
    private Bitmap lightCustomBrandingLogoBitmap, darkCustomBrandingLogoBitmap;
    private WelcomeMessage.State welcomeScreenState = WelcomeMessage.State.LIVE;

    private final int autoScreenRecordMaxDuration = 30000;
    private boolean isInBackground = true;
    private int requestedOrientation = DEFAULT_ORIENTATION;
    private boolean isScreenCurrentlyRecorded = false;
    private boolean isAutoScreenRecordingDenied = false;
    private boolean isRequestPermissionScreenShown = false;
    private boolean isAutoScreenRecordingEnabled = false;
    private boolean reproStepsScreenshotEnable = true;
    private boolean isScreenshotByMediaProjectionEnabled = false;
    private boolean isOnboardingShowing = false;
    @Nullable
    private Report.OnReportCreatedListener onReportCreatedListener;
    private boolean isProcessingForeground = false;
    private Feature.State autoScreenRecordingAudioCapturing = Feature.State.DISABLED;
    private final Collection<View> privateViews = Collections.newSetFromMap(new WeakHashMap<View, Boolean>());
    private boolean isInitialScreenShotAllowed = true;
    @Nullable
    private String instabugAppToken = null;

    @Nullable
    private IBGTheme theme = null;

    // cross platform
    @Nullable
    private Report.OnReportCreatedListener onReportCreatedListenerCrossPlatform;
    @Nullable
    private Report currentReport;
    private @Platform
    int platform = Platform.ANDROID;
    private boolean isCrashedSession = false;
    @LogLevel
    int logLevel = LogLevel.ERROR;
    private boolean shouldIgnoreFlagSecure = false;
    private boolean isNonFatalsDisabledTemporarily = false;
    private boolean isFullscreen = false;

    private float deviceTrimmingPercentage = 1.0F;

    private boolean shouldCaptureDialog = true;

    @Nullable
    private ReproConfigurations reproConfigurations;

    private int devicePerformanceClass = DevicePerformanceClassUtils.PERFORMANCE_CLASS_UNDEFINED;

    private InstabugVideoRecordingButtonPosition videoRecordingButtonPosition =
            InstabugVideoRecordingButtonPosition.BOTTOM_RIGHT;

    @Nullable
    private InstabugInvocationEvent[] instabugInvocationEvents = null;

    private PerSessionSettings() {
        tags = new ArrayList<>();
        extraAttachmentFiles = new LinkedHashMap<>(MAX_FILE_ATTACHMENTS_COUNT);
    }

    public static PerSessionSettings getInstance() {
        if (perSessionSettings == null) {
            synchronized (PerSessionSettings.class) {
                if (perSessionSettings == null) {
                    perSessionSettings = new PerSessionSettings();
                }
            }
        }

        return perSessionSettings;
    }

    @Nullable
    public OnInvokeCallback getOnInvokeCallback() {
        return onInvokeCallback;
    }

    public void setOnInvokeCallback(@Nullable OnInvokeCallback onInvokeCallback) {
        this.onInvokeCallback = onInvokeCallback;
    }

    @Nullable
    public OnSdkInvokedCallback getOnSdkInvokedCallback() {
        return onSdkInvokedCallback;
    }

    public void setOnSdkInvokedCallback(@Nullable OnSdkInvokedCallback onSdkInvokedCallback) {
        this.onSdkInvokedCallback = onSdkInvokedCallback;
    }

    @NonNull
    public Locale getInstabugLocale(@Nullable Context context) {
        if (instabugLocale != null) {
            return instabugLocale;
        } else {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                if (context != null) {
                    return context.getResources().getConfiguration().getLocales().get(0);
                }
            } else {
                if (context != null) {
                    return context.getResources().getConfiguration().locale;
                }
            }
        }
        return Locale.getDefault();
    }

    public void setInstabugLocale(@Nullable Locale instabugLocale) {
        this.instabugLocale = instabugLocale;
    }

    @Nullable
    public Locale getAppLocale() {
        return appLocale;
    }

    public void setAppLocale(@Nullable Locale appLocale) {
        this.appLocale = appLocale;
    }

    /**
     * Gets extra attachment files map contains <fileUri,fileNameWithExtension>.
     *
     * @return the extra attachment files uri
     */
    @Nullable
    public LinkedHashMap<Uri, String> getExtraAttachmentFiles() {
        return extraAttachmentFiles;
    }

    /**
     * Add extra attachment file uri.
     *
     * @param attachmentUri         the attachment uri
     * @param fileNameWithExtension the file name with extension
     */
    public void addExtraAttachmentFile(Uri attachmentUri, String fileNameWithExtension) {
        if (extraAttachmentFiles != null) {
            if (AttachmentsUtility.validateFileSize(attachmentUri, AttachmentsUtility.MAX_FILE_SIZE_IN_MB)) {
                if (extraAttachmentFiles.size() == MAX_FILE_ATTACHMENTS_COUNT &&
                        !extraAttachmentFiles.containsKey(attachmentUri)) {
                    Uri uri = extraAttachmentFiles.keySet().iterator().next();
                    extraAttachmentFiles.remove(uri);
                }
                extraAttachmentFiles.put(attachmentUri, fileNameWithExtension);
            }
        }
    }

    /**
     * Clears all extra attached files uri.
     */
    public void clearExtraAttachmentFiles() {
        if (this.extraAttachmentFiles != null) {
            this.extraAttachmentFiles.clear();
        }
    }

    @Nullable
    public ArrayList<String> getTags() {
        return tags;
    }

    public void addTags(String... tags) {
        if (this.tags == null || tags == null)
            return;

        for (String tag : tags) {
            if (tag != null && !tag.equals("null")) {
                this.tags.add(tag);
            }
        }
    }

    public void resetTags() {
        tags = new ArrayList<>();
    }

    public long getSessionStartedAt() {
        return sessionStartedAt;
    }

    public void setSessionStartedAt(long sessionStartedAt) {
        this.sessionStartedAt = sessionStartedAt;
    }

    public int getPrimaryColor() {
        if (theme != null && theme.getPrimaryColor() != ThemeApplier.DEFAULT_COLOR)
            return theme.getPrimaryColor();
        else
            return primaryColor;
    }

    /**
     * set custom color for customizing SDK icons colors
     *
     * @param primaryColor the color that has been set by the user whatever this
     *                     color was parsed from a resource color or hex color
     * @see #getPrimaryColor()
     */
    public void setPrimaryColor(@ColorInt int primaryColor) {
        this.primaryColor = primaryColor;
    }

    @Nullable
    public InstabugCustomTextPlaceHolder getCustomPlaceHolders() {
        return instabugCustomTextPlaceHolder;
    }

    public void setCustomPlaceHolders(InstabugCustomTextPlaceHolder ibgCustomTextPlaceHolder) {
        this.instabugCustomTextPlaceHolder = ibgCustomTextPlaceHolder;
    }

    public int getStatusBarColor() {
        return statusBarColor;
    }

    public void setStatusBarColor(int statusBarColor) {
        this.statusBarColor = statusBarColor;
    }

    public void setRequestedOrientation(@InstabugAnnotations.ScreenOrientation int requestedOrientation) {
        this.requestedOrientation = requestedOrientation;
    }

    public int getRequestedOrientation() {
        return requestedOrientation;
    }

    public void resetRequestedOrientation() {
        this.requestedOrientation = DEFAULT_ORIENTATION;
    }

    boolean isPromptOptionsScreenShown() {
        return isPromptOptionsScreenShown;
    }

    void setPromptOptionsScreenShown(boolean isPromptOptionsScreenShown) {
        this.isPromptOptionsScreenShown = isPromptOptionsScreenShown;
    }

    boolean isRequestPermissionScreenShown() {
        return isRequestPermissionScreenShown;
    }

    void setRequestPermissionScreenShown(boolean isRequestPermissionScreenShown) {
        this.isRequestPermissionScreenShown = isRequestPermissionScreenShown;
    }

    @Nullable
    public OnSdkDismissCallback getOnSdkDismissCallback() {
        return onSdkDismissCallback;
    }

    public void setOnSdkDismissCallback(@Nullable OnSdkDismissCallback onSdkDismissCallback) {
        this.onSdkDismissCallback = onSdkDismissCallback;
    }

    public boolean isScreenCurrentlyRecorded() {
        return isScreenCurrentlyRecorded;
    }

    public void setScreenCurrentlyRecorded(boolean screenCurrentlyRecorded) {
        isScreenCurrentlyRecorded = screenCurrentlyRecorded;
    }

    public boolean isAutoScreenRecordingDenied() {
        return isAutoScreenRecordingDenied;
    }

    public void setAutoScreenRecordingDenied(boolean autoScreenRecordingDenied) {
        isAutoScreenRecordingDenied = autoScreenRecordingDenied;
    }

    public void setAutoScreenRecordingEnabled(boolean autoScreenRecordingEnabled) {
        this.isAutoScreenRecordingEnabled = autoScreenRecordingEnabled;
    }

    public boolean isAutoScreenRecordingEnabled() {
        return this.isAutoScreenRecordingEnabled;
    }

    public int autoScreenRecordingMaxDuration() {
        return autoScreenRecordMaxDuration;
    }

    public void setOnboardingShowing(boolean isOnboardingShowing) {
        this.isOnboardingShowing = isOnboardingShowing;
    }

    public boolean isOnboardingShowing() {
        return isOnboardingShowing;
    }

    @Nullable
    Report.OnReportCreatedListener getOnReportCreatedListener() {
        return onReportCreatedListener;
    }

    void setOnReportCreatedListener(@Nullable Report.OnReportCreatedListener listener) {
        onReportCreatedListener = listener;
    }

    public void setScreenshotByMediaProjectionEnabled(boolean enabled) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            this.isScreenshotByMediaProjectionEnabled = enabled;
        }
    }

    public boolean isScreenshotByMediaProjectionEnabled() {
        return isScreenshotByMediaProjectionEnabled;
    }

    public boolean isProcessingForeground() {
        return isProcessingForeground;
    }

    public void setProcessingForeground(boolean isProcessingForeground) {
        this.isProcessingForeground = isProcessingForeground;
    }

    public void setAutoScreenRecordingAudioCapturingState(Feature.State autoScreenRecordingAudioCapturing) {
        this.autoScreenRecordingAudioCapturing = autoScreenRecordingAudioCapturing;
    }

    public Feature.State getAutoScreenRecordingAudioCapturingState() {
        return autoScreenRecordingAudioCapturing;
    }

    @Nullable
    public Report.OnReportCreatedListener getOnReportCreatedListenerCrossPlatform() {
        return onReportCreatedListenerCrossPlatform;
    }

    public void setOnReportCreatedListenerCrossPlatform(@Nullable Report.OnReportCreatedListener listener) {
        onReportCreatedListenerCrossPlatform = listener;
    }

    @Nullable
    public Report getCurrentReport() {
        return currentReport;
    }

    public void setCurrentReport(@Nullable Report report) {
        this.currentReport = report;
    }

    public boolean isInitialScreenShotAllowed() {
        return isInitialScreenShotAllowed;
    }

    public void setInitialScreenShotAllowed(boolean initialScreenShotAllowed) {
        isInitialScreenShotAllowed = initialScreenShotAllowed;
    }

    public synchronized static void release() {
        perSessionSettings = null;
    }

    public void addPrivateViews(View... views) {
        privateViews.addAll(Arrays.asList(views != null ? views : new View[0]));
    }

    public void removePrivateViews(View... views) {
        privateViews.removeAll(Arrays.asList(views != null ? views : new View[0]));
    }

    public void removeAllPrivateViews() {
        privateViews.clear();
    }

    @NonNull
    public Collection<View> getPrivateViews() {
        return privateViews;
    }

    @Nullable
    public String getInstabugAppToken() {
        return instabugAppToken;
    }

    public void setInstabugAppToken(@Nullable String instabugAppToken) {
        this.instabugAppToken = instabugAppToken;
    }

    public void setCurrentPlatform(int platform) {
        this.platform = platform;
    }

    public @Platform
    int getPlatform() {
        return platform;
    }

    public InstabugColorTheme getTheme() {
        return instabugColorTheme;
    }

    public void setTheme(InstabugColorTheme instabugTheme) {
        this.instabugColorTheme = instabugTheme;
    }

    @Nullable
    public Bitmap getLightCustomBrandingLogoBitmap() {
        return lightCustomBrandingLogoBitmap;
    }

    public void setLightCustomBrandingLogoBitmap(@Nullable Bitmap lightCustomBrandingLogoBitmap) {
        this.lightCustomBrandingLogoBitmap = lightCustomBrandingLogoBitmap;
    }

    @Nullable
    public Bitmap getDarkCustomBrandingLogoBitmap() {
        return darkCustomBrandingLogoBitmap;
    }

    public void setDarkCustomBrandingLogoBitmap(@Nullable Bitmap darkCustomBrandingLogoBitmap) {
        this.darkCustomBrandingLogoBitmap = darkCustomBrandingLogoBitmap;
    }

    public boolean isInBackground() {
        return isInBackground;
    }

    public void setInBackground(boolean inBackground) {
        isInBackground = inBackground;
    }

    public boolean isCrashedSession() {
        return isCrashedSession;
    }

    public void setCrashedSession(boolean crashedSession) {
        isCrashedSession = crashedSession;
    }

    public void setLogLevel(@LogLevel int logLevel) {
        this.logLevel = logLevel;
    }

    @LogLevel
    public int getLogLevel() {
        return logLevel;
    }

    public boolean shouldIgnoreFlagSecure() {
        return shouldIgnoreFlagSecure;
    }

    public void setIgnoreFlagSecure(boolean shouldIgnore) {
        shouldIgnoreFlagSecure = shouldIgnore;
    }

    public boolean isNonFatalsDisabledTemporarily() {
        return isNonFatalsDisabledTemporarily;
    }

    public void setNonFatalsDisabledTemporarily(boolean nonFatalsDisabledTemporarily) {
        this.isNonFatalsDisabledTemporarily = nonFatalsDisabledTemporarily;
    }


    @Nullable
    public ReproConfigurations getReproConfigurations() {
        return this.reproConfigurations;
    }

    public void setReproConfigurations(ReproConfigurations reproConfigurations) {
        this.reproConfigurations = reproConfigurations;
    }

    public void setFullscreen(boolean isFullscreen) {
        this.isFullscreen = isFullscreen;
    }

    public boolean isFullscreen() {
        return isFullscreen;
    }

    public int getDevicePerformanceClass() {
        return devicePerformanceClass;
    }

    public void setDevicePerformanceClass(int devicePerformanceClass) {
        this.devicePerformanceClass = devicePerformanceClass;
    }

    public float getDeviceTrimmingPercentage() {
        return deviceTrimmingPercentage;
    }

    public void setDeviceTrimmingPercentage(float deviceTrimmingPercentage) {
        this.deviceTrimmingPercentage = deviceTrimmingPercentage;
    }

    public InstabugVideoRecordingButtonPosition getVideoRecordingButtonPosition() {
        return videoRecordingButtonPosition;
    }

    public void setVideoRecordingButtonPosition(InstabugVideoRecordingButtonPosition
                                                        videoRecordingButtonPosition) {
        this.videoRecordingButtonPosition = videoRecordingButtonPosition;
    }


    public void setInstabugInvocationEvents(InstabugInvocationEvent... instabugInvocationEvents) {
        this.instabugInvocationEvents = instabugInvocationEvents;
    }

    @Nullable
    public InstabugInvocationEvent[] getInstabugInvocationEvents() {
        if (instabugInvocationEvents != null)
            return instabugInvocationEvents;
        else
            return new InstabugInvocationEvent[]{InstabugInvocationEvent.SHAKE};
    }

    public void clearInvocationEvents() {
        instabugInvocationEvents = null;
    }

    public void setIBGTheme(@Nullable IBGTheme theme) {
        this.theme = theme;
    }

    @Nullable
    public IBGTheme getIBGTheme() {
        return theme;
    }

    public void setShouldCaptureDialog(boolean shouldCaptureDialog) {
        this.shouldCaptureDialog = shouldCaptureDialog;
    }

    public boolean getShouldCaptureDialog() {
        return shouldCaptureDialog;
    }
}
