package com.instabug.library.sessionreplay.bitmap

import android.graphics.Bitmap
import com.instabug.library.sessionreplay.configurations.SRConfigurationsProvider
import com.instabug.library.sessionreplay.monitoring.ErrorCompressingSRScreenshot
import java.io.OutputStream

fun interface BitmapCompressor {
    operator fun invoke(bitmap: Bitmap, outStream: OutputStream)
}

private const val RATIO_LIMIT = 100

class SRBitMapCompressor(
    private val configurations: SRConfigurationsProvider,
    private val format: Bitmap.CompressFormat = Bitmap.CompressFormat.JPEG
) : BitmapCompressor {

    @Throws(ErrorCompressingSRScreenshot::class)
    override fun invoke(bitmap: Bitmap, outStream: OutputStream) {
        val qualityRatio = configurations.screenshotsCompressionQuality.times(RATIO_LIMIT).toInt()
        bitmap.compressQuality(format, qualityRatio, outStream)
    }

    @Throws(ErrorCompressingSRScreenshot::class)
    private fun Bitmap.compressQuality(
        format: Bitmap.CompressFormat,
        qualityRatio: Int,
        outStream: OutputStream
    ) {
        runCatching { !isRecycled && compress(format, qualityRatio, outStream); }
            .onFailure { t -> throw ErrorCompressingSRScreenshot(t) }
            .getOrThrow()
            .takeIf { it } ?: throw ErrorCompressingSRScreenshot(null)
    }
}