package com.instabug.library.sessionV3.sync

import com.instabug.library.core.plugin.PluginsManager
import com.instabug.library.model.v3Session.CustomStoreRattingMode
import com.instabug.library.model.v3Session.Defaults
import com.instabug.library.model.v3Session.RatingDialogDetection
import com.instabug.library.model.v3Session.RatingDialogDetection.RATE_DETECTION_FILTER_ERROR_MESSAGE
import com.instabug.library.model.v3Session.SyncStatus
import com.instabug.library.sessionV3.cache.SessionCacheManager
import com.instabug.library.sessionV3.cache.SimpleSession
import com.instabug.library.sessionV3.cache.id
import com.instabug.library.sessionV3.cache.syncStatus
import com.instabug.library.sessionV3.configurations.IBGSessionConfigurations
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator
import com.instabug.library.sessionV3.providers.SessionID
import com.instabug.library.util.TimeUtils
import com.instabug.library.util.extenstions.runOrLogError
import com.instabug.library.util.threading.PoolProvider
import org.json.JSONObject
import java.util.concurrent.TimeUnit

interface SessionDataReadinessInspector {
    operator fun invoke(sessionIds: List<String>): Map<String, Boolean>
}

object DefaultSessionDataReadinessInspector : SessionDataReadinessInspector {
    private val featureSessionLazyDataProviders
        get() = PluginsManager.getFeaturesSessionLazyDataProvider()

    override fun invoke(sessionIds: List<SessionID>): Map<String, Boolean> =
        featureSessionLazyDataProviders
            .map { PoolProvider.submitIOTask { it.isDataReady(sessionIds) } }
            .map { future -> future.get() }
            .asSequence()
            .flatMap { it.entries }
            .groupBy { it.key }
            .mapValues { it.value.fold(true) { acc, entry -> acc and entry.value } }
            .toMutableMap()
            .apply { putAll((sessionIds - keys).associateWith { true }) }
}

fun interface SessionBatchingFilter {
    operator fun invoke(sessionIds: List<SimpleSession>): List<SimpleSession>
}

val IntervalAndLimitFilter =
    SessionBatchingFilter { sessions ->
        shouldSync(sessions.idsList).let { shouldSync ->
            if (shouldSync) sessions else emptyList()
        }
    }

val DataReadinessFilter = SessionBatchingFilter { sessions ->
    DefaultSessionDataReadinessInspector(sessions.offlineSessions.idsList)
        .filter { (_, isDataReady) -> isDataReady }
        .map { SimpleSession(it.key, SyncStatus.OFFLINE) }
        .let { offlineSessions -> offlineSessions + sessions.readySessions }
}

val NoneFilter = SessionBatchingFilter { it }

class RateDetectionFilter(
    private val cacheManager: SessionCacheManager
) : SessionBatchingFilter {
    override fun invoke(sessionIds: List<SimpleSession>): List<SimpleSession> =
        runOrLogError(errorMessage = RATE_DETECTION_FILTER_ERROR_MESSAGE) {
            cacheManager.querySessionsStoreRatingData()
                .takeIf { it.isNotEmpty() }
                ?.map(::JSONObject)?.map { rattingData ->
                    rattingData.hasAutomaticRate() || rattingData.hasRedirectedCustomRate()
                }?.fold(false) { acc, isSessionHasValidRating -> acc || isSessionHasValidRating }
                ?.takeIf { hasValidRating -> hasValidRating }
                ?.let { sessionIds }
                ?: emptyList()
        }.getOrElse { emptyList() }

    private fun JSONObject.hasRedirectedCustomRate() = optInt(
        RatingDialogDetection.CUSTOM_STORE_RATING_MODE_KEY, CustomStoreRattingMode.NOT_INVOKED
    ) == CustomStoreRattingMode.INVOKED_AND_REDIRECTED_TO_STORE

    private fun JSONObject.hasAutomaticRate() =
        has(RatingDialogDetection.REVIEW_PROMPT_DURATION_KEY)
}

/**
 * This method filters the sessions that have data ready for synchronization.
 * Then it checks if there are any sessions that have valid rate detection data.
 * If there are no such sessions found, it checks the sync interval and request limit.
 * If these parameters are acceptable, it returns the filtered sessions. Otherwise, it returns an empty list.
 * */
val AllFilter = SessionBatchingFilter { sessions ->
    val filteredSessions = DataReadinessFilter(sessions)
    IBGSessionServiceLocator
        .rateDetectionBatchingFilter(filteredSessions)
        .takeIf { it.isEmpty() }
        ?.let { IntervalAndLimitFilter(filteredSessions) }
        ?.takeIf { it.isEmpty() }
        ?: filteredSessions
}

private val List<SimpleSession>.readySessions
    get() = filter { it.syncStatus == SyncStatus.READY_FOR_SYNC }

private val List<SimpleSession>.offlineSessions
    get() = filter { it.syncStatus == SyncStatus.OFFLINE }

private val List<SimpleSession>.idsList
    get() = map { it.id }

private fun shouldSync(offlineSessionsIds: List<String>) =
    with(IBGSessionServiceLocator.sessionConfigurations) {
        isSessionsReachedRequestLimit(offlineSessionsIds) ||
                isSyncIntervalPassed ||
                inDebugMode
    }

private fun IBGSessionConfigurations.isSessionsReachedRequestLimit(
    offlineSessionsIds: List<String>
) = offlineSessionsIds.size >= sessionRequestLimit

private val IBGSessionConfigurations.isSyncIntervalPassed: Boolean
    get() {
        val passedInterval = TimeUtils.currentTimeMillis() - lastSyncTime
        return lastSyncTime == Defaults.DEFAULT_LAST_SYNC ||
                syncInterval <= TimeUnit.MILLISECONDS.toMinutes(passedInterval)
    }
