package com.instabug.library.screenshot.instacapture

import android.app.Activity
import android.graphics.Rect
import android.view.View
import android.view.WindowManager
import com.instabug.library.instacapture.screenshot.FieldHelper.getRootViews
import com.instabug.library.instacapture.screenshot.windowType
import com.instabug.library.interactionstracking.IBGUINode
import com.instabug.library.interactionstracking.UINodeTransformer
import com.instabug.library.internal.LazyProvider
import com.instabug.library.internal.servicelocator.CoreServiceLocator
import com.instabug.library.internal.servicelocator.CoreServiceLocator.ignoredViewsIds
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.threading.PoolProvider
import java.util.concurrent.Future

/**
 * extracts views rectangles in the provided activity by iterating on the root views
 * and extract views location and map it to rect
 * */
interface HierarchyExtractor {

    fun extract(activity: Activity): Future<List<Rect>>

    companion object Factory {
        fun create(
            viewsFilter: ViewsFilter = CoreServiceLocator.composedViewsFilter,
            nodeTransformerProvider: LazyProvider<UINodeTransformer<View>> =
                LazyProvider { CoreServiceLocator.uiNodesTransformer }
        ): HierarchyExtractor =
            HierarchyExtractorImpl(viewsFilter, nodeTransformerProvider)
    }
}

class HierarchyExtractorImpl(
    private val viewsFilter: ViewsFilter,
    private val nodeTransformerProvider: LazyProvider<UINodeTransformer<View>>
) : HierarchyExtractor {
    override fun extract(activity: Activity): Future<List<Rect>> = PoolProvider.submitIOTask {
        val rects = mutableListOf<Rect>()
        val ignoredViews = ignoredViewsIds
        activity.window?.peekDecorView()
            ?.let { nodeTransformerProvider().transform(it) }
            ?.also { traverseAndExtractRects(it, rects, ignoredViews) }

        val shouldMaskDialogs = SettingsManager.getInstance().shouldCaptureDialog
        getRootViews(activity, ignoredViews)
            .filterNot { root -> root.isOverlayViews && !shouldMaskDialogs }
            .filterNot { root -> root.windowType == WindowManager.LayoutParams.TYPE_BASE_APPLICATION }
            .map { root -> nodeTransformerProvider().transform(root.view) }
            .forEach { node -> traverseAndExtractRects(node, rects, ignoredViews) }
        rects
    }

    private fun traverseAndExtractRects(
        node: IBGUINode,
        rects: MutableList<Rect>,
        ignoredViews: IntArray
    ) {
        if (node.isIBGComponent(ignoredViews)) return
        if (viewsFilter shouldMask node) {
            node.asRect()?.also(rects::add)
            return
        }
        for (index in 0 until node.childCount)
            node.getChildAt(index)
                ?.let { child -> traverseAndExtractRects(child, rects, ignoredViews) }
    }
}
