package com.instabug.library.networkinterception.config

import com.instabug.library.featuresflags.configs.FeatureFlagsConfigsHandler
import com.instabug.library.internal.crossplatform.ICPConfigurationsProvider
import org.json.JSONArray
import org.json.JSONObject

class NetworkInterceptionConfigurationHandler(
    private val configurationProvider: IBGNetworkInterceptionConfigurationProvider,
    private val cpConfigurationsProvider: ICPConfigurationsProvider,
) : FeatureFlagsConfigsHandler {
    override fun handleConfigs(featuresResponse: JSONObject) {
        featuresResponse.optJSONObject(KEY_BE_NETWORK_CONFIG)?.also { networkObject ->
            handleW3cConfigurations(networkObject)
            networkObject.optJSONObject(KEY_BE_AUTO_MASKING).also { autoMaskingObject ->
                handleAutoMaskingConfigurations(autoMaskingObject)
            }
        } ?: configurationProvider.reset()
        cpConfigurationsProvider.onFeaturesStateChanged()
    }

    private fun handleW3cConfigurations(networkObject: JSONObject) {
        networkObject
            .optDouble(
                KEY_BE_W3C_EXTERNAL_TRACE_ID_AVAILABLE,
                DEFAULT_W3C_EXTERNAL_TRACE_ID_PERCENTAGE,
            ).let(configurationProvider::setW3CNetworkExternalTraceIdFeatureAvailabilityPercentage)

        networkObject
            .optBoolean(
                KEY_BE_ATTACHING_GENERATE_W3C_TRACE_ID_AVAILABLE,
                DEFAULT_GENERATED_W3C_ATTACHING_AVAILABLE,
            ).let {
                configurationProvider.isAttachingGeneratedW3CExternalTraceIdFeatureAvailable = it
            }

        networkObject
            .optBoolean(
                KEY_BE_ATTACHING_CAPTURED_W3C_TRACE_ID_AVAILABLE,
                DEFAULT_CAPTURED_W3C_ATTACHING_AVAILABLE,
            ).let {
                configurationProvider.isAttachingCapturedW3CExternalTraceIdFeatureAvailable = it
            }
    }

    private fun handleAutoMaskingConfigurations(autoMaskingJsonObject: JSONObject?) {
        autoMaskingJsonObject?.let { autoMaskingObject ->
            autoMaskingObject
                .optDouble(
                    KEY_BE_AUTO_MASKING_AVAILABLE,
                    DEFAULT_AUTO_MASKING_PERCENTAGE,
                ).let(configurationProvider::setAutoMaskingFeatureAvailabilityPercentage)

            autoMaskingObject
                .optJSONArray(
                    KEY_BE_AUTO_MASKING_HEADER_KEYS,
                ).let(this::parseBEAutoMaskingHeaderKeys)

            autoMaskingObject
                .optJSONArray(
                    KEY_BE_AUTO_MASKING_QUERY_KEYS,
                ).let(this::parseBEAutoMaskingQueryKeys)
        } ?: run {
            configurationProvider.resetAutoMasking()
        }
    }


    private fun parseBEAutoMaskingHeaderKeys(headerKeys: JSONArray?) {
        configurationProvider.autoMaskingBEHeaderKeys = headerKeys?.toStringSet() ?: emptySet()
    }

    private fun parseBEAutoMaskingQueryKeys(queryKeys: JSONArray?) {
        configurationProvider.autoMaskingBEQueryKeys = queryKeys?.toStringSet() ?: emptySet()
    }

    private fun JSONArray.toStringSet() = HashSet<String>().also { set ->
        for (index in 0 until length()) {
            set.add(this[index]?.toString() ?: "")
        }
    }
}
