package com.instabug.library.internal.device;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.media.AudioManager;
import android.os.Build;

import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.internal.utils.stability.execution.ReturnableExecutable;
import com.instabug.library.internal.utils.stability.handler.exception.ExceptionHandler;
import com.instabug.library.util.InstabugSDKLogger;

import androidx.annotation.NonNull;

import static android.os.Build.VERSION_CODES.TIRAMISU;

public class InstabugDeviceProperties {
    public static PackageInfo getPackageInfo(@NonNull Context context) {
        try {
            if (Build.VERSION.SDK_INT >= TIRAMISU) {
                return context.getPackageManager().getPackageInfo(
                        context.getPackageName(), PackageManager.PackageInfoFlags.of(0));
            } else {
                return context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
            }
        } catch (PackageManager.NameNotFoundException e) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "Failed to get package info" + e);
            return new PackageInfo();
        }
    }

    public static String getAppVersion(@NonNull Context context) {
        String appVersion;
        PackageInfo packageInfo = getPackageInfo(context);
        appVersion = String.format("%s (%s)", packageInfo.versionName, packageInfo.versionCode);
        return appVersion;
    }

    public static String getAppVersionName(@NonNull Context context) {
        String appVersionName;
        PackageInfo packageInfo = getPackageInfo(context);
        appVersionName = packageInfo.versionName;
        return appVersionName;
    }

    public static String getPackageName(@NonNull Context context) {
        String packageName;
        try {
            packageName = context.getApplicationInfo().packageName;
        } catch (Exception e) {
            packageName = "Could not get information";
            InstabugSDKLogger.d(Constants.LOG_TAG, "failed to get package name" + e);
        }
        return packageName;
    }

    @NonNull
    public static String getDeviceType() {
        return Build.MANUFACTURER + " " + Build.MODEL;
    }

    public static int getCurrentOSLevel() {
        return Build.VERSION.SDK_INT;
    }

    public static boolean isTablet(@NonNull Context context) {
        return (context.getResources().getConfiguration().screenLayout
                & Configuration.SCREENLAYOUT_SIZE_MASK)
                >= Configuration.SCREENLAYOUT_SIZE_LARGE;
    }

    public static boolean checkRingerIsOn(Context context) {
        AudioManager am = (AudioManager) context.getSystemService(Context.AUDIO_SERVICE);
        if (am != null) {
            return am.getRingerMode() == AudioManager.RINGER_MODE_NORMAL;
        } else
            return true;
    }

    /**
     * Check if the app wasn't updated before
     * HINT: Multiple IDE installs without any codebase changes aren't considered as updates,
     * once the codebase has changes, the next IDE run will install an update.
     *
     * @return true if the app has never been updated since the first install
     */
    public static boolean isFirstInstall(Context context) {
        try {
            long firstInstallTime = context.getPackageManager().getPackageInfo(getPackageName(context), 0).firstInstallTime;
            long lastUpdateTime = context.getPackageManager().getPackageInfo(getPackageName(context), 0).lastUpdateTime;
            return firstInstallTime == lastUpdateTime;
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
            return true;
        }
    }

    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public static Integer getVersionCode() {
        return new ExceptionHandler()
                .withPenaltyLog(Constants.LOG_TAG)
                .executeAndGet(new ReturnableExecutable<Integer>() {
                    @Override
                    public Integer execute() throws Exception {
                        Context context = Instabug.getApplicationContext();
                        if (context != null) {
                            PackageInfo packageInfo = context.getPackageManager()
                                    .getPackageInfo(context.getPackageName(), 0);
                            return packageInfo.versionCode;
                        }
                        return -1;
                    }
                }, -1);
    }

    public static boolean isProbablyAnEmulator() {
        return Build.FINGERPRINT.startsWith("generic")
                || Build.FINGERPRINT.startsWith("unknown")
                || Build.MODEL.contains("google_sdk")
                || Build.MODEL.contains("Emulator")
                || Build.MODEL.contains("Android SDK built for x86")
                || Build.BOARD.equals("QC_Reference_Phone") //bluestacks
                || Build.MANUFACTURER.contains("Genymotion")
                || Build.MANUFACTURER.contains("unknown")
                || Build.HOST.startsWith("Build") //MSI App Player
                || (Build.BRAND.startsWith("generic") && Build.DEVICE.startsWith("generic"))
                || "google_sdk".equals(Build.PRODUCT);
    }
}
