package com.instabug.library.interactionstracking

import android.view.View
import com.instabug.library.model.StepType

/**
 * A functional contract for UI locators to implement
 */
fun interface TargetUILocator<Out> {
    fun locate(rootView: View, x: Float, y: Float, @StepType gestureType: String): Out?
}

class AndroidViewsTargetUILocator(
    private val rootTransformer: () -> UINodeTransformer<View>
) : TargetUILocator<Pair<IBGUINode, String>> {

    /**
     * Takes gesture information as an input and recursively traverse the View(s) hierarchy with DFS
     * algorithm to find the View that the interaction was intended for.
     * If [gestureType] input is [StepType.FLING], the algorithm searches for the deepest scrollable
     * or swipeable view only not considering any other type.
     * Otherwise, the algorithm picks the deepest view that contains the touch coordinates within
     * its bounds respecting the z-order of its siblings.
     * The algorithm in general excludes invisible and IBG views even if a touch was performed
     * within the bounds of it.
     *
     * @param rootView a [View] descendant to start the search from as the master parent node of the hierarchy.
     * @param x a [Float] representing the touch location on the x axis.
     * @param y a [Float] representing the touch location on the y axis.
     * @param gestureType a [StepType] telling the algorithm about what gesture has been made
     * to narrow its search scope
     */
    override fun locate(
        rootView: View,
        x: Float,
        y: Float,
        @StepType gestureType: String
    ): Pair<IBGUINode, String>? =
        recursiveLocate(rootTransformer().transform(rootView), x, y, gestureType)

    private fun recursiveLocate(
        node: IBGUINode,
        x: Float,
        y: Float,
        @StepType gestureType: String
    ): Pair<IBGUINode, String>? {
        // If the UI node is not a valid touch target, (x, y) doesn't fall into its bounds
        // for example, fast return null result.
        if (!node.isValidTouchTarget(x, y)) return null
        // If the UI not is not known to be container, there's no need to process children.
        // So, start testing the view itself against trigger gesture and return the result immediately.
        if (!node.isAContainer) return testViewAgainstGesture(node, gestureType)
        var result: Pair<IBGUINode, String>? = null
        // The Z order value is being reset at the start of children processing, then set to the first
        // valid, rightmost view found.
        var currentZOrder = 0f
        // Obviously, we're searching for the first valid rightmost view regardless of its Z order.
        // Hence, the flag is true until we find the first valid view then processing left views
        // will be subject to their Z order.
        var shouldExcludeZFactor = true
        var childUnderProcessing: IBGUINode?
        for (i in node.childCount - 1 downTo 0) {
            childUnderProcessing = node.getChildAt(i)
            // Only process next child if we didn't find any valid child yet or its Z order is higher
            // than the node that we found.
            val shouldProcessChild = childUnderProcessing != null &&
                    (shouldExcludeZFactor || childUnderProcessing.zOrder > currentZOrder)
            if (!shouldProcessChild) continue
            // Once we're here, the child node is only subject to its validity (which is done at the beginning).
            // If the recurring on the current child yielded an output, well, we have our valid target now.
            // So, we set the base z-order to the current child's and falsify the flag that indicates
            // the inclusion of z-order comparison
            recursiveLocate(requireNotNull(childUnderProcessing), x, y, gestureType)
                ?.also { currentZOrder = childUnderProcessing.zOrder }
                ?.also { shouldExcludeZFactor = false }
                ?.also { result = it }
        }
        // If the result is not null, then we've a nested child that matches the descriptions.
        // Otherwise, let's test the node in hand itself against that same description.
        return result ?: testViewAgainstGesture(node, gestureType)
    }

    private fun testViewAgainstGesture(
        nodeUnderTest: IBGUINode,
        @StepType gestureType: String
    ): Pair<IBGUINode, String>? = when (gestureType) {
        StepType.FLING -> testViewForFlingibility(nodeUnderTest)
        else -> nodeUnderTest.takeIf { it.isValidTappableTarget }?.let { Pair(it, gestureType) }
    }

    private fun testViewForFlingibility(nodeUnderTest: IBGUINode): Pair<IBGUINode, String>? {
        return if (nodeUnderTest.isScrollable) {
            Pair(nodeUnderTest, StepType.SCROLL)
        } else if (nodeUnderTest.isSwipeable) {
            Pair(nodeUnderTest, StepType.SWIPE)
        } else {
            null
        }
    }
}
