package com.instabug.library.featuresflags.managers

import com.instabug.library.featuresflags.caching.FeaturesFlagDBManager
import com.instabug.library.featuresflags.configs.FeatureFlagsConfigsProvider
import com.instabug.library.featuresflags.constants.ErrorMessages
import com.instabug.library.featuresflags.constants.ErrorMessages.FF_EMPTY_KEY
import com.instabug.library.featuresflags.constants.ErrorMessages.FF_EXCEED_LIMIT
import com.instabug.library.featuresflags.constants.FFMode
import com.instabug.library.featuresflags.constants.Limits
import com.instabug.library.featuresflags.model.IBGFeatureFlag
import com.instabug.library.util.extenstions.logDebug
import com.instabug.library.util.extenstions.logError
import com.instabug.library.util.threading.OrderedExecutorService

interface FeatureFlagsManager {

    fun add(vararg ibgFeatureFlags: IBGFeatureFlag)

    fun remove(vararg featuresFlags: String)

    fun removeAllFeaturesFlags()

    fun getAllFeaturesFlags(percentage: Float): List<IBGFeatureFlag?>?
    fun trim()

}

internal class FeatureFlagsManagerImpl(
    private val configsProvider: FeatureFlagsConfigsProvider,
    private val featuresFlagDBManager: FeaturesFlagDBManager,
    private val executor: OrderedExecutorService
) : FeatureFlagsManager {
    companion object {
        const val EXECUTOR = "ff_executor"
    }

    override fun add(vararg ibgFeatureFlags: IBGFeatureFlag) = executor.execute(EXECUTOR) {
        runIfEnabled {
            val filteredList = mutableListOf<IBGFeatureFlag>()
            ibgFeatureFlags.toSet().forEach {
                if (it.key.trim().length > Limits.FF_KEY_LENGTH || (it.value?.trim()?.length
                        ?: 0) > Limits.FF_VALUE_LENGTH
                ) {
                    FF_EXCEED_LIMIT.format(it.key).logError()
                } else if (it.key.trim().isEmpty()) {
                    FF_EMPTY_KEY.logError()
                } else {
                    filteredList.add(it.copy(key = it.key.trim(), value = it.value?.trim()))
                }
            }
            filteredList.takeIf { it.size > 0 }?.also {
                featuresFlagDBManager.insert(*it.toTypedArray())
                featuresFlagDBManager.trimToLimit(configsProvider.storeLimit)
            }

        }
    }

    override fun remove(vararg featuresFlags: String) = executor.execute(EXECUTOR) {
        runIfEnabled {
            featuresFlagDBManager.delete(featuresFlags.map { it.trim() })
        }
    }

    override fun removeAllFeaturesFlags() = executor.execute(EXECUTOR) {
        runIfEnabled {
            featuresFlagDBManager.deleteAll()
        }
    }
    override fun getAllFeaturesFlags(percentage: Float) : List<IBGFeatureFlag?>? = executor.submit(EXECUTOR) {
        runIfEnabled { featuresFlagDBManager.getAll(percentage) }
    }.get()

    override fun trim() = executor.execute(EXECUTOR) {
        runIfEnabled {
            featuresFlagDBManager.trimToLimit(configsProvider.storeLimit)
        }
    }


    private inline fun <T> runIfEnabled(command: () -> T?): T? {
        return if (configsProvider.mode != FFMode.DISABLED) {
            command()
        } else {
            ErrorMessages.FF_DISABLED.logDebug()
            null
        }
    }
}