package com.instabug.library.diagnostics.customtraces.model

import androidx.annotation.WorkerThread
import com.instabug.library.diagnostics.DiagnosticsConstants.TRACE_ATTRIBUTES_SIZE
import com.instabug.library.diagnostics.customtraces.di.CustomTracesServiceLocator
import com.instabug.library.diagnostics.customtraces.utils.CustomTracesValidator.toValidAttributeKeyOrNull
import com.instabug.library.diagnostics.customtraces.utils.CustomTracesValidator.toValidAttributeValueOrNull
import com.instabug.library.tracking.InstabugInternalTrackingDelegate

data class IBGCustomTrace(
    var id: Long = -1,
    val name: String = "",
    //Relative time, used for calculating duration only
    val startTimeMicros: Long = 0,
    var endTimeMicros: Long = 0,
    var duration: Long = -1,
    val startedInBG: Boolean = false,
    var endedInBG: Boolean = false,
    var startTime: Long = 0
) {

    var attributes: HashMap<String, String> = hashMapOf()
        private set
    private val customTracesManager = CustomTracesServiceLocator.getCustomTracesManager()
    private val lock: Any = Any()

    fun setAttributesWithoutCaching(attributes: HashMap<String, String>) {
        this.attributes = attributes
    }

    fun setAttributes(attributes: HashMap<String, String>) {
        attributes.forEach {
            setAttribute(it.key, it.value)
        }
    }

    @WorkerThread
    fun setAttribute(key: String?, value: String?) {
        synchronized(lock) {
            if (endTimeMicros != 0L) {
                return
            }

            val trimmedKey = key?.toValidAttributeKeyOrNull()
            val trimmedValue = value?.toValidAttributeValueOrNull()

            trimmedKey?.let {
                if (attributes.containsKey(it)) {
                    return@let customTracesManager.updateAttribute(
                        id,
                        it,
                        trimmedValue
                    )
                }

                if (attributes.size == TRACE_ATTRIBUTES_SIZE || trimmedValue.isNullOrEmpty()) {
                    return@let false
                }

                return@let customTracesManager.setAttribute(
                    id,
                    it,
                    trimmedValue
                )
            }?.let {
                if (it) {
                    if (trimmedValue == null) {
                        attributes.remove(trimmedKey)
                    } else {
                        attributes[trimmedKey] = trimmedValue
                    }
                }
            }
        }
    }


    @WorkerThread
    @JvmOverloads
    fun end(endTimeMicros: Long = System.nanoTime() / 1000) {
        synchronized(lock) {
            if (endTimeMicros == 0L || endTimeMicros <= startTimeMicros)
                return

            val endedInBG =
                InstabugInternalTrackingDelegate.getInstance().startedActivitiesNumber <= 0
            val duration = endTimeMicros - startTimeMicros
            duration.takeIf {
                customTracesManager.endTrace(id, duration, endedInBG) == true
            }?.let {
                this@IBGCustomTrace.endTimeMicros = endTimeMicros
                this@IBGCustomTrace.duration = duration
                this@IBGCustomTrace.endedInBG = endedInBG
            }
        }
    }
}
