package com.instabug.library;

import static com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.Session.SessionFinished;
import static com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.Session.SessionStarted;
import static com.instabug.library.internal.video.ServiceUtils.hasForegroundServiceRunning;

import android.annotation.SuppressLint;
import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.AppStateEvent;
import com.instabug.library.core.eventbus.AppStateEventBus;
import com.instabug.library.core.eventbus.CurrentActivityLifeCycleEventBus;
import com.instabug.library.core.eventbus.SessionStateEventBus;
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventPublisher;
import com.instabug.library.core.eventbus.eventpublisher.IBGDisposable;
import com.instabug.library.internal.orchestrator.ActionsOrchestrator;
import com.instabug.library.internal.orchestrator.UpdateLastSeenAction;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.video.InternalAutoScreenRecorderHelper;
import com.instabug.library.model.common.ICoreSession;
import com.instabug.library.model.common.Session;
import com.instabug.library.model.session.NullSessionException;
import com.instabug.library.model.session.SessionLocalEntity;
import com.instabug.library.model.session.SessionMapper;
import com.instabug.library.model.session.SessionState;
import com.instabug.library.session.SessionsLocalDataSource;
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator;
import com.instabug.library.sessioncontroller.SessionManualController;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.tracking.ActivityLifeCycleEvent;
import com.instabug.library.user.UserManager;
import com.instabug.library.util.DeviceStateProvider;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import java.util.UUID;
import java.util.concurrent.TimeUnit;

import io.reactivexport.Completable;
import io.reactivexport.Single;
import io.reactivexport.functions.Function;
import io.reactivexport.observers.DisposableCompletableObserver;
import io.reactivexport.schedulers.Schedulers;

public class SessionManager {
    private static final String LAST_SEEN_RECORD_EXECUTOR_KEY = "last-seen-record";
    private static final int DEFAULT_SESSION_TIMEOUT_IN_SECONDS = 1800;
    @Nullable
    private static SessionManager INSTANCE;
    private final SettingsManager settingsManager;
    // RxBus Disposables
    @Nullable
    @VisibleForTesting
    IBGDisposable currentActivityLifeCycleDisposable = null;
    @Nullable
    private volatile ICoreSession currentSession;

    private SessionManager(SettingsManager settingsManager) {
        this.settingsManager = settingsManager;
        subscribeToCarenActivityLifeCycle();
    }

    public static synchronized SessionManager getInstance() {
        return INSTANCE != null ? INSTANCE : (INSTANCE = new SessionManager(SettingsManager.getInstance()));
    }

    public static synchronized void init(SettingsManager settingsManager) {
        if (INSTANCE == null)
            INSTANCE = new SessionManager(settingsManager);
    }

    @VisibleForTesting
    static void tearDown() {
        INSTANCE = null;
    }

    //todo no use for this
    public void release() {
        if (currentActivityLifeCycleDisposable != null) {
            currentActivityLifeCycleDisposable.dispose();
            currentActivityLifeCycleDisposable = null;
        }
    }

    public long getCurrentSessionTimeUntilNow() {
        if (settingsManager.getSessionStartedAt() == 0L)
            return 0L;
        else
            return (System.currentTimeMillis() / 1000L) - settingsManager.getSessionStartedAt();
    }

    public synchronized void finishSession() {
        if (InstabugFeaturesManager.getInstance().getFeatureState(IBGFeature.INSTABUG) == Feature
                .State.ENABLED) {
            SettingsManager.getInstance().setInBackground(true);
            if (InstabugFeaturesManager.getInstance().isSessionStitchingEnabled()) {
                SettingsManager.getInstance().setLastForegroundTime(System.currentTimeMillis());
            }
            handleSessionFinishEvent();
        }
    }

    /**
     * Stop and ignore the current session as the SDK was stopped and the app is alive on the foreground
     */
    public void stopCurrentSession() {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Instabug is disabled during app session, ending current session");
        SettingsManager.getInstance().setInBackground(false);
        handleSessionFinishEvent();

    }


    public synchronized void handleSessionStartEvent() {
        handleSessionStartEvent(false);
    }

    public synchronized void handleSessionStartEvent(boolean startedManually) {
        if (SessionManualController.isEnabled() && !startedManually)
            return;

        if (!InstabugStateProvider.getInstance().getState().equals(InstabugState.BUILDING)) {
            ICoreSession session = createStartedSession();
            if (session != null) {
                setCurrentSession(session);
                // handle session state changed with START case
                handleSessionStateChanged(SessionState.START);

                if (SettingsManager.getInstance().isAutoScreenRecordingEnabled()) {
                    InternalAutoScreenRecorderHelper.getInstance().start();
                }
            }
        }
    }

    @VisibleForTesting
    void handleCurrentActivityStoppedEvent() {
        Context context = Instabug.getApplicationContext();
        if (context != null) {
            InstabugFeaturesManager.getInstance().saveFeaturesToSharedPreferences(context);
        } else InstabugSDKLogger.e(Constants.LOG_TAG,
                "unable to saveFeaturesToSharedPreferences due to null appContext");

        if ((getStartedActivitiesNumber() == 0)
                && (Instabug.getApplicationContext() != null)
                && hasForegroundServiceRunning(Instabug.getApplicationContext())) {
            PoolProvider.postIOTask(this::finishSession);
            AppStateEventBus.INSTANCE.post(new AppStateEvent.BackgroundAppStateEvent());
        }
    }

    private void handleSessionFinishEvent() {
        if (settingsManager.getSessionStartedAt() != 0L) {
            // add last session to cache
            if (currentSession != null) {
                addSessionToCache(currentSession);
                // record that the first session has ended
                recordFirstSession();
                // record the last seen of the app
                recordLastSeen();
                // handle session state changed with FINISH case
                handleSessionStateChanged(SessionState.FINISH);
            }
        } else {
            InstabugSDKLogger.d(Constants.LOG_TAG, "Instabug is enabled after session started, Session ignored");
        }
        resetSession();
    }

    private void resetSession() {
        setCurrentSession(null);
    }

    private void recordLastSeen() {
        long currentTimeMillis = System.currentTimeMillis();
        InstabugCore.setLastSeenTimestamp(currentTimeMillis);
        ActionsOrchestrator.obtainOrchestrator(PoolProvider.getSingleThreadExecutor(LAST_SEEN_RECORD_EXECUTOR_KEY))
                .addWorkerThreadAction(new UpdateLastSeenAction(UserManager.getUUID(), currentTimeMillis))
                .orchestrate();
    }

    private void addSessionToCache(final ICoreSession session) {
        if (session != null) {
            if (SettingsManager.getInstance().isSessionEnabled()) {
                createSessionLocalEntity(session)
                        .flatMapCompletable((Function<SessionLocalEntity, Completable>) sessionLocalEntity -> {
                            if (sessionLocalEntity != null)
                                return new SessionsLocalDataSource()
                                        .saveOrUpdate(sessionLocalEntity);
                            return Completable.error(new NullSessionException("sessionLocalEntity can't be null!"));
                        })
                        .subscribeOn(Schedulers.io())
                        .subscribe(new DisposableCompletableObserver() {
                            @Override
                            public void onComplete() {
                                setIsFirstSession(false);
                            }

                            @Override
                            public void onError(@NonNull Throwable e) {
                                InstabugSDKLogger.e(Constants.LOG_TAG, "Error while caching session", e);
                            }
                        });
            }
        }
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private Single<SessionLocalEntity> createSessionLocalEntity(@NonNull final ICoreSession session) {
        return Single.create(emitter -> {
            Context context = Instabug.getApplicationContext();
            boolean usersPageEnabled = InstabugCore.isUsersPageEnabled();
            boolean s2s = IBGSessionServiceLocator.getSessionDataProvider().isV2SessionSent();
            if (context != null) {
                emitter.onSuccess(new SessionLocalEntity
                        .Factory()
                        .create(context, session, usersPageEnabled, s2s));
            }
        });

    }

    private void updateFirstValues() {
        // update value of sessionStartedAt
        long sessionStartTime = System.currentTimeMillis() / 1000L;
        settingsManager.setSessionStartedAt(sessionStartTime);
        // update isFirstRun() value if not already updated
        if (SettingsManager.getInstance().isFirstRun())
            settingsManager.setIsFirstRun(false);
        // set first run at if not set before
        if (SettingsManager.getInstance().getFirstRunAt().getTime() == 0)
            settingsManager.setFirstRunAt(System.currentTimeMillis());
        // incrementing sessions
        settingsManager.incrementSessionsCount();

        ActionsOrchestrator.obtainOrchestrator()
                .addWorkerThreadAction(new UpdateLastSeenAction(UserManager.getUUID(), sessionStartTime * 1000))
                .orchestrate();
    }

    private void recordFirstSession() {
        // update isFirstDismiss() value if not already updated
        if (SettingsManager.getInstance().isFirstDismiss())
            SettingsManager.getInstance().setIsFirstDismiss(false);
    }

    //todo why session manager responsible for app state
    private void handleSessionStateChanged(SessionState sessionState) {
        if (sessionState.equals(SessionState.FINISH)) {
            SettingsManager.getInstance().setIsAppOnForeground(false);
            IBGCoreEventPublisher.post(SessionFinished.INSTANCE);
        } else {
            SettingsManager.getInstance().setIsAppOnForeground(true);
            IBGCoreEventPublisher.post(SessionStarted.INSTANCE);
        }
        SessionStateEventBus.getInstance().post(sessionState);
    }

    private void subscribeToCarenActivityLifeCycle() {
        currentActivityLifeCycleDisposable =
                CurrentActivityLifeCycleEventBus.INSTANCE.subscribe(activityLifeCycleEvent -> {
                    if (activityLifeCycleEvent == ActivityLifeCycleEvent.STOPPED) {
                        handleCurrentActivityStoppedEvent();
                    } else if (activityLifeCycleEvent == ActivityLifeCycleEvent.DESTROYED
                            && getStartedActivitiesNumber() == 0 && currentSession != null) {
                        handleCurrentActivityStoppedEvent();
                    }
                });
    }

    private int getStartedActivitiesNumber() {
        return CoreServiceLocator.getStartedActivitiesCounter().getCount();
    }

    @SuppressWarnings("SameParameterValue")
    private void setIsFirstSession(boolean firstSession) {
        SettingsManager.getInstance().setIsFirstSession(firstSession);
    }


    private ICoreSession createStartedSession() {
        if (currentSession != null) {
            return currentSession;
        }
        // update first values
        updateFirstValues();
        Context context = Instabug.getApplicationContext();

        return SessionMapper.toSession(UUID.randomUUID().toString(),
                DeviceStateProvider.getOS(),
                UserManager.getUUID(),
                context != null ?
                        DeviceStateProvider.getAppVersion(context) : null,
                TimeUnit.MILLISECONDS.toMicros(System.currentTimeMillis()),
                System.nanoTime(),
                isSessionLead());
    }

    private void setCurrentSession(@Nullable ICoreSession currentSession) {
        this.currentSession = currentSession;
    }

    @Nullable
    public Session getCurrentSession() {
        return currentSession;
    }

    //todo session manager shouldn't be responsible for stitching
    private boolean isSessionLead() {
        if (!InstabugFeaturesManager.getInstance().isSessionStitchingEnabled())
            return false;
        // Check if session should be stitched
        long sessionTimeoutInSeconds = SettingsManager.getInstance().getSessionStitchingTimeoutInSeconds(DEFAULT_SESSION_TIMEOUT_IN_SECONDS);
        long timeInBackground = getTimeInBackgroundInSeconds();
        if (timeInBackground == -1 || timeInBackground > sessionTimeoutInSeconds) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "started new billable session");
            return true;
        } else {
            InstabugSDKLogger.v(Constants.LOG_TAG, "session stitched");
            return false;
        }
    }

    private long getTimeInBackgroundInSeconds() {
        long lastForegroundTime = SettingsManager.getInstance().geLastForegroundTime();
        if (lastForegroundTime != -1) {
            return (System.currentTimeMillis() - lastForegroundTime) / 1000;
        }
        return lastForegroundTime;
    }
}
